u.register({

    /**
     * Объект данных: "условия ретаргетинга" . "цели/сегменты"
     * @namespace retargeting-goal-dto
     */
    'retargeting-goal-dto': {

        /**
         * Типы целей/сегментов
         * для конвертации в форматы приложения, сервера
         * @const
         */
        _TYPE: {
            METRIKA_GOAL: {
                server: {
                    id: 'goal'
                },
                goalsTypes: ['METRIKA_GOAL', 'ECOMMERCE_GOAL']
            },
            METRIKA_SEGMENT: {
                server: {
                    id: 'segment'
                },
                goalsTypes: ['METRIKA_SEGMENT']
            },
            AUDIENCE_SEGMENT: {
                server: {
                    id: 'audience'
                },
                goalsTypes: ['AUDIENCE_SEGMENT']
            },
            ECOMMERCE_GOAL: {
                server: {
                    id: 'ecommerce'
                },
                goalsTypes: []
            },
            CDP_SEGMENT: {
                server: {
                    id: 'cdp_segment'
                },
                goalsTypes: ['METRIKA_SEGMENT']
            },
            LAL_SEGMENT: {
                server: {
                    id: 'lal_segment'
                },
                goalsTypes: ['LAL_SEGMENT']
            }
        },

        /**
         * Названия дополнительного типа аудиторий
         */
        _AUDIENCE_TYPE: {
            UPLOADING_EMAIL: {
                name: iget2('retargeting-goal-dto', 'el-pochta', 'эл. почта'),
                server: {
                    id: 'uploading_email'
                }
            },
            UPLOADING_YUID: {
                name: {
                    7: iget2('retargeting-goal-dto', 'mediana', 'Медиана'),
                    others: iget2('retargeting-goal-dto', 'dannye-yandeks', 'данные Яндекс'),
                },
                server: {
                    id: 'uploading_yuid'
                }
            },
            UPLOADING_PHONE: {
                name: iget2('retargeting-goal-dto', 'telefony', 'телефоны'),
                server: {
                    id: 'uploading_phone'
                }
            },
            UPLOADING_MAC: {
                name: iget2('retargeting-goal-dto', 'mac-adresa', 'MAC-адреса'),
                server: {
                    id: 'uploading_mac'
                }
            },
            UPLOADING_CRM: {
                name: iget2('retargeting-goal-dto', 'crm-data', 'Данные CRM'),
                server: {
                    id: 'uploading_crm'
                }
            },
            UPLOADING_IDFA_GAID: {
                name: iget2('retargeting-goal-dto', 'id-mobilnyh', 'ID мобильных'),
                server: {
                    id: 'uploading_idfa_gaid'
                }
            },
            LOOKALIKE: {
                name: iget2('retargeting-goal-dto', 'pohozhiy-segment', 'похожий сегмент'),
                server: {
                    id: 'lookalike'
                }
            },
            GEO_LAST: {
                name: iget2('retargeting-goal-dto', 'geo-seychas', 'гео сейчас'),
                server: {
                    id: 'geo_last'
                }
            },
            GEO_REGULAR: {
                name: iget2('retargeting-goal-dto', 'geo-regulyarnye', 'гео регулярные'),
                server: {
                    id: 'geo_regular'
                }
            },
            GEO_CONDITION: {
                name: iget2('retargeting-goal-dto', 'geo-uslovie', 'гео условие'),
                server: {
                    id: 'geo_condition'
                }
            },
            COMPOSITION_UNION: {
                name: iget2('retargeting-goal-dto', 'kombinirovannyy', 'комбинированный'),
                server: {
                    id: 'composition_union'
                }
            },
            COMPOSITION_INTERSECT: {
                name: iget2('retargeting-goal-dto', 'kombinirovannyy', 'комбинированный'),
                server: {
                    id: 'composition_intersect'
                }
            },
            COMPOSITION_SUBRTACT: {
                name: iget2('retargeting-goal-dto', 'kombinirovannyy', 'комбинированный'),
                server: {
                    id: 'composition_subtract'
                }
            },
            METRIKA_COUNTER_ID: {
                name: iget2('retargeting-goal-dto', 'segment-metriki', 'сегмент Метрики'),
                server: {
                    id: 'metrika_counter_id'
                }
            },
            METRIKA_GOAL_ID: {
                name: iget2('retargeting-goal-dto', 'segment-metriki', 'сегмент Метрики'),
                server: {
                    id: 'metrika_goal_id'
                }
            },
            METRIKA_SEGMENT_ID: {
                name: iget2('retargeting-goal-dto', 'segment-metriki', 'сегмент Метрики'),
                server: {
                    id: 'metrika_segment_id'
                }
            },
            APPMETRIKA_API: {
                name: iget2('retargeting-goal-dto', 'segment-appmetrica', 'сегмент AppMetrica'),
                server: {
                    id: 'appmetrica_api'
                }
            },
            APPMETRIKA_KEY: {
                name: iget2('retargeting-goal-dto', 'segment-appmetrica', 'сегмент AppMetrica'),
                server: {
                    id: 'appmetrica_key'
                }
            },
            APPMETRIKA_SEGMENT_ID: {
                name: iget2('retargeting-goal-dto', 'segment-appmetrica', 'сегмент AppMetrica'),
                server: {
                    id: 'appmetrica_segment_id'
                }
            },
            PIXEL: {
                name: iget2('retargeting-goal-dto', 'piksel', 'пиксель'),
                server: {
                    id: 'pixel'
                }
            },
            ULOADING_CLIENT_ID: {
                name: iget2('retargeting-goal-dto', 'segment-metriki', 'сегмент Метрики'),
                server: {
                    id: 'uploading_client_id'
                }
            },
            GEO_HOME: {
                name: iget2('retargeting-goal-dto', 'geo-regulyarnyy', 'гео регулярный'),
                server: {
                    id: 'geo_home'
                }
            },
            GEO_WORK: {
                name: iget2('retargeting-goal-dto', 'geo-regulyarnyy', 'гео регулярный'),
                server: {
                    id: 'geo_work'
                }
            }
        },

        /**
         * Конвертирует ID "тип записи" сервера в ID приложения
         * @param {*} idSrv
         * @returns {String|null}
         */
        convTypeToApp: function(idSrv) {
            if (typeof idSrv !== 'string') {

                return null;
            }

            return this._convTypeToApp.call(this._TYPE, idSrv);
        },

        /**
         * Конвертирует ID "тип записи" приложения в ID сервера
         * @param {String} idApp
         * @returns {String}
         */
        convTypeToServer: function(idApp) {
            return this._convTypeToServer.call(this._TYPE, idApp);
        },

        /**
         * Конвертирует ID "дополнительного типа аудиторий" в ID приложения
         * @param {*} idSrv
         * @returns {String|null}
         */
        convAudienceTypeToApp: function(idSrv) {
            if (typeof idSrv !== 'string') {

                return null;
            }

            return this._convTypeToApp.call(this._AUDIENCE_TYPE, idSrv);
        },

        /**
         * Возвращает название дополнительного типа аудиторий
         * @param {String} audienceType
         * @param {String} [sourceId]: идентификатор источника, может быть 7 (медиана) или other (данные Яндекса)
         * @returns {String}
         */
        getAudienceTypeName: function(audienceType, sourceId) {
            if (audienceType == 'UPLOADING_YUID') {
                return this._AUDIENCE_TYPE[audienceType].name[sourceId == 7 ? 7 : 'others'];
            }
            return this._AUDIENCE_TYPE[audienceType] && this._AUDIENCE_TYPE[audienceType].name;
        },

        /**
         * Маппинг типа сегмента и типа цели
         * @param type
         * @param goalType
         * @return {boolean}
         */
        matchGoalType: function(type, goalType) {
            return this._TYPE[type].goalsTypes.indexOf(goalType) !== -1;
        },

        /**
         * Создает Data Transfer Object "счетчики для целей/сегментов"
         * @param {{id:number, name:string, domain:string, allow_to_use:boolean, type:string, [subtype:boolean]}} goal
         * @constructor
         * @throw { message: String, items: String[], sourceData: * } объект ошибки
         * @returns {{
         *      id:number,
         *      name:string,
         *      domain:string,
         *      allowToUse:boolean,
         *      type:string,
         *      counter:number,
         *      [audienceType],
         *      [audienceTypeName]
         *  }}
         */
        RetargetingGoalDTO: function(goal) {
            if (!(this instanceof u['retargeting-goal-dto'].RetargetingGoalDTO)) {
                return new u['retargeting-goal-dto'].RetargetingGoalDTO(goal);
            }

            /**
             * Объект ошибки при валидации
             * @type {{ [message]: String, items: String[], [sourceData]: * }}
             */
            var error = {
                items: []
            };

            /**
             * id записи
             * @type {Number}
             */
            this.id = goal.id;

            /**
             * Название
             * @type {string}
             */
            this.name = goal.name;

            /**
             * Домен, на котором используется "счетчик/сегмент"
             * @type {string}
             */
            this.domain = goal.domain;

            /**
             * Доступен для использования.
             * true (1) = когда метрика возвращает эту "цель/сегмент" front -> core -> metrika-api = есть данные
             * false (0) = когда метрика НЕ возвращает "цель/сегмент" front -> core -> metrika-api = нет данных
             * (удалили цель/сегмент, закрыли доступ к счетчику и т.д.)
             * в случае false - название и тип "цели/сегмента" читаются из БД ядра
             *
             * НЕ доступные цели нам нужны только когда они указаны у "правила". Их нельзя где то выбрать
             * @type {boolean}
             */
            this.allowToUse = goal.allow_to_use;

            /**
             * Тип правила. Константа. Можно использовать в коде
             * @type {String}
             */
            this.type = u['retargeting-goal-dto'].convTypeToApp(goal.type);

            /**
             * Номер счетчика, используется для вывода в имени ecommerce цели
             * @type {number}
             */
            this.counter = goal.counter_id;

            // Специфичные поля для аудиторий
            if (this.type === 'AUDIENCE_SEGMENT') {

                this.audienceType = u['retargeting-goal-dto'].convAudienceTypeToApp(goal.subtype);
                    this.audienceTypeName = u['retargeting-goal-dto']
                        .getAudienceTypeName(this.audienceType, goal.uploading_source_id);
                // this.audienceType || error.items.push(iget('не указан для аудиторий'));

                // для случаев, когда данные аудиторий не валидные
                // это происходит на beta регулярно
                if (!this.audienceType) {
                    return null;
                }
                this.domain = 'audience.yandex.ru';
            }

            // проверка валидности данных
            isFinite(this.id) || error.items.push(iget2('retargeting-goal-dto', 'id-dolzhen-byt-bolshe', 'id должен быть больше 0'));
            this.type || error.items.push(iget2('retargeting-goal-dto', 'ne-ukazan-tip-celi', 'не указан тип цели'));

            if (error.items.length) {
                error.message = iget2('retargeting-goal-dto', 'oshibka-konvertacii-dannyh-v', 'Ошибка конвертации данных в объект dto');
                error.sourceData = goal;
                throw error;
            }
        },

        /**
         * Конвертирует внешние данные в формат приложения
         * @param {String} idSrv
         * @returns {String|null}
         *
         */
        _convTypeToApp: function(idSrv) {
            var idApp;

            for (idApp in this) {
                if (this.hasOwnProperty(idApp) && this[idApp].server.id === idSrv) {

                    return idApp;
                }
            }

            return null;
        },

        /**
         * Конвертирует данные приложения для передачи из приложения
         * @param {String} idApp
         * @returns {String}
         */
        _convTypeToServer: function(idApp) {
            return this[idApp].server.id;
        },

        /**
         * Формирует список поддерживаемых типов целей метрики
         * @returns {Array}
         */
        getMetrikaServerTypes: function() {
            return Object.keys(this._TYPE).map(function(type) {
                return this._TYPE[type].server.id;
            }, this);
        }
    }
});
