describe('tipman', function() {
    var scope,
        sandbox;

    before(function() {
        sandbox = sinon.sandbox.create({ useFakeTimers: true });
        scope = $('<div/>').appendTo('body');
        BEM.DOM.scope = scope;
    });

    after(function() {
        BEM.DOM.scope = $('body');
        BEM.DOM.destruct(scope);
        sandbox.restore();
    });

    describe('Корректно пробрасывается', function() {
        var owner,
            tip,
            tipman;

        before(function() {
            owner = $('<div>owner</div>').appendTo(scope);
            tipman = BEM.create('tipman', {
                tipMods: {
                    'tip-mod-name': 'tip-mod-value'
                },
                tipMix: {
                    block: 'tip-mix-block'
                },
                popupDirections: 'popup-cool-direction'
            });

            tipman.show({ owner: owner, content: 'test' });

            tip = tipman.getTip();
        });

        after(function() {
            tipman.destruct();
            BEM.DOM.destruct(owner);
        });

        it('Модификатор блоку `tooltip`', function() {
            expect(tip).to.haveMod('tip-mod-name', 'tip-mod-value');
        });

        it('`mix` блоку `tooltip`', function() {
            expect(tip.domElem.is('.tip-mix-block')).to.equal(true);
        });

        it('`popupDirections` блоку `popup2`', function() {
            expect(tip.findBlockOn('popup2').params.directions).to.eql(['popup-cool-direction']);
        });
    });

    describe('При повтороном вызове `show`', function() {
        var tipman,
            owner,
            prevTooltipInstance,
            prevTooltipSpyDestruct;

        beforeEach(function() {
            owner = $('<div>owner</div>').appendTo(scope);

            tipman = BEM.create('tipman');
            tipman.show({ owner: owner, content: '12314' });

            prevTooltipInstance = tipman.getTip();
            prevTooltipSpyDestruct = sandbox.spy(prevTooltipInstance, 'destruct');
        });

        afterEach(function() {
            BEM.DOM.destruct(owner);
            sandbox.clock.tick(0);
        });

        it('Если блок был показан - то должен создаться новый инстанс, а старый - закрыться и удалиться', function() {
            var prevTooltipSpyDelMod = sandbox.spy(prevTooltipInstance, 'delMod');

            tipman.show({ owner: owner, content: '56789' });

            sandbox.clock.tick(1000);
            expect(scope.find('.tooltip').length).to.equal(1);
            expect(tipman.getTip()).to.not.equal(prevTooltipInstance);
            expect(prevTooltipSpyDelMod.calledWith('shown')).to.be.true;
            expect(prevTooltipSpyDestruct.called).to.be.true;
        });

        it('Если блок был скрыт - то должен использоваться ранее созданный инстанс', function() {
            tipman.hide();

            sandbox.clock.tick(250);

            tipman.show({ owner: owner, content: '56789' });

            expect(scope.find('.tooltip').length).to.equal(1);
            expect(tipman.getTip()).to.equal(prevTooltipInstance);
            expect(prevTooltipSpyDestruct.called).to.not.be.true;

        });

    });

    describe('Если установлена задержка перед показом, то', function() {
        var owner,
            tipman;

        beforeEach(function() {
            owner = $('<div>owner</div>').appendTo($('body'));
            tipman = BEM.create('tipman', { delay: 100 });
        });

        afterEach(function() {
            tipman.destruct();
            BEM.DOM.destruct(owner);
        });

        it('Сразу после вызова `show` блок не виден', function() {
            tipman.show({ owner: owner, content: 'test' });
            sandbox.clock.tick(0);

            expect(!!tipman._isShown()).to.be.false;
        });

        it('Спустя время после вызова `show` равное переданной задержке блок виден', function() {
            tipman.show({ owner: owner, content: 'test' });

            sandbox.clock.tick(100);

            expect(!!tipman._isShown()).to.be.true;
        });

        it('При вызове `show` и `hide` с разницой во времени меньше задержки блок не будет показан', function() {
            var contentSpy = sandbox.spy(function() {
                return '123';
            });

            tipman.show({
                owner: owner,
                content: contentSpy
            });

            sandbox.clock.tick(50);

            tipman.hide();

            sandbox.clock.tick(50);

            expect(contentSpy.called).to.be.false;
        })
    });

});
