var vow = require('vow'),
    vowFs = require('vow-fs');

/**
 * @class BemtreeTech
 * @augments {BemxjstTech}
 * @classdesc
 *
 * Склеивает *bemtree.xjst* и *utils.js* файлы по deps'ам, обрабатывает `bem-xjst`-транслятором
 * Файлы *utils.js* дополнительно оборачивает в bemtree предикат
 * Также добавляет в результат сборки переводы
 *
 * @param {Object}   [options]                          Options
 * @param {String}   [options.target='?.bemtree.js']    Path to a target with compiled file.
 * @param {String}   [options.filesTarget='?.files']    Path to a target with FileList.
 * @param {String[]} [options.sourceSuffixes]           Files with specified suffixes involved in the assembly.
 * @param {String}   [options.exportName='BEMTREE']     Name of BEMTREE template variable.
 * @param {Object}   [options.requires]                 Names of dependencies which should be available from
 *                                                      code of templates.
 * @param {Object}   [options.engineOptions]            Proxies options to bem-xjst
 * @param {Boolean}  [options.forceBaseTemplates=false] Include base templates if no user templates present
 *
 * @example
 * var BemtreeTech = require('enb-bemxjst/techs/bemtree'),
 *     FileProvideTech = require('enb/techs/file-provider'),
 *     bemTechs = require('enb-bem-techs');
 *
 * module.exports = function(config) {
 *     config.node('bundle', function(node) {
 *         // get FileList
 *         node.addTechs([
 *             [FileProvideTech, { target: '?.bemdecl.js' }],
 *             [bemTechs.levels, { levels: ['blocks'] }],
 *             [bemTechs.deps],
 *             [bemTechs.files]
 *         ]);
 *
 *         // build BEMTREE file
 *         node.addTech(BemtreeTech);
 *         node.addTarget('?.bemtree.js');
 *     });
 * };
 */
module.exports = require('enb-bemxjst/techs/bemtree').buildFlow()
    .name('bemtree-utils')
    .target('target', '?.bemtree.xjst.js')
    .useFileList(['bemtree.js', 'utils.js'])
    .builder(function (fileList) {
        // don't add fat wrapper code of bem-xjst
        if (!this._forceBaseTemplates && fileList.length === 0) {
            return this._mockBEMTREE();
        }

        return this._getBEMTREESources(fileList).then(function(sources) {
            return this._compileBEMTREE(sources);
        }, this);
    })
    .methods({
        /**
         * Reads source code of BEMTREE templates and processes.
         *
         * @param {FileList} fileList — objects that contain file information.
         * @returns {Promise}
         * @private
         */
        _getBEMTREESources: function (fileList) {
            var filenames = fileList.map(function(file) {
                return file.fullname;
            });

            return this._readFiles(filenames)
                .then(this._processSources, this);
        },

        /**
         * Reads source files.
         *
         * Each file will be in a form of an object `{ path: String, contents: String }`.
         *
         * @param {String[]} filenames
         * @returns {Promise}
         * @private
         */
        _readFiles: function (filenames) {
            return vow.all(filenames.map(function(filename) {
                return vowFs.read(filename, 'utf8').then(function(source) {
                    return {
                        path: filename,
                        contents: filename.indexOf('utils.js') != -1 ?
                            this._wrapInTemplate(source, filename) :
                            source
                    };
                }, this);
            }, this));
        },

        /**
         * Оборачивает исходный код utils.js в bemtree шаблон
         *
         * 4 следующих факта:
         * 1) Все утилиты попадают в сборку с одним предикатом
         * 2) Порядок выполнения шаблонов с одним предикатом - от последнего к первому
         * 3) Порядок следования утилит соответствует указанным deps-ам
         * 4) Утилиты, оказавшиеся ниже ожидают, что тела утилит от которых они зависят уже будут выполнены
         * вынуждают в каждом шаблоне первой строкой вызывать applyNext - тогда тела шаблонов выполнятся в нужном порядке
         *
         * @param {String} source содержимое файла
         * @param {String} filename название файла
         * @returns {String}
         */
        _wrapInTemplate: function(source, filename) {
            return [
                'block(\'i-utils\').match(function() { return this.utilsElem }).def()(function() {',
                'var prevResult = applyNext();',
                source,
                'return prevResult;',
                '',
                '});'
            ].join('\n');
        }
    })
    .createTech();
