var WK = require('walk'),
    FS = require('fs'),
    Q = require('q'),
    PATH = require('path'),
    BEM = require('bem').api,
    BORSCH = require('borschik').api,
    PJOIN = PATH.join,

    bembl = process.argv[2] || 'islands-components/libs/bem-bl/', // путь к библиотеке bem-bl
    bemControls = process.argv[3] || 'islands-components/libs/bem-controls/', // путь к библиотеке bem-controls
    islandsComponents = process.argv[4] || 'islands-components/', // путь к библиотеке islands-components
    httpPath = process.argv[5] || '/', // путь для ссылок на http ( пример /schema.docs/ )

    projectFolder = process.cwd(), // текущая рабочая директория
    docsFolder = 'schema.docs', // папка в которую попадает результат сборки

    schemaBlocks = 'schema-docs/schema.blocks', // папка с блоками для документации
    blocksList = [],

    pushlvl = [],
    searchIndex = [], //массив с объектами для поиска
    searchCnt = 0; //счетчик объектов

var LOG = require('custom-logger')
    .config({ format: '[%timestamp%] %event%%padding%:%message%' })
    .new({
        convert: { color: 'blue', event: 'Converting' },
        build: { color: 'magenta', event: 'Build' },
        create: { color: 'green', event: 'Create' },
        error: { color: 'red', event: 'ERROR' }
    });


/**
 * Building result bemjson.
 *
 * @param {String} blockName
 * @param {String|Array} content
 */
function BemjsonBuild(blockName, content) {

    LOG.build('bemjson for block ' + blockName);

    // Шаблон страницы
    var isCatalog = blockName === 'catalog';
        pageTemplate = {
        block: 'b-page',
        title: 'Страница ' + blockName,
        head: [
            { elem: 'css', url: '_' + blockName + '.css', ie: false }
        ],
        content:[
            {
                block: 'schema-page',
                mods: !isCatalog ? { type: 'block' } : { type: 'catalog' },
                js: isCatalog ? false : { searchIndex: searchIndex },
                content: content
            },
            { block: 'i-jquery', elem: 'core' },
            { elem: 'js', url: blockName + '.js' }
        ]
    };

    FS.writeFileSync(PJOIN(docsFolder, blockName, blockName + '.bemjson.js'), JSON.stringify([pageTemplate], null, 4), 'utf8');

}

/**
 * Create html using BEM.create and html.js tech.
 *
 * @param {String} blockName
 * @returns {Object} Q promise.
 */
function CreateHtml(blockName) {
    return Q.when(BEM.create({
            level: docsFolder,
            block: blockName,
            forceTech: PJOIN(bembl, 'blocks-common/i-bem/bem/techs/html.js')
        }))
        .then(function () {
            LOG.create('HTML for ' + blockName + ' finish');
        })
        .fail(function () {
            LOG.error('HTML create fail');
        });
}

/**
 * Create declaration for block using BEM.create.
 *
 * @param {String} blockName
 * @returns {Object} Q promise.
 */
function CreateBemdecl(blockName) {
    return Q.when(BEM.create({
            level: docsFolder,
            block: blockName,
            forceTech: 'bemdecl.js'
        }))
        .then(function () {
            LOG.create('bemdecl for ' + blockName + ' finish');
        })
        .fail(function () {
            LOG.error('bemdecl for ' + blockName + ' create fail');
        });
}

/**
 * Build bt declaration using BEM.build.
 *
 * @param {String} blockName
 * @param {Array} techs list
 * @param {String} decl for for build
 * @returns {Object} Q promise.
 */
function build(blockName, techs, decl) {

    var levels = [
        bembl + 'blocks-common',
        bembl + 'blocks-desktop',
        bemControls + 'common.blocks',
        bemControls + 'desktop.blocks',
        islandsComponents + 'common.blocks',
        islandsComponents + 'desktop.blocks',
        schemaBlocks
    ];

    return Q.when(BEM.build({
            outputDir: PJOIN(docsFolder, blockName),
            outputName: blockName,
            declaration: PJOIN(docsFolder, blockName, decl),
            level: levels,
            tech: techs
        }))
        .then(function () {
            LOG.build(blockName + ' - tech - ' + techs + ' finish');
        })
        .fail(function () {
            LOG.error(blockName + ' - tech - ' + techs + ' fail');
        });
}

/**
 * Chain of build and create bem commands.
 *
 * @param {String} blockName
 */
function CreateDocs(blockName) {
    Q.when(CreateBemdecl(blockName))
        .then(function () {
            return build(blockName, ['deps.js'], blockName + '.bemdecl.js');
        })
        .then(function () {
            return build(blockName, [PJOIN(bembl, 'blocks-common/i-bem/bem/techs/bemhtml.js')], blockName + '.deps.js');
        })
        .then(function () {
            return build(blockName, ['js'], blockName + '.deps.js');
        })
        .then(function () {
            return build(blockName, ['css'], blockName + '.deps.js');
        })
        .then(function () {

            return Q.when(BORSCH({
                'freeze': false,
                'input': PJOIN(docsFolder, blockName, blockName + '.css'),
                'minimize': true,
                'output': PJOIN(docsFolder, blockName, '_' + blockName + '.css'),
                'tech': 'css'
            }))
                .then(function() {
                    LOG.build('Open CSS imports for '+blockName+' finish');
                })
                .fail(function () {
                    LOG.error('Open CSS imports for '+blockName+' fail');
                });
        })
        .then(function () {
            return CreateHtml(blockName);
        });
}

/**
 * Create catalog of schema documentation.
 *
 */
function CreateCatalog() {
    var blockName = 'catalog';
    blocksList.unshift({ tag: 'h1', content: 'Страница списка документаций' });

    FS.mkdirSync(PJOIN(docsFolder, blockName));

    BemjsonBuild(blockName, blocksList);

    CreateDocs(blockName);
}

/**
 * Converting schema.json data to bemjson.
 *
 * @param {Object} schema data.
 * @param {String|Array|Object} title.
 * @param {Number|String} parentIndex.
 * @param {Array} [required] required params.
 * @param {String} [parent] parent param.
 * @returns {Object} Return bemjson object.
 */
function Schema2Bemjson(schema, title, parentIndex, lvl, required) {

    var key,
        propsContent,
        isStringTitle = typeof title == 'string',
        childCount = 0;

    isStringTitle && (parentIndex = searchCnt++);

    !pushlvl[lvl] && (pushlvl[lvl] = 0);
    pushlvl[lvl]++;

    if (schema.properties || schema.items) {

        schema = schema.properties ? schema : schema.items;

        propsContent = [];
        for (key in schema.properties) {
            childCount++;
            if (schema.properties.hasOwnProperty(key)) {
                propsContent.push(Schema2Bemjson(schema.properties[key], key, parentIndex, lvl + 1, schema.required));
            }
        }
    }

    !searchIndex[lvl] && (searchIndex[lvl] = []);
    searchIndex[lvl].push({
        title: title,
        info: schema.description || '',
        parent: { index: pushlvl[lvl - 1] - 1, lvl: lvl - 1 },
        root: parentIndex
    });

    return {
        elem: 'root',
        key: title,
        type: Array.isArray(schema.type) ? schema.type.join(', ') : schema.type,
        required: required && ~required.indexOf(title),
        childCount: childCount,
        description: schema.description,
        content: propsContent ? propsContent : ''
    };

}

/**
 * Delete folder with data inside.
 *
 * @param {String} path to directory.
 */
function DeleteFolderRecursive(path) {
    if (FS.existsSync(path)) {
        FS.readdirSync(path).forEach(function (file) {
            var curPath = path + '/' + file;
            if (FS.statSync(curPath).isDirectory()) {
                DeleteFolderRecursive(curPath);
            } else {
                FS.unlinkSync(curPath);
            }
        });
        FS.rmdirSync(path);
    }
}

if (FS.existsSync(docsFolder)) {
    // удаляем папку если она существует
    Q.when(DeleteFolderRecursive(PJOIN(projectFolder, docsFolder)))
        .then(FS.mkdirSync(docsFolder));
} else {
    FS.mkdirSync(docsFolder);
}

var WKR = WK.walk(projectFolder, { followLinks: false });

WKR.on('file', function onFile(root, fileStats, next) {
    var fileNameArray = fileStats.name.split('.'),
        filePrefix = fileNameArray[fileNameArray.length - 2] + '.' + fileNameArray[fileNameArray.length - 1];

    // Берем файлы в папке и проверяем их расширение, если это .schema.json то продолжаем
    if (filePrefix !== 'schema.json') {
        next();
        return;
    }

    //сброс массива и счетчика для поиска
    searchIndex = [];
    searchCnt = 0;
    pushlvl.length = 0;

    var blockName = fileStats.name.split('.')[0],
        schemaJson = JSON.parse(FS.readFileSync(PJOIN(root, fileStats.name), 'utf8')),
        content = [
            {
                block: 'schema-hint',
                content: [
                    {
                        block: 'input',
                        mods: { size: 's' },
                        mix: [{ block: 'schema-page', elem: 'search' }],
                        content: { elem: 'control'}
                    },
                    {
                        block: 'b-link',
                        content: 'Вернуться в каталог',
                        mix: [{ block: 'schema-hint', elem: 'link' }],
                        url: PJOIN(httpPath, docsFolder, 'catalog/catalog.html')
                    },
                    {
                        block: 'schema-info',
                        attrs: { style: 'color: #EAC968; padding: 10px 0 5px 0;' },
                        content: 'обязательные'
                    },
                    {
                        block: 'schema-info',
                        attrs: { style: 'color: #AC8336; padding: 5px 0;' },
                        content: 'необязательные'
                    }
                ]
            },
            {
                block: 'schema-catalog',
                tag: 'h1',
                content: blockName
            },
            {
                block: 'schema-tree',
                js: true,
                content: Schema2Bemjson(schemaJson, 'data', '', 0, ['data'])
            }
        ];

    LOG.convert(blockName + ' to BemJson');

    FS.mkdirSync(PJOIN(docsFolder, blockName));

    BemjsonBuild(blockName, content);

    blocksList.push({
        block: 'schema-catalog',
        elem: 'item',
        content: {
            block: 'b-link',
            mix: [{ block: 'schema-catalog', elem: 'link' }],
            url: PJOIN(httpPath, docsFolder, blockName, blockName + '.html'),
            content: blockName
        }
    });

    CreateDocs(blockName);

    next();
});

WKR.on('errors', function (root, nodeStatsArray, next) {
    next();
});

WKR.on('end', function () {
    LOG.info('Blocks pages created, start build catalog page');
    CreateCatalog();
});
