(function(window, $) {

var HOST = 'https://yastatic.net',
    PATHNAME = '/tableau/tableau.html',
    BLOCK = 'tableau',
    MOD_OPENED = BLOCK + '_opened',
    MOD_CLOSED = BLOCK + '_closed',
    MOD_DEVICE_ = BLOCK + '_device_',
    ELEM_CONTENT = BLOCK + '__content',
    ELEM_TAIL = BLOCK + '__tail',
    EVENT_PREFIX = 'tableau.',
    EVENT_READY = EVENT_PREFIX + 'ready',
    EVENT_CLOSE = EVENT_PREFIX + 'close',
    EVENT_HEIGHT = EVENT_PREFIX + 'height',
    BORDER_WIDTH = 1,
    ANIMATION_DURATION = 150,
    STYLE = /*{STYLE}*/'',
    isStyleAdded = false,
    targetDevices = [],
    deviceMethods = {};

/**
 * Конструктор обвязки.
 *
 * Если передать параметры `headerElem` и `triggerElem`, табло начинает работать в соответствии с портальными гайдами:
 *   - Табло позиционируется по вертикали относительно нижней границы шапки.
 *   - Хвостик табло устанавливается по центру триггера.
 *   - На десктопах табло открывается после остановки курсора над лого.
 *
 * @constructor
 * @param {Object} params
 * @param {String} params.serviceId ID сервиса. Нужен для сбора статистики. Обязательный параметр
 * @param {String} [params.preset] Название набора сервисов
 * @param {String} [params.lang] Язык Табло
 * @param {String} [params.domain] Домен верхнего уровня в ссылках в Табло
 * @param {String} [params.device] Тип устройства: 'desktop', 'touch-pad' или 'touch-phone'
 * @param {String} [params.target] Тип открытия ссылок из Табло
 * @param {jQuery} [params.headerElem] DOM-элемент шапки
 * @param {jQuery} [params.triggerElem] Элемент-активатор (лого или кнопка)
 * @param {Number} [params.width] Ширина попапа
 * @param {Number} [params.height] Высота попапа
 * @param {Number} [params.zIndex] z-index попапа
 * @param {String} [params.host] Хост Табло. По умолчанию https://yastatic.net
 * @param {String} [params.path] Путь до страницы с Табло (включая query). По умолчанию строится из остальных параметров
 */
function Tableau(params) {
    if(!params.serviceId) { throw new Error('Parameter \'serviceId\' is required'); }
    if(this.constructor !== Tableau) { return new Tableau(params); }
    if(STYLE) { this._addStyle(STYLE); }
    if(!params.device) {
        if(targetDevices.length !== 1) { throw new Error('Parameter \'device\' is required'); }
        params.device = targetDevices[0];
    }
    this._useDeviceSpecificMethods(params.device);
    this._params = params;
    this._isOpened = false;
    this._isInited = false;
    this._handlers = [];
    this._bindHandler($(window), 'message', onPostMessage);
    this._$iframe = this._buildIframe(this._buildUrl(this._params));
    this._$tail = this._buildTail();
    this._$tableau = this._buildAndAppendTableau(this._params, this._$iframe, this._$tail);
    if(params.headerElem) { this._bindWiringHandlers(); }
}

/**
 * Табло внутри `<iframe>` проинициализировалось.
 *
 * @event Tableau#init
 */

/**
 * Табло открывается.
 *
 * @event Tableau#open
 */

/**
 * Табло закрывается.
 *
 * @event Tableau#close
 */

/**
 * Табло сейчас будет удалено из DOM.
 *
 * @event Tableau#destruct
 */

/**
 * Указать текст текущего поискового запроса.
 * Текст будет подставлен в ссылки на сервисы.
 *
 * @param {String} text Текст поискового запроса
 * @returns {Tableau}
 */
Tableau.prototype.setSearchText = function(text) {
    this._postMessage({
        fnName: 'updateUrls',
        fnArgs: [{
            text: text,
            serviceId: this._params.serviceId
        }]
    });
    return this;
};

Tableau.prototype._postMessage = function(messageObj) {
    var win = this._$iframe[0].contentWindow,
        message = JSON.stringify(messageObj);
    if(this._isInited) {
        postMessage();
    } else {
        this._$tableau.on('init', postMessage);
    }
    function postMessage() {
        win.postMessage(message, '*');
    }
};

/**
 * Выставить координаты табло.
 *
 * @deprecated Используйте параметры `headerElem` и `triggerElem` для автоматического позиционирования табло.
 * @param {Object} pos
 * @param {number} [pos.top] Отступ сверху
 * @param {number} [pos.left] Отступ слева
 * @param {number} [pos.tail] Сдвиг хвостика
 * @returns {Tableau}
 */
Tableau.prototype.setPos = function(pos) {
    typeof pos.top === 'number' && (this._top = pos.top);
    typeof pos.left === 'number' && (this._left = pos.left);
    typeof pos.tail === 'number' && (this._tail = pos.tail);
    this._isOpened && this._reposition();
    return this;
};

Tableau.prototype._reposition = function() {
    typeof this._top === 'number' && this._$tableau.css('top', this._top - BORDER_WIDTH);
    typeof this._left === 'number' && this._$tableau.css('left', this._left - BORDER_WIDTH);
    typeof this._tail === 'number' && this._$tail.css('left', this._tail);
};

/**
 * Показать попап.
 *
 * Попап откроется в координатах выставленных при вызове `Tableau#setPos()`.
 *
 * @emits Tableau#open
 * @returns {Tableau}
 */
Tableau.prototype.open = function() {
    var wasOpened = this._isOpened;
    this._isOpened = true;
    if(!wasOpened && this._isInited) {
        this._open();
    }
    return this;
};

Tableau.prototype._open = function() {
    this._reposition();
    this._$tableau.removeClass(MOD_CLOSED).addClass(MOD_OPENED);
    this._postMessage({fnName: 'open'});
    this._$tableau.trigger('open');
};

/**
 * Скрыть попап.
 *
 * @emits Tableau#close
 * @returns {Tableau}
 */
Tableau.prototype.close = function() {
    var wasOpened = this._isOpened;
    this._isOpened = false;
    if(wasOpened && this._isInited) {
        this._close();
    }
    return this;
};

Tableau.prototype._close = function() {
    this._$tableau.removeClass(MOD_OPENED);
    var _this = this;
    setTimeout(function() {
        if(!_this._isOpened) {
            _this._$tableau.addClass(MOD_CLOSED);
        }
    }, ANIMATION_DURATION);
    this._$tableau.trigger('close');
};

/**
 * Открыт ли попап.
 *
 * @returns {Boolean}
 */
Tableau.prototype.isOpened = function() {
    return this._isOpened;
};

/**
 * Загружено ли содержимое iframe.
 *
 * @returns {Boolean}
 */
Tableau.prototype.isInited = function() {
    return this._isInited;
};

/**
 * Корневой DOM-элемент табло.
 *
 * @returns {HTMLElement}
 */
Tableau.prototype.getNode = function() {
    return this._$tableau[0];
};

/**
 * Удалить Табло из DOM.
 * После вызова этого метода ссылки на табло следует обнулить.
 *
 * @emits Tableau#destruct
 */
Tableau.prototype.destruct = function() {
    var _this = this;
    $.when(this._$tableau.trigger('destruct')).done(function() {
        $.each(_this._handlers, function(idx, h) {
            h.$node.off(h.event, h.callback);
        });
        _this._handlers = [];
        _this._$tableau.remove();
    });
};

Tableau.prototype._bindHandler = function($node, event, callback) {
    var wrappedCallback = $.proxy(callback, this);
    $node.on(event, wrappedCallback);
    this._handlers.push({$node: $node, event: event, callback: wrappedCallback});
    return this;
};

function onPostMessage(e) {
    var data = e.originalEvent.data;
    if(typeof data !== 'string' || data.indexOf(EVENT_PREFIX) !== 0) { return }
    var dataArr = data.split(':');
    switch(dataArr[0]) {
        case EVENT_READY:
            this._isInited = true;
            if(this._isOpened) {
                this._open();
            }
            this._$tableau.trigger('init');
            break;
        case EVENT_HEIGHT:
            if(!this._params.height) {
                this._$tableau.height(parseInt(dataArr[1], 10) + 2 * BORDER_WIDTH);
            }
            break;
        case EVENT_CLOSE:
            this.close();
            break;
    }
}

Tableau.prototype._buildUrl = function(params) {
    var urlParams = {};
    $.each(['serviceId', 'preset', 'services', 'lang', 'domain', 'device', 'target'], function(idx, key) {
        var value = params[key];
        if(!value) { return; }
        var urlKey = key.replace(/[A-Z]/g, function(letter) {
            return '-' + letter.toLowerCase();
        });
        urlParams[urlKey] = value;
    });

    var host = params.host || HOST,
        path = params.path || PATHNAME;
    return host + path + (path.indexOf('?') > -1 ? '' : '?' + $.param(urlParams));
};

Tableau.prototype._buildIframe = function(url) {
    return $('<iframe/>', {
        'class': ELEM_CONTENT,
        src: url,
        frameborder: 0,
        allowtransparency: true
    });
};

Tableau.prototype._buildTail = function() {
    return $('<div/>', {'class': ELEM_TAIL});
};

Tableau.prototype._buildAndAppendTableau = function(params, $iframe, $tail) {
    return $('<div/>', {
            'class': [BLOCK, MOD_DEVICE_ + params.device, MOD_CLOSED].join(' '),
            css: {
                width: params.width + 2 * BORDER_WIDTH,
                height: params.height && params.height + 2 * BORDER_WIDTH,
                zIndex: params.zIndex
            }
        })
        .append($tail)
        .append($iframe)
        .appendTo('body');
};

Tableau.prototype._addStyle = function(text) {
    if(isStyleAdded) { return }

    var cssNode = document.createElement('style');
    cssNode.type = 'text/css';

    document.getElementsByTagName('head')[0].appendChild(cssNode);

    // в IE сначала надо присоединить ноду, потом писать туда
    // http://msdn2.microsoft.com/en-us/library/ms533698(VS.85).aspx
    if(cssNode.styleSheet) { // IE
        cssNode.styleSheet.cssText = text;
    } else if('textContent' in cssNode) {
        cssNode.textContent = text;
    } else {
        cssNode.innerHTML = text;
    }
    isStyleAdded = true;
};

Tableau._onDevice = function(devices, newMethods) {
    if(typeof devices === 'string') {
        addDevice(0, devices);
        targetDevices.push(devices);
    } else {
        $.each(devices, addDevice);
    }
    function addDevice(idx, device) {
        var methods = deviceMethods[device] || (deviceMethods[device] = {});
        $.each(newMethods, function(name, newMethod) {
            var overrides = methods[name] || (methods[name] = []);
            overrides.push(newMethod);
        });
    }
};

Tableau.prototype._useDeviceSpecificMethods = function(device) { // оверрайдинг методов на коленке
    var _this = this,
        noop = function() {};
    $.each(deviceMethods[device], function(methodName, overrides) {
        var method = _this[methodName] ? $.proxy(_this[methodName], _this) : noop;
        $.each(overrides, function(idx, override) {
            method = $.proxy(override, _this, method); // последним аргументом передаётся __base
        });
        _this[methodName] = method;
    });
};

$.Tableau = Tableau;

})(window, jQuery);
