#!/usr/bin/env node

/**
 * Считывает файлы по маске regions_*.json из папки data и формирует файлы регионов в папке
 * data3/desktop.blocks/i-utils/i-utils.regions
 *
 * Для каждой локали в папке data есть 2 типа файлов - regions_{locale}.js и regions_{locale}_for_ru.js
 * Из этой пары файлов будет сформирован один файл data3/desktop.blocks/i-utils/i-utils.regions/{locale}.regions.json
 * с ключами {locale} и {locale}_for_ru
 *
 * Так же формирует файл
 * data3/desktop.blocks/i-utils/i-utils.regions/all.regions.json для всех локалей (нужен для bemtree)
 *
 * В дальнейшем файлы из папки data3/desktop.blocks/i-utils/i-utils.regions будут включены в сборку.
 * Способ включения в сборку для bemtree и js
 * немного отличаются (см. data3/.bem/enb-techs/regions-json.js и
 * data3/desktop.blocks/i-utils/__regions/i-utils__regions.bemtree.xjst)
 */

var Vow = require('vow'),
    vowFs = require('vow-fs'),
    path = require('path'),
    TECH_NAME = 'regions',
    BLOCK_NAME = 'i-utils',
    REGION_FILE_MASK = 'regions_*.json',
    srcFilesRootDir = path.join(__dirname, '../../data'),
    regionsTreeDestDir = path.join(__dirname, '../desktop.blocks/' + BLOCK_NAME + '/' + BLOCK_NAME + '.regions');

console.log('Формирование блока ' + BLOCK_NAME, '\n');

/**
 * Возвращает путь до локализованного файла с деревом регионов
 *
 * @param {String} locale
 * @returns {String}
 */
function getDestFileName(locale) {
    return path.join(regionsTreeDestDir, [locale, TECH_NAME, 'json'].join('.'));
}

/**
 * Формирует локализованные файлы с деревом регионов и all.regions.json для всех локалей
 *
 * Н: для локали ru будет будет сформирован файл
 * data3/desktop.blocks/i-utils/i-utils.regions/ru.regions.json
 * внутри него будет лежать
 * {
 *  "en": здесь данные из файла data/regions_en.js
 *  "en_for_ru": здесь данные из файла data/regions_en_for_ru.js
 * }
 *
 *
 * @param {Array<RegionTreeFileData>} regionsFilesData информация о файлах в папке data
 * @returns {Vow.Promise}
 */
function buildIUtilsRegionsTreeFiles(regionsFilesData) {

    var groupedFiles = Vow
        .all(regionsFilesData.map(function(localizedFilesData) { // считываем все локализаванные файлы деревьев

            return Vow.all({
                content: vowFs.read(localizedFilesData.file, 'utf8'),
                localizedFilesData: localizedFilesData
            });
        }))
        .then(function(srcFilesContent) { // группируем контент файлов по локали

            return srcFilesContent.reduce(function(result, srcFileContent) {
                var localizedFilesData = srcFileContent.localizedFilesData;

                result[localizedFilesData.locale] = result[localizedFilesData.locale] || {};
                result[localizedFilesData.locale][localizedFilesData.key] = JSON.parse(srcFileContent.content);

                return result;
            }, {});
        });


        return Vow.all(
            groupedFiles.then(function(allFilesContent) { // записываем контент в виде файла для каждой локали
                var locales = Object.keys(allFilesContent);

                return Vow.all({
                    allFilesContent: allFilesContent,
                    writeFilesPromise: Vow.all(locales.map(function(locale) {

                        return vowFs.write(
                            getDestFileName(locale),
                            JSON.stringify(allFilesContent[locale]),
                            'utf8');
                    }))
                });
            }),
            groupedFiles.then(function(allFilesContent) { // записываем деревья для всех локалей в файл all.regions.json

                return vowFs.write(
                    getDestFileName('all'),
                    JSON.stringify(allFilesContent),
                    'utf8');
            }));
}

/**
 * @typedef {Object} RegionTreeFileData
 * @property {String} file полный путь до файла с деревом регионов
 * @property {String} key ключ по которому содержимое файла будет хранится в data3
 *
 * @param {Array<String>} list массив путей до файлов с деревом регионов в папке data
 * @returns {Array<RegionTreeFileData>}
 */
function getFilesDataGroupedByLocale(list) {

    var localeFromFileNameReg = /regions_(\w{2})/,
        regionsTreeKeyFromFileNameReg = /regions_(\w{2})_?(\w{6})?\.json/;

    return list.reduce(function(result, fileName) {

        var locale = fileName.match(localeFromFileNameReg)[1],
            treeKeyRegMatches = fileName.match(regionsTreeKeyFromFileNameReg);

        result.push({
            file: fileName,
            key: treeKeyRegMatches[2] ?
                treeKeyRegMatches[1] + '_' + treeKeyRegMatches[2] :
                treeKeyRegMatches[1],
            locale: locale
        });

        return result;
    }, []);
}

vowFs
    .makeDir(regionsTreeDestDir)
    .then(function() {

        return vowFs.glob(path.join(srcFilesRootDir, REGION_FILE_MASK))
    })
    .then(function(filesList) {

        return buildIUtilsRegionsTreeFiles(getFilesDataGroupedByLocale(filesList));
    })
    .fail(function(err) {
        console.trace(err);

        return Vow.reject(err);
    })
    .done();
