const path = require('path');
const glob = require('glob');
const child_process = require('child_process');
const R = require('ramda');

const cpus = Math.max(require('os').cpus().length - 1, 1);
const CHUNK_SIZE = 50;

const globExpr = path.resolve(__dirname, '../..{/node_modules/bricks/,/}@(common.blocks|desktop.blocks|touch.blocks|source.blocks)', './**/*([^.])@(.js|.bemtree.js|.bemhtml.js|.utils.js|.vm.js)');
const bricksExpr = path.resolve(__dirname, '../../bricks.translations/');
const isBricksKeyset = key => /bricks/.test(key);

glob(globExpr, function (err, foundFiles) {
    const files = foundFiles.filter(file => {
        isBricksKeyset(file) && (file = bricksExpr + file.split('source.blocks/')[1]);

        return !/\.i18n/.test(file);
    });

    const numFiles = files.length;

    if (numFiles === 0) {
        process.stdout.write('No files selected, nothing to do. \n');
        return;
    }

    const processes = Math.min(numFiles, cpus);
    const getChunkSize = collection => processes > 1 ?
          Math.min(Math.floor(collection.length / processes), CHUNK_SIZE) :
          collection.length;
    const fileChunks = R.splitEvery(getChunkSize(files), files);

    const workers = [];
    for (let i = 0; i < processes; i++) {
        workers.push(child_process.fork(require.resolve('./worker'), []));
    }

    const allKeys = [];

    Promise
        .all(workers.map(child => {
            child.send({ action: 'collect-keys', files: fileChunks.pop() });

            return new Promise(resolve => {
                child.on('message', message => {
                    switch (message.action) {
                    case 'result':
                        allKeys.push.apply(allKeys, message.keys);
                        break;

                    case 'free':
                        const nextFiles = fileChunks.pop();

                        if (nextFiles) {
                            child.send({ action: 'collect-keys', files: nextFiles });
                        } else {
                            resolve(child);
                        }
                        break;
                    }
                });
            });
        }))
        .then(children => {
            let duplicateKeys = {};
            const isBricksKeyset = key => /bricks/.test(key);
            const keysets = allKeys.reduce((acc, { keyset, key, value, path }) => {
                keyset = isBricksKeyset(keyset) ? keyset.replace(/^bricks:/, '') : keyset;

                acc[keyset] = acc[keyset] || {};

                acc[keyset][key] || (acc[keyset][key] = { value: value, path: path });

                if (acc[keyset][key].value !== value) {
                    duplicateKeys[keyset] = duplicateKeys[keyset] || [];

                    duplicateKeys[keyset].push({ [key]: value }, { [key]: acc[keyset][key].value });
                }

                return acc;
            }, {});

            if (R.keys(duplicateKeys).length) {
                console.log('\x1b[1m\x1b[41m\x1b[37m%s\x1b[0m', '\nИсправьте дублирующие ключи в переводах, иначе новые ключи не будут добавлены\n' + JSON.stringify(duplicateKeys, null, 4) + `\n`);
            }

            const keys = R.keys(keysets);
            const keysetsChunks = R.pipe(
                R.splitEvery(getChunkSize(keys)),
                R.map(chunk => R.reduce((acc, key) => R.assoc(key, keysets[key], acc), {}, chunk))
            )(keys);

            return Promise.all(children.map(child => {
                child.send({ action: 'fill-translations', keysets: keysetsChunks.pop() });

                return new Promise(resolve => {
                    child.on('message', message => {
                        switch (message.action) {
                            case 'free':
                                const nextKeysets = keysetsChunks.pop();

                                if (nextKeysets) {
                                    child.send({ action: 'fill-translations', keysets: nextKeysets });
                                } else {
                                    resolve(child);
                                }
                                break;
                        }
                    });
                });
            }));
        })
        .then(children => {
            return Promise.all(children.map(c => {
                c.send({ action: 'terminate' });
                return new Promise(resolve => c.on('disconnect', resolve));
            }));
        })
        .catch(e => {
            console.error(e);
        });

});
