const path = require('path');
const Promise = require('bluebird');
const fs = Promise.promisifyAll(require('fs'));
const rootDir = path.join(__dirname, '../..');
const parser = require(path.join(rootDir, '.tanker/parser'));
const R = require('ramda');
const tpl = data => `module.exports = ${JSON.stringify(data, null, 4)};\n`;

const mkdirAsync = function (dirPath) {
    return new Promise((resolve, reject) => {
        return fs.mkdirAsync(dirPath)
            .then(resolve)
            .catch(e => {
                if (e.code === 'EEXIST') {
                    resolve();
                } else {
                    reject(e);
                }
            });
    });
};

const finish = () => setImmediate(() => process.disconnect());
const notify = data => process.send(data);

process.on('message', data => {
    // console.log('message ', process.pid, ' ', data);
    switch(data.action) {
    case 'collect-keys':
        collectKeys(data.files);
        break;
    case 'fill-translations':
        notify({ action: 'free' });
        // fillTranslations(data.keysets);
        break;
    case 'terminate':
        finish();
        break;
    default:
        console.error('unrecognized action: ', data);
        finish();
    }
});

function collectKeys (files) {
    const jobsPromises = files.map(file => {
        return fs.readFileAsync(file)
            .then(content => {
                const getPath = file => {
                    if (!file) return 'desktop.blocks';

                    let pathI18N = file.replace(/.*(common.blocks|desktop.blocks|touch.blocks|bricks).*/,'$1');

                    pathI18N === 'bricks' && (pathI18N = 'bricks.translations');

                    return pathI18N;
                };
                const parseData = parser({
                    data: content,
                    path: getPath(file)
                });

                const keys = parseData.map(R.pick(['keyset', 'key', 'value', 'path']));

                notify({ action: 'result', keys });
            });
    });

    Promise.all(jobsPromises)
        .then(() => notify({ action: 'free' }))
        .catch(e => {
            console.error(e);
        });
}

function fillTranslations (keysets) {
    const jobsPromises = Object.keys(keysets).map(keyset => {
        // b-report-links undefined
        // b-banner-adv-alert2 undefined
        // b-groups-list undefined
        // b-group-tags2-controller undefined
        // b-edit-groups-error-header undefined
        // b-banner-reject-reasons undefined

        let currentDict = { [keyset]: {} };
        let dictLevels = {};

        ['common.blocks', 'desktop.blocks', 'touch.blocks', 'bricks.translations'].forEach(level => {
            let levelDict;

            dictLevels[level] || (dictLevels[level] = {});

            try {
                levelDict = require(path.join(rootDir, level, keyset, `${keyset}.i18n`, 'ru.js'));
            } catch(e) {
            }

            if (levelDict) {
                dictLevels[level][keyset] = levelDict[keyset];

                currentDict[keyset] = R.merge(currentDict[keyset], levelDict[keyset]);
            }
        });

        const currentKeys = R.keys(currentDict[keyset]);
        const srcKeys = R.keys(keysets[keyset]);
        const newKeys = R.difference(srcKeys, currentKeys);

        if (newKeys.length) {
            newKeys.forEach(key => {
                const path = keysets[keyset][key].path;

                dictLevels[path][keyset] || (dictLevels[path][keyset] = {});

                dictLevels[path][keyset][key] = keysets[keyset][key].value;
            });

            return R.keys(dictLevels).forEach(level => {
                const dirPath = path.join(rootDir, level, keyset);
                const i18nDirPath = path.join(rootDir, level, keyset, `${keyset}.i18n`);
                const i18nFilePath = path.join(i18nDirPath, 'ru.js');
                let sortKeys = {};

                if (R.keys(dictLevels[level]).length) {

                    R.keys(dictLevels[level][keyset]).sort().forEach(key => {
                        sortKeys[keyset] || (sortKeys[keyset] = {});

                        sortKeys[keyset][key] = dictLevels[level][keyset][key];
                    });

                    return mkdirAsync(dirPath).then(() => {
                        mkdirAsync(i18nDirPath).then(() => {
                            return fs.writeFileAsync(i18nFilePath, tpl(sortKeys));
                        });
                    })
                }
            });
        } else {
            return Promise.resolve();
        }
    });

    Promise.all(jobsPromises)
        .then(() => notify({ action: 'free' }))
        .catch(e => {
            console.error(e);
        });
}
