/**
 * Утилита получения "Истории успеха".
 * Забирает из бункера данный историй успеха для разных локалей.
 * На выходе формирует JSON вида:
 * {
 *     ru: [
 *         {
 *             "company": "adidas Group",
 *             "image": "//img-fotki.yandex.ru/get/9755/190355746.4/0_1563ed_829f7f9_orig",
 *             "quote": "Нельзя три года использовать одно и то же объявление – это снижает эффективность",
 *             "date": "2013-09-26T21:00:00.000Z",
 *             "url": "https://yandex.ru/adv/story/adidas"
 *         }
 *         ...
 *     ],
 *     en: [...]
 *     uk: [...]
 *     tr: [...]
 * }
 *
 * Сформированный JSON записывается в файл desktop.blocks/i-utils/__success-stories/all.stories.json, который
 * затем фключается в сборку
 */
'use strict';

var fs = require('fs'),
    http = require('http'),
    path = require('path'),

    BUNKER_HOST = 'bunker-api.yandex.net',
    BASE_PATH = '/v1/',
    NODE_DIR_TEMPLATE = 'promo-direct-stories-2015/pages/{lang}/direct',
    STORIES_LANGS = ['ru', 'en', 'uk', 'tr'],
    STORIES_PATH = path.join(
        __dirname,
        '../../desktop.blocks/i-utils/__success-stories/all.stories.json');

module.exports = {};

/**
 * Выполняет команду API Бункера
 * @param {String} cmd  – команда API
 * @param {String} node – нода
 * @returns {Promise}
 * @private
 */
function _getDataFromBunker(cmd, node) {
    var options = {
        protocol: 'http',
        hostname: BUNKER_HOST,
        pathname: BASE_PATH + cmd,
        query: {
            node: node,
            version: 'latest'
        }
    };

    return new Promise(function(resolve, reject) {
        http.get(require('url').format(options), function(res) {
            var body = '';

            res.setEncoding('utf8');
            res
                .on('data', function(chunk) { body += chunk })
                .on(
                    'error', function() {
                        console.log('Bunker ERROR: ' + cmd + ' not executed for node ' + node);
                        resolve('');
                    })
                .on(
                    'end', function() {
                        console.log('Bunker: ' + cmd + ' executed for node ' + node);
                        resolve(body);
                    });
        });
    });

}

/**
 * Возвращает список файлов (историй успеха) для заданного языка
 * @param {String} lang - язык
 * @returns {Promise}
 * @private
 */
function _getFilesList(lang) {
    var node = NODE_DIR_TEMPLATE.replace('{lang}', lang);

    return _getDataFromBunker('ls', node).then(function(json) {

        var items = JSON.parse(json);

        return items
            .filter(function(item) {
                   return item.isDeleted === false;
               })
           .map(function(item) {
               return item.fullName;
           });
    });

}

/**
 * Возвращает историю из файла в Bunker
 * @param {String} node – путь к файлу в Bunker
 * @returns {Promise}
 * @private
 */
function _getStoryFromFile(node) {
    return _getDataFromBunker('cat', node).then(function(json) {
        var data = JSON.parse(json) || {};

        return {
            company: data.company || '',
            image: data.image,
            quote: data.quote || '',
            date: (function(str) {
                var date = str.split('.');

                return new Date(date[2], date[1] - 1, date[0]);
            })(data.date)
        };
    });

}

/**
 * Возвращает истории успеха для заданного языка
 * @param {String} lang – язык
 * @returns {Promise}
 */
function getStories(lang) {
    var bunkerLang = lang === 'uk' ? 'ua' : lang,
        domain = lang === 'tr' || lang === 'uk' ?
            null :
            lang === 'en' ? 'com' : 'ru';

    return _getFilesList(bunkerLang).then(function(files) {

        return Promise.all(files.map(function(node) {

            return _getStoryFromFile(node).then(function(story) {

                if (story) {
                    domain && (story.url = 'https://yandex.' + domain + '/adv/story/' + node.match(/.*\/(\S+)$/)[1]);
                    story.quote = story.quote.replace(/(<\S+?>|\n)/g, '');
                }

                return story;
            });
        }));
    });
}


Promise
    .all(STORIES_LANGS.map(function(lang) {
        return getStories(lang);
    }))
    .then(function(stories) {
        var res= {};

        STORIES_LANGS.forEach(function(lang, index) {
            res[lang] = stories[index].filter(function(story) { return !!story });
        });

        fs.writeFileSync(
            STORIES_PATH,
            JSON.stringify(res));

        done();
    });
