require('dotenv').config({ path: '/etc/direct-tokens/twilight/.env' });

const path = process.argv[2];
const PATH = require('path');
const PROJECT = 'direct2';
const {execSync} = require('child_process');
const request = require('request');
const { get } = require('lodash');
const { TankerApi } = require('tanker-branch');
const { publish } = require('auto-issues');

const { getSvnBranch, getStartrekClient, mergeTankerBranches } = require('./utils');

const branchName = getSvnBranch();
const DEFAULT_ISSUE_FOLLOWERS = ['dima117a', 'lento4ka'];
const DEFAULT_ASSIGNEE = 'robot-baron-cohen';
const DEFAULT_SOURCE_LANGUAGE = 'ru-RU';
const DEFAULT_TARGET_LANGUAGE = ['en-US', 'tr-TR'];
const DEFAULT_SERVICE = 'Direct';

const tankerApi = new TankerApi(PROJECT, process.env.TWILIGHT_TANKER_TOKEN);

const issueClient = getStartrekClient(process.env.TWILIGHT_OAUTH_TOKEN, branchName);

function unique(arr) {
    return [...(new Set(arr || []))]
}

function getFollowerLogin(obj) {
    return typeof obj === 'string' ? obj : obj.id;
}

function getBuildCommitters(buildId, teamcityHost) {
    const path = `changes?locator=build:(id:${buildId})`;

    function getHeader() {
        return 'Basic ' + new Buffer(`robot-twilight:${process.env.TWILIGHT_PASS}`).toString('base64');
    }

    return new Promise(function(resolve, reject) {
        request({
            url: `https://${teamcityHost}/httpAuth/app/rest/${path}`,
            method: 'GET',
            headers: {
                Authorization : getHeader(),
                Accept: 'application/json'
            }
        }, function(error, response, body) {
            if (error) {
                reject(error);
            } else {
                try {
                    const response = JSON.parse(body);
                    const committers = unique([].concat(response.change || []).map(obj => obj.username));

                    resolve(committers);
                } catch (err) {
                    console.log(body);
                    reject(err);
                }
            }
        });
    });
}

async function getIssueFollowers(issue) {
    let followers = (issue.getField('followers') || [])
        .map(getFollowerLogin)
        .concat(DEFAULT_ISSUE_FOLLOWERS);

    const buildId = process.env.BUILD_ID;
    const teamcityHost = process.env.SERVER_URL;

    if (buildId && teamcityHost) {
        const committers = await getBuildCommitters(buildId, teamcityHost);

        followers = followers.concat(committers);
    }

    return unique(followers);
}

function tankerPush(branch) {
    const env = Object.assign({}, process.env);
    const SEPARATOR = process.platform === "win32" ? ";" : ":";

    env.PATH = PATH.resolve("./node_modules/.bin") + SEPARATOR + env.PATH;
    env.TANKER_BRANCH = branch;

    execSync('tanker push', {
        cwd: path,
        env: env,
        stdio: 'inherit',
        shell: true
    });
}

async function getIssueContent(branch) {
    const data = await tankerApi.getDiff('master', branch);
    const filterKeyData = (({data}) => {
        return data.action === 'added' ||
            get(data, 'translations.ru.form.action') === 'changed';
    });

    const allKeys = Object.keys(data.keysets || {})
        .filter(keyset => !/^dna:/.test(keyset))
        .map(keyset => {
            const obj = data.keysets[keyset];

            if (obj.action === 'added') {
                return [{ keyset }];
            }

            return Object.keys(obj.keys || {})
                .map(key => ({
                    key, data: obj.keys[key]
                }))
                .filter(filterKeyData)
                .map(obj => ({
                    keyset,
                    key: obj.key,
                    ...get(obj.data, 'translations.ru.form')
                }));
        });

    const allKeysLinks = [].concat(...allKeys).map(el => {
        let link = `https://tanker.yandex-team.ru/?project=direct2&branch=${ branchName }&keyset=${ el.keyset }`;

        if (el.key) {
            link += `&key=${ el.key }&focus=ru`;
        }

        return link;
    }).sort().join('\n\n');

    if (!allKeysLinks) {
        return;
    }

    return `Тип задачи:
Перевод

Вид текста:
Интерфейс и формы

Ссылки на ключи в Танкере:
${allKeysLinks}`;
}

function executeTransition(issue, transitionId) {
    return new Promise((resolve, reject) => {
        issue.executeTransition(transitionId, null, function(error) {
            if (error) {
                reject(error);
            } else {
                resolve();
            }
        });
    });
}

async function applyTransition(issue, transitionId) {
    const list = await getTransitions(issue);

    if (list.some(t => t.getId() === transitionId)) {
        await executeTransition(issue, transitionId);
    } else {
        console.log(`Не удалось выполнить действие ${transitionId} (отсутствует в списке допустимых: ${list})`)
    }
}

function getTransitions(issue) {
    return new Promise((resolve, reject) => {
        issue.getTransitions(function(error, transitions) {
            if (error) {
                reject(error);
            } else {
                resolve(transitions.toArray());
            }
        });
    });
}

(async () => {
    const list = await tankerApi.listBranches();

    // создаем ветку в Танкере, если такой еще нет
    if (!list.some(br => br.name === branchName)) {
        console.log(`tanker create branch: ${branchName}`);
        await tankerApi.createBranch(branchName);
    }

    // примерживаем в Танкере master в ветку
    console.log(`tanker merge master => ${branchName}`);
    await mergeTankerBranches(tankerApi, 'master', branchName);

    // выгружаем ключи в ветку Танкера
    console.log(`##teamcity[blockOpened name='tanker push ${branchName}']`);
    tankerPush(branchName);
    console.log(`##teamcity[blockClosed name='tanker push ${branchName}']`);

    // создаем тикет, если есть изменения
    const content = await getIssueContent(branchName);

    if (content) {
        const { issue, isNew } = await issueClient.apply(`[Direct] — Перевод — ru-RU > en-US, tr-TR — Переводы для ${branchName}`, content);

        console.log(`Тикет: https://st.yandex-team.ru/${issue.getKey()} (${isNew ? 'new' : 'existing'})`);

        if (isNew) {
            await applyTransition(issue, 'pending');

            issue
                .setField('sourceLanguage', DEFAULT_SOURCE_LANGUAGE)
                .setField('targetLanguage', DEFAULT_TARGET_LANGUAGE)
                .setField('serviceServiceGroup', DEFAULT_SERVICE)
                .setAssignee(DEFAULT_ASSIGNEE);
        }

        // followers
        const followers = await getIssueFollowers(issue);
        issue.setField('followers', followers);

        await publish(issue);
    } else {
        console.log(`empty diff → delete tanker branch: ${branchName}`);
        await tankerApi.deleteBranch(branchName);
    }

})().catch(err => {
    console.log(err);
    process.exit(1);
});
