// Скрипт предназначен для сбора статистики по браузерам для испотльзования в autoprefixer`е
// Производит сделующие действия:
//  1) Идет в API счетчика метрики директа, получает даннные по браузерам за период времени
//  2) Перекладывает статистику в формат autoprefixer`а и сохраняет ее в файл
//
// Логин и пароль пользователя метрики хранится отдельно от скрипта на ppcdev
// Положить их в свою домашнюю директорию надо самостоятельно
require('dotenv').config({ path: '/etc/direct-tokens/twilight/.env' });

// jscs:disable
var https = require('https'),
    querystring = require('querystring'),
    fs = require('fs'),
    chalk = require('chalk'),
    metrikaToken = process.env.METRIKA_TOKEN,
    browserStatPath = 'browser-stat.js',
    // перевод имен браузеров из формата метрики в формат autoprefixer
    browserNamesMap = {
        'Firefox': 'firefox',
        'Яндекс.Браузер': 'yandex',
        'Google Chrome': 'chrome',
        'Opera': 'opera',
        'Safari': 'safari',
        'Samsung Internet': 'samsung',
        'Android Browser': 'android',
        'Mobile Safari': 'ios_saf',
        'Opera Mobile': 'op_mob',
        'MSIE': 'ie',
        'MSIE Mobile': 'ie_mob',
        'Edge': 'edge',
        'UCWEB': 'and_uc',
        'Opera Mini': 'op_mini',
        'Chrome Mobile': 'and_chr'
    },
    metrikaBrowserRegexps = Object.keys(browserNamesMap).map(function(metrikaName) {
        return new RegExp('^(' + metrikaName + ')\\s(\\d+)$');
    }),
    log = {
        error: createLogFn('red'),
        success: createLogFn('green'),
        info: createLogFn('white')
    };

log.info('Fetching metrics data...');

if (metrikaToken) {
    fetchMetrikaStat();
} else {
    log.error('Failed to fetch browser usage data. Make sure you have METRIKA_TOKEN environment variable set');
}

function fetchMetrikaStat() {
    callMetrikaApi({
        path: '/stat/v1/data',
        data: {
            ids: [34],
            metrics: ['ym:s:users'],
            dimensions: ['ym:s:browserAndVersionMajor'],
            filters: ["ym:s:startURLPathLevel2=@'registered' OR ym:s:startURLPathLevel2=@'dna'"],
            date1: '90daysAgo', // статистику берем за 90 дней
            date2: 'today',
            sort: ['-ym:s:users'], // сортируем по убыванию визитов
            limit: 100 // берем первые 100 браузеров
        }
    }, function(err, response) {        
        if (err || response.errors) {
            log.error('Failed to fetch metrika data\n', JSON.stringify(err || response));
            return;
        }

        var stat = {};

        log.success('Received', response.data.length, 'browsers');
        response.data.forEach(function(row) {
            var browserName = row.dimensions[0].name;

            metrikaBrowserRegexps.some(function(browserNameRegexp) {
                var matches = browserName.match(browserNameRegexp),
                    apBrowserName,
                    browserStat;

                if (matches) {
                    apBrowserName = browserNamesMap[matches[1]];
                    browserStat = stat[apBrowserName] || (stat[apBrowserName] = {});
                    browserStat[matches[2]] = ((row.metrics[0] / response.totals) * 100).toPrecision(2);
                    return true;
                }
            });
        });

        writeStatFile(stat, browserStatPath);
        log.success('Wrote new stats successfully');
    });
}

function createLogFn(color) {
    return function() {
        console.log.call(console, chalk[color].apply(chalk, arguments))
    }
}

function writeStatFile(result, path) {
    var content = [
        '// Файл сгенерирован автоматически ' + (new Date()).toISOString(),
        '// Пожалуйста используйте команду `make browser-stat` для обновления статистики',
        '',
        '// jscs:disable',
        '',
        'module.exports = ' + JSON.stringify(result, null, 2) + ';'
    ];

    fs.writeFileSync(path, content.join('\n'), 'utf8');
}

function callMetrikaApi(opts, callback) {
    // https://yandex.ru/dev/metrika/doc/api2/api_v1/data-docpage/
    var req = https.get({
        port: 443,
        method: 'GET',
        hostname: 'api-metrika.yandex.ru',
        headers: {
            'Authorization': 'Bearer ' + metrikaToken
        },
        path: opts.path + '?' + querystring.stringify(opts.data),
    }, function(response) {
        var body = '';

        response.on('data', function(chunk) {
            body += chunk;
        });

        response.on('end', function() {
            callback(null, JSON.parse(body));
        });
    });

    req.on('error', function (e) {
        callback(e);
    });

    req.end();
}
