#!/usr/bin/perl -w

=pod

    Скрипт перемещает данные из таблиц в логи. Работает порядка 2-3 часов. 
    Запускать лучше утром/днем.

=cut

use strict;

use FindBin qw/$Bin/;
use lib "$Bin/../protected/";

use Data::Dumper;

use Settings;
use DBTools;
use Yandex::TimeCommon;

my @DOMAIN_STAT_FIELDS = qw/filter_domain stat_date shows_approx
                                clicks_approx sum_approx bad_ctr_phrases
                                accepted_items declined_items good_ctr_phrases
                                /;
# bad_reasons declined_phrases                                

my $yesterday = yesterday();

warn "Start clearing api_domain_stat";

 _clear_api_domain_stat($yesterday);

warn "End.";

=head2 _clear_api_domain_stat

    Перемещает часть данных из таблицы PPCDICT.api_domain_stat в логи
    Данные за последние 28 дней на каждый домен.

=cut

sub _clear_api_domain_stat
{
    get_dbh(PPCDICT)->{use_mysql_result} = 1;
    
    warn "starting collect";
    
    my $sth = exec_sql(PPCDICT, "select filter_domain, stat_date 
                                      from api_domain_stat 
                                  order by filter_domain desc, stat_date desc");
    
    my %stat = ();
    my ($cnt, $prev_domain) = (0, "");
    while (my $row = $sth->fetchrow_hashref()) {
        
        if ($prev_domain ne $row->{filter_domain}) {
            $cnt = 0;
        }
        
        if ($cnt >= 28 && $prev_domain eq $row->{filter_domain}) {
            # на перемещение
            push @{$stat{$row->{stat_date}}}, $row->{filter_domain};
        }
        
        $cnt++;
        $prev_domain = $row->{filter_domain};
    }
    
    warn "start move. count move=".scalar(keys %stat);
    
    #print Dumper {move => \%stat};
    
    _move_stat_to_logs(\%stat);
    
    warn "end move";
    
    return;
}

=head2 _move_stat_to_logs

    Копирует устаревшие данные из ppcdict.api_domain_stat в логи

=cut

sub _move_stat_to_logs
{
    my $data = shift;
    
    warn "start moving(collect data)";
    
    my $cnt = 0;
    foreach my $day (sort {$a cmp $b} keys %$data) {
        warn "Start process date: $day";
        
        my $values = [];        
        my $domains = $data->{$day};
        
        my $res = get_all_sql(PPCDICT, ["SELECT ".join(",", @DOMAIN_STAT_FIELDS)."
                                         FROM api_domain_stat",
                                            where => {
                                                filter_domain => $domains,
                                                stat_date => $day,
                                            }
                                        ]);
        
        foreach my $r (@$res) {
            push @$values, [map {$r->{$_}} @DOMAIN_STAT_FIELDS];
        }
        
        my $suffix = unix2human(mysql2unix($day), "%Y%m");
        
        _check_and_create_table(PPCLOG, "api_domain_stat", $suffix);
        
        my $sql = "INSERT IGNORE INTO api_domain_stat_$suffix (".join(",", @DOMAIN_STAT_FIELDS).") 
                                     VALUES %s";
        
        my $count = do_mass_insert_sql(PPCLOG, $sql, $values, {max_row_for_insert => 10000});
        if ($count) {
            do_sql(PPCDICT, ["DELETE FROM api_domain_stat", 
                                where => {
                                    filter_domain => $domains,
                                    stat_date => $day
                                }
                             ]);
        }
    }
    
    warn "end moving";
    
    return 1;
}

{
    my %checked = ();
    
sub _check_and_create_table
{
    my ($dbh, $tablename, $suffix) = @_;

    my %TABLE_DESCS = (
        'api_user_domain_stat' => "
            CREATE TABLE IF NOT EXISTS `api_user_domain_stat_%s` (
                `uid` int(10) unsigned NOT NULL,
                `filter_domain` varchar(100) NOT NULL,
                `stat_date` date NOT NULL,
                `shows_approx` int(10) unsigned default NULL,
                `clicks_approx` int(10) unsigned default NULL
            ) ENGINE=ARCHIVE DEFAULT CHARSET=utf8",
        
        'api_domain_stat' => "
            CREATE TABLE IF NOT EXISTS `api_domain_stat_%s` (
                `filter_domain` varchar(100) NOT NULL,
                `stat_date` date NOT NULL default '0000-00-00',
                `shows_approx` int(10) unsigned default NULL,
                `clicks_approx` int(10) unsigned default NULL,
                `sum_approx` int(10) unsigned default NULL,
                `bad_ctr_phrases` int(10) unsigned default NULL,
                `accepted_items` int(10) unsigned default NULL,
                `declined_items` int(10) unsigned default NULL,
                `good_ctr_phrases` int(10) default NULL,
                `bad_reasons` int(10) default NULL, 
                `declined_phrases` int(10) default NULL
            ) ENGINE=ARCHIVE DEFAULT CHARSET=utf8",
    );
    
    if ($TABLE_DESCS{$tablename}) {        
        if (! defined $checked{$tablename.$suffix}) {
            $checked{$tablename.$suffix} = 1;
            return do_sql($dbh, sprintf($TABLE_DESCS{$tablename}, $suffix));
        }
    } else {
        die "Table $tablename, suffix = $suffix not described";
    }
    
    return;
}
}
