#!/usr/bin/perl

=head1 DEPLOY

# approved by lena-san
# .migr
{
  type => 'script',
  when => 'after',
  time_estimate => '~ 30 секунд',
  comment => 'на том сервере, где установлен пакет yandex-direct-scripts, с параметрами:
  - убрать запуск metrica_notify.pl из крона
  - дождаться окончания процесса выполняющего metrica_notify.pl (если есть)
  - выполнить ./deploy/20111228_fix_metrica_alive_domains.pl /var/www/ppc.yandex.ru/protected/logs/metrica_notify.log.2011122{6,7,8}
  - вернуть запись в crontab',
}

=cut

use strict;
use warnings;

use FindBin qw/$Bin/;
use lib "$Bin/../protected";

use Settings;
use Yandex::DBTools;
# захватываем лог скрипта metrica_notify.pl, чтобы избежать гонок с ним
use ScriptHelper get_file_lock => [1, 'metrica_notify.pl'];

use List::Util qw/max/;
use List::MoreUtils qw/ zip uniq /;
use Tools ();
use Settings;
use Yandex::SendMail;
use MailService ();
use POSIX qw/strftime/;
use Primitives;
use Property;
use Yandex::TimeCommon;
use Yandex::MirrorsTools::Hostings qw/strip_www/;
use Yandex::Validate qw/is_valid_int is_valid_ip/;
use Yandex::IDN qw/idn_to_ascii idn_to_unicode is_valid_domain/;
use Yandex::ScalarUtils;

use open ':std' => ':utf8';
use utf8;

use constant TWINKLE_NUM => 5;
use constant FREEZING_TIME  => 2; # сколько часов не трогаем баннеры, если пользователь принудительно изменил состояние

my $NEW_LAST_ID = 33358522; # последний обработанный по состоянию на 16:52 26.12.2011

my $debug_banners_sms = 0;
my $debug_banners = 0;

my %last_counter_state;

$log->out('START');

$log->out('reading logs');
for my $line (<>) {
    next unless $line =~ /\$VAR1 = ({.+});$/;
    my $rec = eval $1;
    next if $rec->{id} <= $NEW_LAST_ID;
    my $key = "$rec->{counter_id}:$rec->{domain}";
    if (!exists $last_counter_state{$key} || ($last_counter_state{$key} && $last_counter_state{$key}->{id} < $rec->{id})) {
        $last_counter_state{$key} = $rec;
    }
}

$log->out('PROCESSING');
my %domains_data;
for my $row(values %last_counter_state) {
    next if $row->{state} ne 'alive'; # интересуют только баннеры, которые в конечном итоге остались живыми
    $row->{domain} = normalize_domain($row->{domain});
    push @{$domains_data{alive}{$row->{domain}}{by_owner}{ $row->{owner} }}, $row;
    push @{$domains_data{alive}{$row->{domain}}{by_counter_id}{ $row->{counter_id} }}, $row;
}

# удаляем домены, ставшие доступными, из таблицы metrica_dead_domains
if ( $domains_data{alive} && %{$domains_data{alive}} ) {
    my $t = timer(op => 'delete_alive_domains');
    my $conds = get_metrica_dead_domains_cond($domains_data{alive}, with_zero_counter_id => 1);
    do_sql(MONITOR, ['DELETE FROM metrica_dead_domains', WHERE => $conds]) if $conds;
}

my @cids_to_sync = ();  # запоминаем в каких кампаниях были изменения (чтобы отправить их в крутилку)
for my $state('alive') {
    if ( $domains_data{$state} && %{$domains_data{$state}} ) {
        # выбираем баннеры, состояние которых должно измениться
        $log->out("fetching banners to become $state");
        my $banners = get_banners_to_change_state( $state, $domains_data{$state} );

        if (@$banners) {
            # размораживаем баннеры, о которых получили событие "оживания" и которые уже отлежались достаточное время
            if ($state eq 'alive') {
                my @bids_to_unfreeze = map { $_->{bid} } grep { $_->{need_unfreezing} } @$banners;
                if (@bids_to_unfreeze) {
                    $log->out('unfreezing '.scalar(@bids_to_unfreeze).' bids: ' . join(',', @bids_to_unfreeze));
                    do_delete_from_table(PPC, 'metrica_frozen_banners', where => {bid => \@bids_to_unfreeze});
                }
            }
            # обрабатываем данные баннеров и формируем данные для уведомления пользователей по почте и SMS
            $log->out("processing banners to become $state");
            my ($changed_banners, $changed_companies, $msg_banner, $sms_banner) = process_banners($state, $banners);
            if (%$changed_banners) {
                $log->out('banners ' . join(',', keys %$changed_banners) . " are now $state");
                update_banners_state($state, $changed_banners);
            }

            if (%$changed_companies) {
                push @cids_to_sync, keys %$changed_companies;
            }

            # отправляем пользователю уведомления о вкл/выкл баннеров
            $log->out("sending notifications about banners became $state");
            send_problems_with_banners($state, $msg_banner);
            send_sms_banner($state, $sms_banner);
        }
    }
}

# помещаем кампании в спец-очередь для быстрой передачи в баннерную крутилку
if (@cids_to_sync) {
    do_mass_insert_sql(PPC, 'INSERT INTO bs_export_specials (cid, par_type) VALUES %s ON DUPLICATE KEY UPDATE cid=cid',
        [ map {[$_, 'metrica']} @cids_to_sync ] );
}

$log->out('FINISH');

##########################################################################################################3
# процедуры целиокм перетащены из metrica_notify.pl

sub get_banners_to_change_state {
    my ($new_state, $domains_data) = @_;

    if ( $new_state ne 'alive' && $new_state ne 'dead' ) {
        die "only 'alive' and 'dead' states are supported";
    }

    my $t = timer(ns => $new_state);

    my $can_use_heavy = ($new_state eq 'dead') ? 1 : 0;

    # собираем два условия, результаты объединяем через UNION DISTINCT
    # делаем так, чтобы искать по двум небольшим индексам (по uid и по counter_id->cid->bid) вместо одного большого (по reverse_domain)
    # чтобы лучше искать делаем два больших IN по обоим ключевым полям, парность домен<->UID и домен<->счётчик дофильтровываем в перле
    # campaigns.uid IN (123,456,789) AND banners.reverse_domain IN ('ur.tset', 'ur.tset.www')
    # metrika_counters.metrica_counter IN (789,1011) AND banners.reverse_domain IN ('ur.tset', 'ur.tset.www')
    my (@owners, @counter_ids, @revdomains);
    while( my ($domain, $domain_data) = each %$domains_data ) {
        push @revdomains, map {reverse_domain $_} uniq map { ($_, idn_to_unicode $_) } ($domain, "www.$domain");
        push @owners, keys %{$domain_data->{by_owner}};
        push @counter_ids, keys %{$domain_data->{by_counter_id}};
    }

    my (@conds, @conds_heavy);
    if (@revdomains && (@owners || @counter_ids)) {
        if (@owners) {
            my $sql_begin = get_banners_in_state_sql_begin(
                debug_banners_sms => $debug_banners_sms,
                debug_banners => $debug_banners,
                new_state => $new_state,
                select_by => 'owner',
            );
            my $cond = {'c.uid' => [uniq @owners], 'b.reverse_domain' => \@revdomains};
            push @conds, $sql_begin, 'AND', $cond;
            if ($can_use_heavy) {
                my $sql_begin_heavy = get_banners_in_state_sql_begin(
                    debug_banners_sms => $debug_banners_sms,
                    debug_banners => $debug_banners,
                    new_state => $new_state,
                    select_by => 'owner',
                    bids_only => 1,
                );
                push @conds_heavy, $sql_begin_heavy, 'AND', $cond;
            }
        }
        if (@counter_ids) {
            my $sql_begin = get_banners_in_state_sql_begin(
                debug_banners_sms => $debug_banners_sms,
                debug_banners => $debug_banners,
                new_state => $new_state,
                select_by => 'counter_id',
            );
            push @conds, "\nUNION DISTINCT\n" if @conds;
            my $cond = {'mc.metrika_counter' => [uniq @counter_ids], 'b.reverse_domain' => \@revdomains};
            push @conds, $sql_begin, 'AND', $cond;
            if ($can_use_heavy) {
                my $sql_begin_heavy = get_banners_in_state_sql_begin(
                    debug_banners_sms => $debug_banners_sms,
                    debug_banners => $debug_banners,
                    new_state => $new_state,
                    select_by => 'counter_id',
                    bids_only => 1,
                );
                push @conds_heavy, "\nUNION DISTINCT\n" if @conds_heavy;
                push @conds_heavy, $sql_begin_heavy, 'AND', $cond;
            }
        }
    } else {
        return [];
    }
    # запрос списка баннеров, которые должны изменить своё состояние, может получиться тяжёлым
    # при этом нам критично не потерять баннеры при включении, но не критично пропустить часть 
    # умерших баннеров: в крайнем случае мы найдём их в ходе следующей полной проверки
    if ($can_use_heavy) {
        my $bids = get_one_column_sql(PPC_HEAVY, \@conds_heavy) || [];
        if ($bids && @$bids) {
            push @conds, 'AND', {'b.bid' => $bids};
        }
    }
    # чтобы не отключить объявления несколько раз, в конечном итоге всегда выбираем из мастера
    # иначе из-за отставания реплики можем не увидеть прошлое включение/отключение этим же скриптом или пользователем
    my $banners = get_all_sql( PPC, \@conds );

    # фильтруем баннеры, подпадающие под пары домен<->UID или домен<->counter_id + прокидываем время последней проверки в баннеры
    my @filtered_banners;
    for my $banner(@$banners) {
        my $domain = normalize_domain($banner->{domain});
        if ( ref($domains_data->{$domain}{by_owner}{$banner->{uid}}) eq 'ARRAY' ) {
            $banner->{checked_time} = $domains_data->{$domain}{by_owner}{$banner->{uid}}->[0]->{checked_time};
        } elsif ( $banner->{metrika_counter} && ref($domains_data->{$domain}{by_counter_id}{$banner->{metrika_counter}}) eq 'ARRAY' ) {
            $banner->{checked_time} = $domains_data->{$domain}{by_counter_id}{$banner->{metrika_counter}}->[0]->{checked_time};
        } else {
            next;
        }
        push @filtered_banners, $banner;
    }

    return \@filtered_banners;
}

=item get_banners_in_state_sql_begin

    Возвращает первую часть SQL-запроса для выбора баннеров, которые должны перейти в новое состояние жив/мёртв
    Принимает именованные параметры:
        debug_banners_sms => 1|0
        debug_banners => 1|0
        new_state => dead|alive -- обязательный
        select_by => counter_id|owner -- обязательный
        bids_only => 1|0

=cut

sub get_banners_in_state_sql_begin {
    my (%O) = @_;

    my @sql = ('SELECT');
    if ($O{bids_only}) {
        push @sql, 'b.bid';
    } else {
        push @sql, 'STRAIGHT_JOIN';
        push @sql, q/b.bid, c.cid, c.uid, b.domain, IFNULL(o.email, u.email) email, IFNULL(o.fio, u.fio) fio, u.lang, o.sms_time, c.name camp_name/;
        push @sql, ', ', ($O{debug_banners_sms} ? 1 : q/FIND_IN_SET('notify_metrica_control_sms', o.sms_flags)/), ' enable_sms';
        push @sql, q/, c.sum AS camp_sum, c.sum_spent AS camp_sum_spent, c.statusModerate AS camp_statusModerate, c.statusShow AS camp_statusShow, c.archived AS camp_archived/;
        push @sql, q/, b.statusModerate, b.statusPostModerate, b.statusShow, b.statusArch/;
        push @sql, q/, c.type AS camp_type, c.finish_time AS camp_finish_time, b.statusMetricaStop/;
        if ($O{new_state} eq 'alive') {
            push @sql, q/, (mfb.bid IS NOT NULL AND mfb.start_time < NOW() - INTERVAL / . FREEZING_TIME . q/ HOUR) AS need_unfreezing/;
        } else {
            push @sql, q/, NULL AS need_unfreezing/;
        }
        if ($O{select_by} eq 'owner') {
            push @sql, q/, NULL AS metrika_counter/;
        } else {
            push @sql, q/, mc.metrika_counter/;
        }
    }
    if ($O{select_by} eq 'owner') {
        push @sql, q/
            FROM campaigns c
        /;
    } else {
        push @sql, q/
            FROM metrika_counters mc
            INNER JOIN campaigns c ON mc.cid = c.cid
        /;
    }
    push @sql, q/
        INNER JOIN camp_options o ON c.cid = o.cid
        INNER JOIN banners b ON c.cid = b.cid
        LEFT JOIN metrica_frozen_banners mfb ON mfb.bid = b.bid
    /;
    push @sql, q/LEFT JOIN users u ON c.uid=u.uid/ unless $O{bids_only};
    push @sql, q/WHERE c.statusEmpty = 'No'/;
    push @sql, q/AND o.statusMetricaControl = 'Yes'/ unless $O{debug_banners};
    if ($O{new_state} eq 'alive') {
        push @sql, q/AND (/;
        push @sql, q/(mfb.bid IS NULL AND b.statusMetricaStop = 'Yes') OR/ unless $O{debug_banners};
        # при оживании смотрим на время замороженности баннера. баннер должен разморозиться по первому событию "оживания" после окончания времени заморозки.
        push @sql, q/(mfb.bid IS NOT NULL AND mfb.start_time < NOW() - INTERVAL / . FREEZING_TIME . q/ HOUR)/;
        push @sql, q/)/;
    } else {
        # отключаем только активные объявления. если в дальнейшем они станут активными, то будут отключены при следующей полной проверке
        push @sql, q/
            AND mfb.bid IS NULL
            AND b.statusMetricaStop = 'No'
            AND c.archived = 'No'
            AND b.statusArch = 'No'
            AND c.sum - c.sum_spent > 0.01
            AND c.statusModerate = 'Yes'
            AND b.statusModerate = 'Yes'
            AND b.statusPostModerate = 'Yes'
            AND c.statusShow = 'Yes'
            AND b.statusShow = 'Yes'
            AND (YEAR(c.finish_time) = 0 OR DATEDIFF(NOW(), c.finish_time) > 0)
        /;
    }
    return join ' ', @sql;
}

=item normalize_domain

    Приводит переданный домен к виду, в котором домены хранятся в metrica_dead_domains:
    - отрезается незначащий префикс www.
    - из Punycode-вида переводится в ASCII
    - приводится к нижнему регистру

=cut

sub normalize_domain {
    my ($domain) = @_;

    return lc(idn_to_ascii(strip_www($domain)));
}

=head2 get_metrica_dead_domains_cond

    Формирует из хеша с данными о доменах условие для отбора строк из таблицы metrica_dead_domains вида
    (domain,uid) IN (('example.com',123), ('example.com', 456)) OR (domain,counter_id) IN (('example.com', 789), ('example.com', 1011))
    Первым параметром принимает ссылку на хеш с данными о доменах.
    Также принимает именованный параметры:
        with_zero_counter_id => 1|0 -- добавлять ли нулевой 0 счётчика (старые записи, когда ещё не смотрели на доп. счётчики)
    Возвращает условия в виде строки. Если данных не достаточно (например, нет ни владельцев ни счётчиков), возвращает undef.

    $domaindata == {
        $domain => {
            by_owner => {
                $uid => [
                    { uid => $uid, domain => $domain, checked_time => $checked_time, counter_id => $counter_id },
                    { uid => $uid, domain => $domain, checked_time => $checked_time2, counter_id => $counter_id2 },
                ],
            },
            by_counter_id => {
                $counter_id => [
                    { uid => $uid, domain => $domain, checked_time => $checked_time, counter_id => $counter_id },
                ],
                $counter_id2 => [
                    { uid => $uid2, domain => $domain, checked_time => $checked_time2, counter_id => $counter_id2 },
                ],
            },
        },
    };
    $cond = get_metrica_dead_domains_cond($domaindata, with_zero_counter_id => 1);
    $cond => q/(domain,uid) IN (('example.com',123), ('example.com', 456)) OR (domain,counter_id) IN (('example.com', 789), ('example.com', 1011), ('example.com', 0))/

=cut

sub get_metrica_dead_domains_cond {
    my ($domaindata, %O) = @_;

    my (@owner_conds, @counter_conds);
    my $dbh = get_dbh(PPC);
    while ( my($domain, $domain_data) = each %{$domaindata} ) {
        my @owners = keys %{$domain_data->{by_owner}};
        if (@owners) {
            push @owner_conds, map { '('.$dbh->quote($domain).','.$dbh->quote($_).')' } @owners;
        }
        my @counter_ids = keys %{$domain_data->{by_counter_id}};
        if (@counter_ids) {
            push @counter_ids, 0 if $O{with_zero_counter_id};
            push @counter_conds, map { '('.$dbh->quote($domain).','.$dbh->quote($_).')' } @counter_ids;
        }
    }
    if (@owner_conds || @counter_conds) {
        my @checked_dead_records;
        if (@owner_conds) {
            push @checked_dead_records, '(domain,uid) IN (', join(',', @owner_conds), ')';
        }
        if (@counter_conds) {
            push @checked_dead_records, 'OR' if @checked_dead_records;
            push @checked_dead_records, '(domain,counter_id) IN (', join(',', @counter_conds), ')';
        }
        my @domains = keys %{$domaindata};
        my @domains_cond = ('domain IN (', join(',', map {$dbh->quote($_)} @domains), ')');
        return join(' ', '(', @domains_cond, 'AND (', @checked_dead_records, '))');
    } else {
        return;
    }
}

sub process_banners {
    my ($new_state, $banners) = @_;

    if ( $new_state ne 'alive' && $new_state ne 'dead' ) {
        die "only 'alive' and 'dead' states are supported";
    }

    my (%changed_banners, %changed_companies, %msg_banner, %sms_banner);

    foreach my $banner (@$banners) {
        $changed_banners{ $banner->{bid} } = undef;
        $changed_companies{ $banner->{cid} } = undef;

        # Не отправляем письмо, если у баннера уже нужное состояние остановки. Такое может быть, например, при возникновениии
        # события "оживания" по замороженному баннеру. Его состояние остановленности в этом случае не меняется, но
        # баннер размораживается.
        next if $banner->{statusMetricaStop} eq ($new_state eq 'alive' ? 'No' : 'Yes');

        # Среди включаемых баннеров могут попадаться баннеры, которые в данный момент не показываются по нескольким причинам сразу.
        # Например, сайт был недоступен, баннер остановился мониторингом Метрики. После этого законичились деньги на кампании
        # и сайт снова заработал. При этом показы, несмотря на снятие statusMetricaStop, не возобновятся. Обманывать пользователя
        # письмом "показы ранее отключенных объявлений возобновлены" не хочется. Поэтому не посылаем уведомления по таким баннерам.
        my $finish_ts;
        $finish_ts = eval { ts_round_day( mysql2unix($banner->{camp_finish_time}) ) } if ($banner->{camp_type} || 'text') eq 'text';
        next if $new_state eq 'alive' && !(
               $banner->{camp_sum} - $banner->{camp_sum_spent} > 0.01

            && $banner->{camp_statusModerate} eq 'Yes'
            && $banner->{statusModerate} eq 'Yes'
            && $banner->{statusPostModerate} eq 'Yes'

            && $banner->{camp_statusShow} eq 'Yes' 
            && $banner->{statusShow} eq 'Yes'

            && $banner->{statusArch} eq 'No' 
            && $banner->{camp_archived} eq 'No'

            && (!$finish_ts || ($finish_ts && $finish_ts > ts_round_day(time)))
        );

        # не посылаем уведомления по гео-кампаниям, чтобы не смущать ничего не подозревающего клиента
        next if $banner->{camp_type} && $banner->{camp_type} eq 'geo';

        my $email = $banner->{email};
        my $fio = $banner->{fio};
        my $uid = $banner->{uid};
        $msg_banner{$email}->{$fio}->{domain_list}->{ idn_to_unicode( strip_www($banner->{domain}) ) } = $banner->{checked_time};
        $msg_banner{$email}->{$fio}->{camp_list}->{ $banner->{cid} } = $banner->{camp_name};
        $msg_banner{$email}->{$fio}->{lang} = $banner->{lang};
        $msg_banner{$email}->{$fio}->{uid} = $uid;

        if ($banner->{enable_sms}) {
            my $sms_time = $banner->{sms_time} ? $banner->{sms_time} : Tools::sms_time2string();
            $sms_banner{$uid}->{domain_list}->{ idn_to_unicode( strip_www($banner->{domain}) ) } = $banner->{checked_time};
            $sms_banner{$uid}->{camp_list}->{ $banner->{cid} } = $banner->{camp_name};
            $sms_banner{$uid}->{sms_time_list}->{$sms_time} = undef;
            $sms_banner{$uid}->{lang} = $banner->{lang};
            $sms_banner{$uid}->{fio} = $fio;
        }
    }
    return (\%changed_banners, \%changed_companies, \%msg_banner, \%sms_banner);
}

sub update_banners_state {
    my ($new_state, $changed_banners) = @_;

    if (%$changed_banners) {
        my @bids = keys %$changed_banners;
        do_sql(PPC, [q#UPDATE banners b SET b.statusBsSynced='No', b.statusMetricaStop=?, b.LastChange=b.LastChange#,
                       WHERE => {'b.bid' => \@bids}
                     ], $new_state eq 'alive' ? 'No' : 'Yes');
    }
}

sub send_msg($$$;$) {
    my ($dbh, $state, $msg_banners, $param) = @_;
    $param ||= {};

    while ( my($email, $data_level1) = each(%$msg_banners) ) {
        while ( my($fio, $data_level2) = each(%$data_level1) ) {
            $log->out("fio=$fio, email=$email, banners");
            my $dlist = $data_level2->{domain_list};
            my $clist = $data_level2->{camp_list};
            $log->out("$state: ". join(',', map {"$_=>$dlist->{$_}"} keys %{$dlist})."; camp ". join(',', map {"$_=>$clist->{$_}"} keys %{$clist}) );

            my $mailvars = {
                            client_fio  => $fio,
                            state       => $state,
                            domain_list => [map {{domain => $_, 'time' => $dlist->{$_}}} keys %{$dlist}],
                            camp_list   => [map {{cid => $_, name => $clist->{$_}}} keys %{$clist}],
                            uid         => $data_level2->{uid},
                           };
            MailService::send_prepared_mail($dbh, $param->{template}, $data_level2->{uid}, $param->{from}, $mailvars) if $email;
        }
    }
}

sub send_problems_with_banners {
    my ($new_state, $msg_banners) = @_;
    send_msg(PPC, $new_state, $msg_banners, {template=>'metrica_monitor_banner', from=>'"Yandex.Direct" <support@direct.yandex.ru>'});
}

sub calculate_sms_time {
    my ($cur_hour, $cur_min, $sms_time_list) = @_;
    my $sms_time;
    my %time_diff;
    foreach my $time (@{$sms_time_list}) {
        my ($hour_from, $min_from, $hour_to, $min_to) = Tools::string2sms_time($time);
        # check sms time
        $hour_to = 24 if $hour_from == 0 && $min_from == 0 && $hour_to == 0 && $min_to == 0; # круглосуточно
        if (
            $cur_hour * 60 + $cur_min >= $hour_from * 60 + $min_from
            && $cur_hour * 60 + $cur_min <= $hour_to * 60 + $min_to
        ) {
            $sms_time = $time;
            last;
        } else {
            if ($cur_hour * 60 + $cur_min > $hour_to * 60 + $min_to) {
                my $diff = $hour_from * 60 + $min_from + (24 - $cur_hour) * 60 - $cur_min;
                $time_diff{$diff} = $time;
            } elsif ($cur_hour * 60 + $cur_min < $hour_from * 60 + $min_from) {
                my $diff = $hour_from * 60 + $min_from - ($cur_hour * 60 + $cur_min);
                $time_diff{$diff} = $time;
            }
        }
    }
    unless ($sms_time) {
        my ($min_diff) = sort {$a<=>$b} keys %time_diff;
        $sms_time = $time_diff{$min_diff};
    }
    return $sms_time;
}

sub send_sms {
    my ($dbh, $state, $sms_banners, $param) = @_;
    $param ||= {};

    while ( my($uid, $user_data) = each(%$sms_banners) ) {
        my $fio = $user_data->{fio};
        my (undef, $cur_min, $cur_hour) = localtime(time);

        $log->out("uid=$uid, banner sms");
        my $dlist = $user_data->{domain_list};
        my $clist = $user_data->{camp_list};
        my $sms_time = calculate_sms_time($cur_hour, $cur_min, [ keys %{$user_data->{sms_time_list}} ]);

        $log->out("$state cid:".join(', ', keys %{$clist})."; domains: ". join(',', map {"$_"} keys %{$dlist}));
        my $smsvars = {
                       client_fio  => $fio,
                       state       => $state,
                       domain_list => [keys %{$dlist}],
                       camp_list   => [map {{cid=>$_,name=>$clist->{$_}}} keys %{$clist}],
                       uid         => $uid,
                       sms_time    => $sms_time,
                       lang        => $user_data->{lang},
                       uid         => $uid,
                      };
        warn "can't send sms" unless(MailService::send_prepared_metrica_sms($dbh, $param->{template}, $smsvars));
    }
}

sub send_sms_banner {
    my ($new_state, $sms_banners) = @_;

    send_sms(PPC, $new_state, $sms_banners, {template=>'notify_metrica_control_sms'});
}
