#!/usr/bin/perl

=head1 DESCRIPTION

    20120912_fill_adgroup_relations.pl [OPTIONS]
        --sleep-coeff - коэффициент для вычисления паузы между итерациями (sleep-coeff * iteration-duration)

=head1 DEPLOY

# approved by zhur
# .migr
    {
        type => 'script',
        when => 'after',
        time_estimate => '5 дней на test2',
        comment => 'заполнение связей между таблицами для групп объявлений        
        
        перед выполнением запустить(для удаление устаревших данных)
        protected/ppcSendWarnClean.pl
        
        можно запускать с коэффициентом для вычисления паузы между итерациями
        (что бы реплики не отставали)
        
        20120912_fill_adgroup_relations.pl --sleep-coeff=7
        если реплики отстают, можно увеличить --sleep-coeff'
    }

=cut

use strict;
use warnings;
use utf8;

use FindBin qw/$Bin/;
use lib "$Bin/../protected";

use Getopt::Long;
use Time::HiRes qw/time sleep/;
use List::MoreUtils qw/uniq/;

use Settings;
use Yandex::DBTools;
use ScriptHelper;

use constant BIDS_PER_ITERATION => 5000;

my $sleep_coeff = 1;
GetOptions(
    'sleep-coeff=i' => \$sleep_coeff,
    'help' => \&usage
);

$log->out('start');

while (my $bids = get_next_bids()) {
    my $st = time;
    fill_relations($bids);
    
    if (my $pause = $sleep_coeff * (time - $st)) {
        $log->out(sprintf "sleep %f", $pause);
        sleep $pause;
    }
}

$log->out('finish');

sub fill_relations {
    
    my $bids = shift;
    
    my $phrases = get_hashes_hash_sql(PPC_HEAVY, ['SELECT bid, pid FROM phrases',
                                        WHERE => {bid => [map {$_->{bid}} @$bids]}]);
    
    my (%banners, %groups);
    foreach (@$bids) {
        
        my $pid = $phrases->{$_->{bid}}->{pid};
        next unless $pid;
        $banners{$_->{bid}} = $pid;
        $groups{$pid} = $_->{cid};
    }

    my $cids = [uniq values %groups];
    my $uids = get_one_column_sql(PPC, ['SELECT DISTINCT uid FROM campaigns', WHERE => {cid => $cids}]);

    do_update_table(PPC, 'bids', {
        modtime__dont_quote => 'modtime',
        pid__dont_quote => sql_case(PPC, 'bid', \%banners)
    }, where => {bid => [keys %banners]});
    
    my $case = sql_case(PPC, 'bid', {map {($_ => 1)} keys %banners},
                default => '0', dont_quote_value => 1);
                
    do_update_table(PPC, 'bids_arc', {
        modtime__dont_quote => 'modtime',
        pid__dont_quote => sql_case(PPC, 'bid', \%banners, default => 0)
    }, where => {cid => $cids, _TEXT => $case});
        
    do_update_table(PPC, 'warnctr', {
        addtime__dont_quote => 'addtime',
        pid__dont_quote => sql_case(PPC, 'bid', \%banners)
    }, where => {
        uid => $uids,
        bid => [keys %banners]
    });
    do_update_table(PPC, 'warnplace', {
        addtime__dont_quote => 'addtime',
        pid__dont_quote => sql_case(PPC, 'bid', \%banners)
    }, where => {
        cid => $cids,
        bid => [keys %banners]
    });
    
    do_update_table(PPC, 'bids_phraseid_associate', {
        logtime__dont_quote => 'logtime',
        pid__dont_quote => sql_case(PPC, 'bid', \%banners, default => 0)
    }, where => {cid => $cids, _TEXT => $case});
    
    do_update_table(PPC, 'phrases', {
        LastChange__dont_quote => 'LastChange',
        cid__dont_quote => sql_case(PPC, 'pid', \%groups)
    }, where => {pid => [keys %groups]});    
    
    do_update_table(PPC, 'banners', {
        LastChange__dont_quote => 'LastChange',
        pid__dont_quote => sql_case(PPC, 'bid', \%banners)
    }, where => {bid => [keys %banners]});
    
}

{
my ($max_bid, $offset);
sub get_next_bids {
    
    $max_bid = get_one_field_sql(PPC, 'SELECT MAX(bid) FROM banners') unless defined $max_bid;
    $offset =  get_one_field_sql(PPC, 'SELECT MIN(bid) FROM banners WHERE pid = 0') unless defined $offset;

    my $bids = [];
    while (!@$bids && $offset <= $max_bid) {
        $bids = get_all_sql(PPC, 'SELECT bid, cid FROM banners WHERE pid = 0 AND bid BETWEEN ? AND ?',
                                $offset, $offset + BIDS_PER_ITERATION);
        $offset += BIDS_PER_ITERATION + 1;
    }
    
    $log->out(sprintf "got %d bids; processed %d/%d", scalar(@$bids), $offset, $max_bid); 
    
    return @$bids ? $bids : undef;
}
}
    