#!/usr/bin/perl

=head1 DESCRIPTION

    20121030_fix_group_relations.pl [OPTIONS]
        --sleep-coeff - коэффициент для вычисления паузы между итерациями (sleep-coeff * iteration-duration)

=head1 DEPLOY

# approved by zhur
# .migr
    {
        type => 'script',
        when => 'after',
        time_estimate => '5 часов',
        comment => 'исправление ошибочно заполненных связей в таблицах bids_arc|bids_phraseid_associate
        
        если деплой завершается c out of memory
        нужно увеличить лимит ulimit -v 10000000
        
        можно запускать с коэффициентом для вычисления паузы между итерациями
        (что бы реплики не отставали)
        
        20121030_fix_group_relations.pl --sleep-coeff=2
        если реплики отстают, можно увеличить --sleep-coeff'
    }

=cut

use strict;
use warnings;
use utf8;

use FindBin qw/$Bin/;
use lib "$Bin/../protected";

use Getopt::Long;
use Time::HiRes qw/time sleep/;
use List::MoreUtils qw/uniq/;

use Settings;
use Yandex::DBTools;
use ScriptHelper;

use constant BIDS_PER_ITERATION => 5000;

my $sleep_coeff = 1;
GetOptions(
    'sleep-coeff=i' => \$sleep_coeff,
    'help' => \&usage
);

$log->out('start');
foreach my $t (qw/bids_arc bids_phraseid_associate/) {
  
    while (my $bids = get_wrong_bids(from => $t)) {
        while (my @part = splice @$bids, 0, BIDS_PER_ITERATION) {
            my $st = time;
            fix_relations(\@part, $t eq 'bids_phraseid_associate');
            if (my $pause = $sleep_coeff * (time - $st)) {
                $log->out(sprintf "sleep %f", $pause);
                sleep $pause;
            }
        }
    }
}
$log->out('finish');

sub fix_relations {
    
    my ($bids, $only_assoc) = @_;
    
    # bid cid pid
    my %banners = map {
        # bid => pid
        ($_->[0] => $_->[2])
    } @$bids;
    my $cids = [uniq map {$_->[1]} @$bids];
    
    my $case = sql_case(PPC, 'bid', {map {($_ => 1)} keys %banners},
                default => '0', dont_quote_value => 1);
                
    unless ($only_assoc) {
        do_update_table(PPC, 'bids_arc', {
            modtime__dont_quote => 'modtime',
            pid__dont_quote => sql_case(PPC, 'bid', \%banners, default => 0)
        }, where => {cid => $cids, _TEXT => $case});
    }
    
    do_update_table(PPC, 'bids_phraseid_associate', {
        logtime__dont_quote => 'logtime',
        pid__dont_quote => sql_case(PPC, 'bid', \%banners, default => 0)
    }, where => {cid => $cids, _TEXT => $case});
}


{
my %is_last;
sub get_wrong_bids {

    my %options = @_;
    my @bids;
    
    return undef if $is_last{$options{from}};

    my $limit = 2000000;
    my $sql;
    
    if ($options{from} eq 'bids_arc') {
        
        $sql = q[SELECT STRAIGHT_JOIN
                    b.bid, b.cid, b.pid
                FROM
                    banners b
                    JOIN bids_arc bi ON b.bid = bi.bid AND b.cid = bi.cid  AND b.pid <> bi.pid
                GROUP BY
                    b.bid
                ORDER BY
                    b.bid
                LIMIT ?];
    } else {
        $sql = q[SELECT 
                    b.bid, b.cid, b.pid
                FROM
                    banners b
                    JOIN bids_phraseid_associate a ON b.bid = a.bid AND b.cid = a.cid AND a.pid <> b.pid
                GROUP BY
                    b.bid
                ORDER BY
                    b.bid
                LIMIT ?];
    }
    
    my $sth = exec_sql(PPC, $sql, $limit);
    while (my @row = $sth->fetchrow_array) {
        push @bids, \@row;
    }
    $is_last{$options{from}} = 1 if @bids < $limit;

    return @bids ? \@bids : undef;
}
}