#!/usr/bin/perl

=head1 DESCRIPTION

    20121206_fix_bids_relations.pl [OPTIONS]
        --sleep-coeff - коэффициент для вычисления паузы между итерациями (sleep-coeff * iteration-duration)

=head1 DEPLOY

# approved by zhur
# .migr
    {
        type => 'script',
        when => 'after',
        time_estimate => '~ 8 часов',
        comment => '
            исправление связей между фразами и группой
            (связь не сохранялась при принятии медиаплана)
        
            можно запускать с коэффициентом для вычисления паузы между итерациями
            (что бы реплики не отставали)
        
            20121206_fix_bids_relations.pl --sleep-coeff=1
                если реплики отстают, можно увеличить --sleep-coeff'
    }

=cut

use strict;
use warnings;
use utf8;

use FindBin qw/$Bin/;
use lib "$Bin/../protected";

use Getopt::Long;
use Time::HiRes qw/time sleep/;
use List::MoreUtils qw/uniq/;

use Settings;
use Yandex::DBTools;
use ScriptHelper;

my $sleep_coeff = 1;
GetOptions(
    'sleep-coeff=i' => \$sleep_coeff,
    'help' => \&usage
);

$log->out('start');
foreach my $t (qw/bids bids_arc/) {
    while (my $phrases = get_wrong_phrases($t)) {
        my $st = time;
        $t eq 'bids' ? fix_bids($phrases) : fix_bids_arc($phrases);
        if (my $pause = $sleep_coeff * (time - $st)) {
            $log->out(sprintf "sleep %f", $pause);
            sleep $pause;
        }
    }
}
$log->out('finish');

sub fix_bids {
    
    my $phrases = shift;
    
    do_update_table(PPC, 'bids', {
        modtime__dont_quote => 'modtime',
        pid__dont_quote => sql_case(PPC, id => $phrases, default => 0)
    }, where => {id => [keys %$phrases]});
}

sub fix_bids_arc {
    
    my $phrases = shift;
    
    my (%cids, %pids);
    foreach (@$phrases) {
        $pids{$_->[0]} = $_->[1];
        $cids{$_->[2]} = 1;
    }
    
    my $case = sql_case(PPC, id => \%pids, default => 0);
    do_update_table(PPC, 'bids_arc', {
        modtime__dont_quote => 'modtime',
        pid__dont_quote => $case
    }, where => {cid => [keys %cids], id => [keys %pids]})
}

sub get_wrong_phrases {

    my $t = shift;

    my $limit = 20000;
    if ($t eq 'bids') {
        my $sth = exec_sql(PPC, "SELECT bi.id, b.pid FROM bids bi JOIN banners b USING(bid) WHERE bi.pid = 0 and b.pid > 0 LIMIT ?", $limit);
        my %phrases;
        while (my ($id, $pid) = $sth->fetchrow_array) {
            $phrases{$id} = $pid;
        }
        return keys %phrases ? \%phrases : undef;
    } else {
        my $sth = exec_sql(PPC, q[
            SELECT STRAIGHT_JOIN bi.id, b.pid, b.cid
            FROM banners b JOIN bids_arc bi USING(cid, bid)
            WHERE bi.pid = 0 AND b.pid > 0
            LIMIT ?], $limit);
        my @phrases;
        while (my @row = $sth->fetchrow_array) {
            push @phrases, \@row
        }
        return @phrases ? \@phrases : undef;
    }
}
