#!/usr/bin/perl

=head1 DESCRIPTION

    20130122_fix_bids_pids.pl [OPTIONS]
        --sleep-coeff - коэффициент для вычисления паузы между итерациями (sleep-coeff * iteration-duration)

=head1 DEPLOY

# approved by zhur
# .migr
    {
        type => 'script',
        when => 'after',
        time_estimate => '~ 3 часа',
        comment => '
            исправление связей между фразами и группой
            (связь не сохранялась при принятии первой помощи)
        
            можно запускать с коэффициентом для вычисления паузы между итерациями
            (что бы реплики не отставали)
        
            20130122_fix_bids_pids.pl --sleep-coeff=1
                если реплики отстают, можно увеличить --sleep-coeff'
    }

=cut

use strict;
use warnings;
use utf8;

use FindBin qw/$Bin/;
use lib "$Bin/../protected";

use Getopt::Long;
use Time::HiRes qw/time sleep/;
use List::MoreUtils qw/uniq/;

use Settings;
use Yandex::DBTools;
use ScriptHelper;
use Yandex::ListUtils;
use Yandex::HashUtils;

my $sleep_coeff = 1;
GetOptions(
    'sleep-coeff=i' => \$sleep_coeff,
    'help' => \&usage
);

$log->out('start');
foreach my $t (qw/bids bids_arc/) {
    my $phrases = get_wrong_phrases($t);
    my @chunks = chunks($t eq 'bids' ? [keys %$phrases] : $phrases, 5000);
    foreach my $chunk (@chunks) {
        my $st = time;
        $t eq 'bids' ? fix_bids(hash_cut($phrases, $chunk)) : fix_bids_arc($chunk);
        if (my $pause = $sleep_coeff * (time - $st)) {
            $log->out(sprintf "sleep %f", $pause);
            sleep $pause;
        }
    }
}
$log->out('finish');

sub fix_bids {
    
    my $phrases = shift;
    
    do_update_table(PPC, 'bids', {
        modtime__dont_quote => 'modtime',
        pid__dont_quote => sql_case(PPC, id => $phrases, default => 0)
    }, where => {id => [keys %$phrases]});
}

sub fix_bids_arc {
    
    my $phrases = shift;
    
    my (%cids, %pids);
    foreach (@$phrases) {
        $pids{$_->[0]} = $_->[1];
        $cids{$_->[2]} = 1;
    }
    
    my $case = sql_case(PPC, id => \%pids, default => 0);
    do_update_table(PPC, 'bids_arc', {
        modtime__dont_quote => 'modtime',
        pid__dont_quote => $case
    }, where => {cid => [keys %cids], id => [keys %pids]})
}

sub get_wrong_phrases {

    my $t = shift;

    if ($t eq 'bids') {
        my $sth = exec_sql(PPC_HEAVY, "SELECT bi.id, b.pid FROM bids bi JOIN banners b USING(bid) WHERE b.pid <> bi.pid");
        my %phrases;
        while (my ($id, $pid) = $sth->fetchrow_array) {
            $phrases{$id} = $pid;
        }
        return \%phrases;
    } else {
        my $sth = exec_sql(PPC_HEAVY, q[
            SELECT STRAIGHT_JOIN bi.id, b.pid, b.cid
            FROM banners b JOIN bids_arc bi USING(cid, bid)
            WHERE b.pid <> bi.pid]);
        my @phrases;
        while (my @row = $sth->fetchrow_array) {
            push @phrases, \@row
        }
        return \@phrases;
    }
}
