#!/usr/bin/perl

=head1 DEPLOY

# approved by lena-san
# .migr
{
  type => 'script',
  when => 'after',
  time_estimate => "1 час",
  comment => "Исправляем некорректные geo (несуществующие регионы, повторные включения, лишние исключения и т.п.)",
}

=cut

use strict;
use warnings;
use utf8;



use FindBin qw/$Bin/;
use lib "$Bin/../protected/";

use List::MoreUtils qw/part uniq none/;
use Settings;
use Yandex::DBTools;
use Yandex::HashUtils qw/hash_map/;
use Yandex::ListUtils;
use Yandex::Validate;
use Yandex::ScalarUtils;
use Yandex::I18n;
use ScriptHelper;
use GeoTools;
use geo_regions;
use geobase;

use open ':std' => ':utf8';

my $SELECT_CHUNK_SIZE = 500_000;
my $APPLY_CHANGES_LIMIT = 10_000;

sub apply_changes($$$);

$log->out('START');

my %tkeys = (
    campaigns => { alias => 'c', key => 'cid', join => ''},
    phrases   => { alias => 'p', key => 'pid', join => ''},
    mediaplan_banners => { alias => 'mb', key => 'mbid', join => '' },
);

for my $table (keys %tkeys) {
    my $table_cnt = 0;
    my $key = $tkeys{$table}->{key};
    my $alias = $tkeys{$table}->{alias};
    $log->out("updating $table");
    my ($min, $max) = get_one_line_array_sql(PPC_HEAVY, "select min($key), max($key) from $table $alias");

    my %update;
    while ($min < $max) {
        $log->out("$table $min $min+$SELECT_CHUNK_SIZE");
        my $rows = get_all_sql(PPC_HEAVY, ["select $key as id, cid, geo from $table $alias $tkeys{$table}->{join}", where => { 
            geo__is_not_null => 1, 
            geo__ne => '',
            $key.'__between' => [ $min, $min+$SELECT_CHUNK_SIZE],
        }]);
        $log->out("selected ".(scalar @$rows)." rows");
        for my $row (@$rows) {
            my $geo = fix_geo($row->{geo});
            $geo = '' unless defined $geo;
            next if str($geo) eq str($row->{geo});
            # группируем апдейты по результирующему значению регионов, т.к. часто в баннерах одной кампании рядом одинаковые значения
            $update{$geo}->{old_geo}->{$row->{geo}} = 1;
            push @{$update{$geo}->{resync_data}->{$row->{cid}}}, $row->{id};
            $table_cnt++;
            $log->out("FINAL $row->{id} $row->{geo} => ".($geo||'(undef)'));
            if (keys %update >= $APPLY_CHANGES_LIMIT) {
                apply_changes($table, $key, \%update);
                %update = ();
            }
        }
        $min += $SELECT_CHUNK_SIZE;
    }
    $log->out("$table : $table_cnt records fixed");
    if (keys %update) {
        apply_changes($table, $key, \%update);
        %update = ();
    }
}


$log->out('FINISH');


sub apply_changes($$$) {
    my ($table, $key, $update_data) = @_;

    while (my($new_geo, $data) = each %$update_data) {
        my $key_values = [];
        if ($table eq 'campaigns') {
            my @cids = keys %{$data->{resync_data}};
            do_mass_insert_sql(PPC,"INSERT IGNORE INTO bs_resync_queue (cid) VALUES %s", [map { [$_] } @cids]);
            $key_values = \@cids;
        } elsif ($table eq 'phrases') {
            my @resync_data;
            while (my($cid, $ids) = each %{$data->{resync_data}}) {
                for my $id(@$ids) {
                    push @resync_data, [$cid, $id];
                }
                push @$key_values, @$ids;
            }
            do_mass_insert_sql(PPC,"INSERT IGNORE INTO bs_resync_queue (cid, pid) VALUES %s", \@resync_data);
        } elsif ($table eq 'mediaplan_banners') {
            while (my($cid, $ids) = each %{$data->{resync_data}}) {
                push @$key_values, @$ids;
            }
        } else {
            $log->die("Unknown table $table");
        }
        do_update_table(PPC, $table, {geo => $new_geo}, where => { $key => $key_values, geo => [keys %{$data->{old_geo}}] });
    }
}

{
sub _region_text {
    my ($region_id) = @_;
    return '' unless defined $region_id;

    my $id;
    $id = $1 if $region_id =~ /^[,\s]*(-?\d+)[,\s]*$/;
    return '' unless defined $id;

    if (defined $id && $id == 0) {
        return 'Весь мир';
    } elsif (defined $id) {
        if (exists $geobase::Region{abs($id)} && $geobase::Region{abs($id)}->{name}) {
            return "$region_id (" . $geobase::Region{abs($id)}->{name} . ")";
        } else {
            return "$region_id [не существует]";
        }
    } else {
        die "cannot convert region: $region_id";
    }
}

my %geo_cache;

sub fix_geo
{
    my ($geo) = shift;
    my $geo_raw = $geo;
    $geo =~ s![^0-9,-]!!g; # да, и такое тоже случается
    my $safe_cnt = 0;
    my %geo_error;
    my @changes;

    if (exists $geo_cache{$geo_raw}) {
        $geo = $geo_cache{$geo_raw};
        push @changes, "cache hit '$geo_raw' => '$geo'";
        #$log->out("cache hit '$geo_raw' => '$geo'");
    }
    else {
        # отсекаем невалидные номера регионов (,-, или ,1-2, или ...)
        my @geos_raw = split /,+/, $geo;
        my @geos;
        for my $cur_geo (@geos_raw) {
            # выкидываем вычитание всей Земли как бесполезное
            if (is_valid_int($cur_geo) && $cur_geo ne '-0') {
                push @geos, $cur_geo;
            } else {
                push @changes, "удалён невалидный регион $cur_geo";
            }
        }
        $geo = join ',', @geos;
        $geo ||= '0';

        while (defined my_validate_geo($geo, \%geo_error)) {
            die("'$geo' this is taking too long") if $safe_cnt++ > 500;
            if ($geo_error{not_exists}) {
                if ($geo_error{not_exists} =~ /^-/) {
                    $geo =~ s!$geo_error{not_exists}([,\-]|\b)!$1!;
                    #$log->out("'$geo_raw' removing non-existing minus region '$geo_error{not_exists}' => '$geo'");
                    push @changes, "удалён несуществующий минус-регион " . _region_text($geo_error{not_exists});
                }
                else {
                    my $new_region = find_closest_region($geo_error{not_exists});
                    if (!defined $new_region) {
                        $new_region = ''; # удаляем совсем несуществующие регионы
                    }
                    $geo =~ s!([,\-]|\b)$geo_error{not_exists}([,\-]|\b)!$1$new_region$2!;
                    #$log->out("'$geo_raw' replacing '$geo_error{not_exists}' with '$new_region' => '$geo'");
                    push @changes, "несуществующий регион " . _region_text($geo_error{not_exists}) . " заменён на " . _region_text($new_region);
                }
            }
            elsif ($geo_error{duplicate}) {
                $geo =~ s!([,\-]|\b)$geo_error{duplicate}([\-,]|\b)!$2!;
                #$log->out("'$geo_raw' duplicate $geo_error{duplicate}, revmoving first occurence => '$geo'");
                push @changes, "удалён дублирующийся регион " . _region_text($geo_error{duplicate});
            }
            elsif ($geo_error{minus_only}) {
                $geo = '0,' . $geo;
                #$log->out("'$geo_raw' minus-regions only => undef");
                push @changes, "только минус регионы";
            }
            elsif ($geo_error{includes}) {
                my ($bigger, $less) = @{$geo_error{includes}};
                $geo =~ s!([\-,]|\b)$less([,\-]|\b)!$2!x;
                #$log->out("'$geo_raw' removing '$less' since '$bigger' includes it => '$geo'");
                push @changes, _region_text($bigger) . " включает в себя " . _region_text($less);
            }
            elsif ($geo_error{excludes}) {
                my ($excluded, $region) = @{$geo_error{excludes}};
                $geo =~ s!(\b|[\-,])$region(\b|[\-,])!$2!x;
                #$log->out("'$geo_raw' '$region' is excluded by '$excluded' => '$geo'");
                push @changes, _region_text($excluded) . " полностью вычитается " . _region_text($region);
            }
            elsif ($geo_error{nowhere_to_exclude}) {
                $geo =~ s!,?\-$geo_error{nowhere_to_exclude}([\-,]|\b)!$1!;
                #$log->out("'$geo_raw' minus-region '$geo_error{nowhere_to_exclude}' is not in any of plus-regs => '$geo'");
                push @changes, _region_text($geo_error{nowhere_to_exclude}) . " неоткуда исключить";
            }
            %geo_error = ();
        }
        # если никаких существенных изменений не сделали, то не пересортируем регионы в нужном порядке (refine_geoid), чтобы
        # уменьшить единоразовое количество изменений. порядок регионов влияет только на отображение в интерфейсе.
        # при последующих сохранениях баннеров, регионы будут пересортировываться через refine_geoid.
        $geo = my_refine_geoid($geo) if $geo and $geo ne $geo_raw and @changes;
        $geo_cache{$geo_raw} = $geo;
    }
    # если никаких изменений не сделали, то возвращаем регионы в исходном виде, незатронутом refine_geo
    return (@changes > 0) ? $geo : $geo_raw;
}
}

sub find_closest_region
{
    my $id = shift;
    my $sign = $id < 0 ? '-' : '';
    $id = abs $id;
    my $parents = $geobase::Region{$id}->{parents};
    return undef unless $parents && @$parents;
    my $p = $parents->[-1];
    while (!exists $geo_regions::GEOREG{$p}) {
        $p = find_closest_region($p);
        return undef unless $p;
    }
    return $sign.$p;
}

=begin testing

    my %tests = (
        '225,-1' => '225,-1',
        '1' => '1',
        '225,1,-2' => '225,-2',
        '225,-2,1' => '225,-2',
        '225,-2,-1' => '225,-2,-1',
        '225,-1,-2' => '225,-1,-2',

        '225,143,-24' => '143,225',
        '143,24' => '143,10904',
        '225,143,225' => '143,225',
        '225,143,213' => '143,225',
        '213,143,-225' => '143',
        '10904,143,-213' => '143,10904',
        '213,-225' => '0,-225',
        '255' => '',
        '255,225,143' => '143,225',
        '0,-0' => '0',
        '0,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,-0,' => '0',
        '225,-0' => '225',
        '225,-213,-' => '225,-213',
        '225,28,29,-1,-2' => '225,-1,-2',
        'http://www.shop.mts.ru/accessories/20646147.html?utm_source=market.yandex.ru&amp;amp;utm_medium=cpc&amp;amp;utm_term=explay_mp3_m10_8gb_20646147&amp;amp;utm_content=market.yandex.ru&amp;amp;articul=0202-0335&amp;amp;directlink=yes' => '0',
    );

    while (my ($src_region, $result_region) = each %tests) {
        is(fix_geo($src_region), $result_region, "testing fix_geo for $src_region");
    }

=end testing


# Функции дальше перенесены из GeoTools в том виде, как они будут выглядеть после включения валидации, с добавлением префикса my_


=head2 refine_geoid

 Нормализовать список id регионов, получить GeoFlag

    $geo_str = refine_geoid('213,-10650', \$geoflag);
    $geo_str = refine_geoid([213,-10650], \$geoflag);
    $geoflag => 1|0

=cut

sub my_refine_geoid {
    my ($idlist, $geoflag) = @_;

    return undef unless defined $idlist;

    my @geo_ids;
    if (ref($idlist) eq 'ARRAY') {
        @geo_ids = uniq @$idlist;
    } elsif(ref($idlist) eq '') {
        $idlist =~ s/\s+//g;
        @geo_ids = uniq grep {/\S/} split( /\s*,\s*/, $idlist );
    } else {
        die 'invalid idlist';
    }

    @geo_ids = grep {is_valid_int($_) && exists $geo_regions::GEOREG{abs($_)}} @geo_ids;

    my ($plus_regions, $minus_regions) = part {$_ >= 0 ? 0 : 1} @geo_ids;

    if (!$plus_regions || !@$plus_regions || (@$plus_regions == 1 && !$plus_regions->[0])) {
        $plus_regions = [0];
    }

    my %minus_regions_for_plus_region;
    if ($minus_regions && @$minus_regions) {
        for my $minus_region (map {abs} @$minus_regions) {
            my $plus_region = get_geo_projection($minus_region, {geo_list => $plus_regions});
            if (defined $plus_region) {
                push @{$minus_regions_for_plus_region{$plus_region}}, $minus_region;
            } # минус-регионы, которые неоткуда вычитать, выкидываем
        }
    }

    my @refined_geo_ids;
    for my $plus_region (sort {$a <=> $b} @$plus_regions) {
        push @refined_geo_ids, $plus_region;
        if ($minus_regions_for_plus_region{$plus_region} && @{$minus_regions_for_plus_region{$plus_region}}) {
            push @refined_geo_ids, map {-$_} sort {$a <=> $b} @{$minus_regions_for_plus_region{$plus_region}};
        }
    }

    # если 1 - то нужно показать регион, если 0 - то не надо
    $$geoflag = (none { !$geo_regions::GEOREG{abs($_)}->{geo_flag} } @refined_geo_ids) ? 1 : 0; 

    return join ',', @refined_geo_ids;
}

=begin testing

    is(_get_intersections_in_regions([10650, 10658]), undef, 'непересекающиеся регионы');
    is(_get_intersections_in_regions([213]), undef, 'единственный регион');
    is(_get_intersections_in_regions([0]), undef, 'единственный нулевой регион');
    cmp_deeply(_get_intersections_in_regions([213, 0, 219]), [213, 0], 'пересекающиеся регионы');
    cmp_deeply(_get_intersections_in_regions([10002, 1, 213]), [213, 1], 'пересекается не первый в списке регион');

=end testing

=head2 _get_intersections_in_regions

    Проверяет, есть ли среди переданных регионов пересекающиеся.
    Если есть -- возвращает первую пару пересекающихся регионов.

=cut

sub _get_intersections_in_regions {
    my ($regions) = @_;

    for (my $i=0; $i < @$regions; $i++) {
        my $region = $regions->[$i];
        my @other_regions = @$regions;
        splice @other_regions, $i, 1;
        my $intersect_region = get_geo_projection($region, {geo_list => \@other_regions});
        if (defined $intersect_region) {
            return [$region, $intersect_region];
        }
    }

    return undef;
}

=head2 validate_geo

    Проверяет строку с идентификаторами регионов вида "255,-128,-42"
    Возвращает undef если строка корректна и текст ошибки в противном случае
    Вторым параметром принимает ссылку на хеш, в который в случае ошибки заносятся подробности.
    Возвращается первая найденная ошибка и её подробности. Порядок проверки не гарантируется.
    Минус-регионы в подробностях возвращаются положительными числами. Понять, что это был минус-регион
    можно в зависимости от ошибки.

    $error_text = validate_geo("255,-128,-42");
    $error_text = validate_geo("255,-128,-42", \%errors);
    $error_text => undef | "error abcd"
    %errors = (
        not_exists => $region_str, # Неверный или несуществующий регион %s
        duplicate => $region_id, # Регион %s повторяется несколько раз
        minus_only => 1, # Регионы не могут только исключаться
        includes => [$region_id_bigger, $region_id_smaller], # Регион %s уже включает в себя регион %s
        excludes => [$region_id_bigger, $region_id_smaller], # Регион %s полностью исключает регион %s
    );

=cut

sub my_validate_geo {
    my ($geo_str, $out_err) = @_;

    my @geo_ids;
    if (defined $geo_str) {
        $geo_str =~ s/\s+//g;
        @geo_ids = grep {/\S/} split( /\s*,\s*/, $geo_str );
    }

    @geo_ids = (0) unless @geo_ids;

    my %already_seen_geo_ids;
    for my $geo_id ( @geo_ids ) {
        if ( !is_valid_int($geo_id) || !exists $geo_regions::GEOREG{ abs($geo_id) } ) {
            $out_err->{not_exists} = $geo_id if $out_err;
            return iget('Неверный или несуществующий регион %s', $geo_id);
        }
        if ($already_seen_geo_ids{$geo_id}) {
            $out_err->{duplicate} = $geo_id if $out_err;
            return iget('Регион %s повторяется несколько раз', get_geo_names($geo_id));
        }
        $already_seen_geo_ids{$geo_id} = 1;
    }

    my ($plus_regions, $minus_regions) = part {$_ >= 0 && $_ !~ /^-/ ? 0 : 1} @geo_ids;
    $minus_regions = [map {abs($_)} @$minus_regions] if $minus_regions;
    $plus_regions = [map {abs($_)} @$plus_regions] if $plus_regions;

    if ($minus_regions && @$minus_regions && !($plus_regions && @$plus_regions)) {
        $out_err->{minus_only} = 1 if $out_err;
        return iget('Регионы не могут только исключаться');
    }

    # регионы не должны включаться друг в друга
    for my $regions($plus_regions, $minus_regions) {
        next unless $regions && @$regions;
        my $intersect_data = _get_intersections_in_regions($regions);
        if ($intersect_data) {
            $out_err->{includes} = [ $intersect_data->[1], $intersect_data->[0] ] if $out_err;
            return iget("Регион %s уже включает в себя регион %s", get_geo_names($intersect_data->[1]), get_geo_names($intersect_data->[0]));
        }
    }

    # минус-регионы не должны целиком перекрывать плюс-регионы
    if ($plus_regions && @$plus_regions && $minus_regions && @$minus_regions) {
        for my $region(@$plus_regions) {
            my $excluded_by_minus_region = get_geo_projection($region, {geo_list => $minus_regions});
            if (defined $excluded_by_minus_region) {
                $out_err->{excludes} = [ $excluded_by_minus_region, $region ] if $out_err;
                return iget("Регион %s полностью исключает регион %s", get_geo_names($excluded_by_minus_region), get_geo_names($region));
            }
        }
    }

    # каждому минус-региону должен соответствовать плюс-регион
    if ($minus_regions && @$minus_regions) {
        for my $region(@$minus_regions) {
            my $excluded_from_this_plus_region = get_geo_projection($region, {geo_list => $plus_regions});
            if (!defined $excluded_from_this_plus_region) {
                $out_err->{nowhere_to_exclude} = $region if $out_err;
                return iget("Регион %s неоткуда исключать", get_geo_names($region));
            }
        }
    }

    return undef;
}
