#!/usr/bin/perl

=head1 DEPLOY

# approved by pankovpv
# .migr
    {
        type => 'script',
        when => 'after',
        time_estimate => '> 120 часов',
        comment => q[
            Добавляем новое поле `goals_income` в таблицы статистики ppcstat_data: order_goals_nnnnnnn, order_stat_nnnnnnn, order_stat_small_nnnnnnn,
                         а также `fp_shows`, `fp_shows_pos`, `fp_clicks`, `fp_clicks_pos` в таблицы статистики: order_stat_nnnnnnn, order_stat_small_nnnnnnn,
            Убираем unsigned для полей с исчисляемыми данными: `sessions_len`, `sessions_num`, `goals_num`

            Поддерживаемые параметры:
                --one-shard - для тестовых систем, будет использован только первый шард
                --test-only - запуск без применения альтера
                --skip-column-check - пропуск проверки наличия колонки goals_num в таблицах

            Скрипт можно останавливать/перезапускать.
            При первом запуске лучше указать --skip-column-check для экономии времени.
            После перезапуска можно --skip-column-check указывать также, но в этом случае в логах могут появиться сообщения о том,
            что нужные колонки уже были добавлены.
        ],
    }

=cut

use strict;
use warnings;

use my_inc '..';

use utf8;
use open ':std' => ':utf8';

use lib::abs '../protected';

use Settings;
use Yandex::DBTools;
use ScriptHelper;
use Tools;
use Parallel::ForkManager;

my ($one_shard, $test_only, $skip_column_check);
GetOptions(
    'one-shard' => \$one_shard,
    'test-only' => \$test_only,
    'skip-column-check' => \$skip_column_check,
);

$log->out('START');

my %result;
{;
    my %childs;

    local $SIG{INT} = sub {
        kill 9 => keys %childs;
        exit 1;
    };
    local $SIG{TERM} = sub {
        kill 9 => keys %childs;
        exit 1;
    };

    my $shards = get_db_childs("ppcstat");
    $shards = [shift @$shards] if $one_shard && @$shards;

    my $pm = new Parallel::ForkManager(scalar @$shards);
    $pm->run_on_finish(sub {
        my ($pid, $exit, $shard) = @_;
        $result{$shard} = defined $exit && $exit == 0 ? 1 : 0;
    });

    for my $shard (@$shards) {
        $log->out("starting process for $shard");
        if (my $pid = $pm->start($shard)) {
            $childs{$pid} = 1;
            next;
        }

        $SIG{INT} = $SIG{TERM} = 'DEFAULT';

        my ($shard_id) = ($shard =~ /^ppcstat:(\d+)$/);
        my $log_file_name = get_script_name(shardid => $shard_id).".log";
        $0 .= " SHARD - $shard_id";
        my $log = Yandex::Log->new(log_file_name => $log_file_name, date_suf => '%Y%m%d');
        eval {
            $log->out("$shard start");
            add_columns($shard, $log);
            $log->out("$shard finish");
            1;
        } or do {
            $log->out("$shard: $@");
        };

        $pm->finish($@ ? 1 : 0);
    }

    $pm->wait_all_children;
};

sub add_columns {
    my ($shard, $log) = @_;

    my $dbh = get_dbh($shard);

    my $alter = q{
        ALTER TABLE ppcstat_data.%s
            ADD COLUMN `goals_income` decimal(16,6) NOT NULL DEFAULT '0' AFTER `goals_num`,
            ADD COLUMN `fp_shows` int NOT NULL DEFAULT 0 AFTER `goals_income`,
            ADD COLUMN `fp_shows_pos` int NOT NULL DEFAULT 0 AFTER `fp_shows`,
            ADD COLUMN `fp_clicks` int NOT NULL DEFAULT 0 AFTER `fp_shows_pos`,
            ADD COLUMN `fp_clicks_pos` int NOT NULL DEFAULT 0 AFTER `fp_clicks`,
            MODIFY COLUMN `sessions_len` int NOT NULL DEFAULT 0,
            MODIFY COLUMN `sessions_num` int NOT NULL DEFAULT 0,
            MODIFY COLUMN `goals_num` int NOT NULL DEFAULT 0
    };

    my $alter_goals = q{
        ALTER TABLE ppcstat_data.%s
            ADD COLUMN `goals_income` decimal(16,6) NOT NULL DEFAULT '0' AFTER `goals_num`,
            MODIFY COLUMN `goals_num` int NOT NULL DEFAULT 0
    };

    my $tables = get_one_column_sql($dbh, q{
        SELECT t.TABLE_NAME
        FROM information_schema.tables t
        WHERE
            t.TABLE_SCHEMA = "ppcstat_data" AND (}.join(' OR ', map { "t.TABLE_NAME LIKE \"${_}%\"" } qw/order_goals order_stat order_stat_small/).q{)
    });
    my ($q, $total) = (0, scalar @$tables);

    $log->out("$shard has $total tables");
    @{$dbh}{qw/PrintError RaiseError/} = (0, 1);
    foreach my $t (@$tables) {
        my $is_altered = $skip_column_check ? 0 : get_one_field_sql($dbh, q{
            SELECT count(*)
            FROM information_schema.columns c
            WHERE
                c.TABLE_SCHEMA = "ppcstat_data" AND c.TABLE_NAME = ? AND c.COLUMN_NAME = "goals_income"
        }, $t);

        if (!$is_altered && !$test_only) {
            $log->out("altering table $t");
            eval {
                do_sql($dbh, sprintf($t =~ /^order_goals/ ? $alter_goals : $alter, $t));
                1;
            } or do {
                $log->out("$shard $t: $@");
            };
        }

        $log->out("$shard $q/$total tables done") unless ++$q % 1000;
    }
}

$log->out("$_ exit status: ".($result{$_} ? "OK" : "ERROR")) for keys %result;

$log->out('FINISH');
