#!/usr/bin/perl

=head1 DEPLOY

# approved by pankovpv
# .migr
{
  type => 'script',
  when => 'after',
  time_estimate => "3-4 минуты",
}

=cut

use warnings;
use strict;
use utf8;

use my_inc '..';

use Yandex::DBTools;

use Settings;
use ScriptHelper;

use ShardingTools;
use Primitives qw/reverse_domain/;

use Yandex::IDN qw/idn_to_ascii idn_to_unicode/;
use Yandex::MirrorsTools::Hostings qw/strip_www/;
use Yandex::ListUtils qw/xisect/;
use Yandex::HashUtils;

use List::MoreUtils qw/uniq/;

use constant SETTLING_TIME => 10;
my $FILE_WITH_DOMAINS_TO_PROCESS = '/var/www/ppc.yandex.ru/deploy/20150323_switch_on_banners_of_alive_domains.data';

$log->out('START');

# получем список доменов, состояние баннеров по которым нужно проверить
$log->out('Fetching domains to process');
open(my $fh, "<:encoding(UTF-8)", $FILE_WITH_DOMAINS_TO_PROCESS) or $log->die("Error opening $FILE_WITH_DOMAINS_TO_PROCESS:$!");
my @domains_to_process = map { chomp($_); $_ } <$fh>;
my @revdomains_to_process = map {reverse_domain $_} 
                            uniq map { ($_, idn_to_unicode $_) } 
                            map { ($_, "www.$_") } 
                            map { normalize_domain($_) }
                            grep { $_ } @domains_to_process;
$log->out('domains to process: ', @domains_to_process);

# получем список действительно мёртвых доменов
$log->out('Fetching dead domains data');
my $dead_data = get_all_sql(MONITOR, 'SELECT uid, counter_id, domain
                                        FROM metrica_dead_domains
                                       WHERE start_time < NOW() - INTERVAL ? MINUTE'
                                           , SETTLING_TIME );
$log->out('dead domains:', @$dead_data);
# группируем мёртвые домены по uid'у чтобы чуть-чуть сократить SQL-условие
my %dead_domains_by_owner = ();
for my $dead_rec(@$dead_data) {
    push @{ $dead_domains_by_owner{$dead_rec->{uid}} }, $dead_rec->{domain};
}

# собираем SQL-условие вида (b.reverse_domain IN ('domain1.ru', 'www.domain1.ru', 'domain2.ru') AND c.uid => 123456) OR (b.reverse_domain IN ('domain3.ru', 'www.domain3.ru') AND c.uid => 32323)
# по которому в дальнейшем будем выбирать баннеры
my @cond_by_owner_groups;
while ( my($uid, $dead_domains) = each %dead_domains_by_owner ) {
    my @new_domains = uniq map { ($_, idn_to_unicode $_) } map { ($_, "www.$_") } @$dead_domains;
    my @revdomains = map {reverse_domain $_} @new_domains;

    push @cond_by_owner_groups, 'OR' if @cond_by_owner_groups;
    push @cond_by_owner_groups, '(', { 'b.reverse_domain' => \@revdomains, 'c.uid' => $uid }, ')';
}

# собираем SQL-условие вида (b.reverse_domain IN ('domain1.ru', 'www.domain1.ru', 'domain2.ru') AND mc.metrika_counter => 123456) OR (b.reverse_domain IN ('domain3.ru', 'www.domain3.ru') AND mc.metrika_counter => 7890123)
# по которому в дальнейшем будем выбирать баннеры
my @cond_by_counter_groups;
for my $dead_rec (@$dead_data[0..3]) {
    my @new_domains = uniq map { ($_, idn_to_unicode $_) } map { ($_, "www.$_") } $dead_rec->{domain};
    my @revdomains = map {reverse_domain $_} @new_domains;

    push @cond_by_counter_groups, 'OR' if @cond_by_counter_groups;
    push @cond_by_counter_groups, '(', { 'b.reverse_domain' => \@revdomains, 'mc.metrika_counter' => $dead_rec->{counter_id} }, ')';
}

for my $shard (ppc_shards()) {
    # выбираем баннеры (по PPC_HEAVY после синхронизации с мастером), у которых судя по данным metrica_dead_domains не должно быть statusMetricaStop
    $log->msg_prefix("[shard_$shard] ");
    $log->out("start processing banners");
    my $sql_begin = q#
        SELECT b.bid, b.cid, c.uid, b.domain
        FROM banners b
        INNER JOIN campaigns c ON c.cid = b.cid
        INNER JOIN camp_options o ON c.cid = o.cid #;
    my $sql_where = {'c.statusEmpty' => 'No',
                     'o.statusMetricaControl' => 'Yes',
                     'b.statusMetricaStop' => 'Yes',
                     'b.reverse_domain' => \@revdomains_to_process};

    $log->out('wait master');
    wait_master([PPC(shard => $shard), PPC_HEAVY(shard => $shard)]);
    
    $log->out('Fetching banners by owner');    
    my $banners_by_owner = get_hashes_hash_sql(PPC_HEAVY(shard => $shard), [$sql_begin, 
                                                                    'where' => $sql_where, 'AND NOT (', @cond_by_owner_groups, ')']);

    $log->out('Found ' . scalar(keys %$banners_by_owner) . ' banners by owner');

    my $banners_by_counter = get_hashes_hash_sql(PPC_HEAVY(shard => $shard), [$sql_begin, 
                                                                      'LEFT JOIN metrika_counters mc 
                                                                              ON mc.cid = c.cid AND (', @cond_by_counter_groups, ')',
                                                                      'where' => $sql_where, 'AND mc.cid IS NULL']);
    $log->out('Found ' . scalar(keys %$banners_by_counter) . ' banners by counter');

    my $bugged_banners = hash_cut $banners_by_owner, xisect([keys %$banners_by_owner], [keys %$banners_by_counter]);

    $log->out('Found ' . scalar(keys %$bugged_banners) . ' banners to update');
    # собираем из выбранных баннеров список bid'ов, с которых надо снять statusMetricaStop, и cid'ов, которые надо добавить в очередь экспорта в БК metrica
    my (@bids_to_update, %cids_to_sync);
    for my $banner(values %$bugged_banners) {
        $log->out($banner);
        push @bids_to_update, $banner->{bid};
        $cids_to_sync{ $banner->{cid} } = undef;
    }

    while ( my @bids = splice(@bids_to_update, 0, 1000) ) {
        $log->out('Updating one more banners bundle');
        # условие на statusMetricaStop нужно, чтобы лишний раз не поставить statusBsSynced на кампанию, 
        # с которой statusMetricaStop сняли между выборкой баннеров и снятьем флага
        do_update_table(PPC(shard => $shard), 'banners', {statusMetricaStop => 'No', statusBsSynced => 'No'}, 
                                                         where => {bid => \@bids, statusMetricaStop => 'Yes'});
    }

    # добавляем затронутые кампании в очередь metrica на экспорт в БК
    $log->out('Adding ' . scalar(keys %cids_to_sync) . ' touched campaigns to BS export queue named metrica');
    do_mass_insert_sql(PPC(shard => $shard), 'INSERT INTO bs_export_specials (cid, par_type) VALUES %s 
                                                               ON DUPLICATE KEY UPDATE cid=cid',
                                              [ map {[$_, 'metrica']} keys %cids_to_sync ] );
    $log->out("finish processing banners");
}
$log->out('DONE');

sub normalize_domain {
    my ($domain) = @_;

    return lc(idn_to_ascii(strip_www($domain)));
}
