#!/usr/bin/perl

=head1 DEPLOY

# approved by pankovpv
# .migr
{
  type => 'script',
  when => 'after',
  time_estimate => "на devtest 2 часа (1_800_000 записей / 15_000 зап/мин)",
  comment => 'Запускать после подтверждения n-boy (нужно убедиться что файл со статистикой актуален, и лежит по указанному пути).

  			  Втягиваем доход по целям за прошедшие периоды из файла. 
  			  Перед запуском выполнить scp root@ppcdev1:/tmp/campaigns_goals_income_stat /tmp/campaigns_goals_income_stat
  			  Если начнут сильно отставать реплики - можно перезапустить скрипт с параметром --sleep-coef 3	 (по умолчанию 1)

  			  ПЕСОЧНИЦА: следует запустить с параметром --sandbox (работает пару минут)'
}

=cut

use strict;
use warnings;
use utf8;

use my_inc "..";

use List::MoreUtils qw/any zip uniq/;

use Settings;
use ScriptHelper;
use BSStatImport;
use Currency::Rate;

use Yandex::DBTools;
use Yandex::DBShards;
use Yandex::ListUtils qw/xminus chunks/;
use Yandex::Retry qw/relaxed/;
use Yandex::HashUtils;

my $SLEEP = 1;
my $SELECT_LIMIT = 1_000_000;
my $UPDATE_LIMIT = 2_000;
my $UPDATE_LOG_LIMIT = 10_000;
my $FILE_WITH_STAT = '/tmp/campaigns_goals_income_stat';
my $GOALS_INCOME_START_DATE = 20140501;

my %O;
extract_script_params(
    'sandbox' => \$O{sandbox},
    'sleep-coef=i' => \$SLEEP,
) or die "can't parse options";

$log->out('START');

if ($O{sandbox}) {
	# заполняем goals_income в песочнице (генерим значения автоматически)
	$log->out('start updating goals_income in sandbox');
	# в песочнице многих заказов нет в метабазе, а шард при этом только один, потому задаем шард жестко
	do_sql(PPC(shard => 1), ["update bs_order_target_stat set goals_income = goals_num*(100 + round(rand()*100))",
							   where => {stat_date__gt => $GOALS_INCOME_START_DATE}]);
	$log->out('finish updating goals_income in sandbox');
} else {
	open (my $fh, '<', $FILE_WITH_STAT) or $log->die("Can't open file $FILE_WITH_STAT: $!");
	my $field_names_row = <$fh>;

	$log->die("Invalid line with field names: $field_names_row") unless $field_names_row =~ /^#([\w\t]+)$/;
	my @field_names = split /\t/, $1, -1;
	my @required_fields = qw/OrderID  UpdateTime  TargetType  PriceCur/;
	    
    my $missing_fields = xminus(\@required_fields, \@field_names);
    if (@$missing_fields) {
        my $missing_fields_str = join(', ', @$missing_fields);
        $log->die("Fields $missing_fields_str are required but was not found in response header: $field_names_row");
    }

	my (@lines, $processed);
	my %fields_format = (	OrderID 	=> 'uint',
				            UpdateTime 	=> 'datetime',
				            TargetType 	=> {type => 'uint', max => 9},
				            PriceCur 	=> {type => 'uint', min => 0}, );

	while (my $line = <$fh>) {
		next unless $line =~ /\S+/;
		chomp($line);

		my @values = split /\t/, $line, scalar(@field_names);
		my %named_line = zip(@field_names, @values);

		if (any {BSStatImport::_check_field_value($named_line{$_}, $fields_format{$_})} keys %fields_format ) {
			$log->out("Skip line with incorrect fields format: $line");
			next;
		}
		if ($named_line{PriceCur} == 0) {
			$log->out("Skip line with zero PriceCur: $line");
			next;
		}

		push @lines, \%named_line;

		if (scalar(@lines) >= $SELECT_LIMIT) {
			process_lines_from_file(\@lines);
		    
		    $processed += scalar(@lines);
		    $log->out("$processed rows processed (file import)");

		    @lines = ();
		}
	}
	if (@lines) {
		process_lines_from_file(\@lines);

		$processed += scalar(@lines);
		$log->out("$processed rows processed (file import)");
	}
	$log->out('finish import file with stats data')
}

$log->out('FINISH');

sub process_lines_from_file {
	my $lines = shift;
	
	my $updated = 0;
	my $updated_after_log = 0;
	foreach_shard OrderID => $lines, sub {	
		my ($shard, $sharded_lines) = @_;	

		my $order_currency = get_hash_sql(PPC(shard => $shard), ['
                SELECT c.OrderID
                     , IFNULL(c.currency, "YND_FIXED") AS currency
                  FROM campaigns c
              ', WHERE => {'c.OrderID' => [uniq map {$_->{OrderID}} @$sharded_lines]}
        ]);

		my %lines_by_date_and_targettype = ();
		for my $line (@$sharded_lines) {
			$line->{PriceCur} /= 1e6;
		
	        # БК для у.е. кампаний отдает статистику в рублях, а для всех остальных -- в валюте кампании.
	        # Клиент в метрике стоимость цели указывает без НДС, поэтому дополнительно убирать НДС для перевода не нужно.
	        if ($order_currency->{$line->{OrderID}} eq 'YND_FIXED') {
	        	$line->{PriceCur} = convert_currency($line->{PriceCur}, 'RUB', $order_currency->{$line->{OrderID}}, with_nds => 1);
	    	}

			my $key = join '_', $line->{UpdateTime}, $line->{TargetType};
			$lines_by_date_and_targettype{join('_', $line->{UpdateTime}, $line->{TargetType})}->{$line->{OrderID}} = $line->{PriceCur};
		}

		foreach my $key (keys %lines_by_date_and_targettype) {
			my ($stat_date, $target_type) = split /_/, $key;
			for my $orders_chunk (chunks([keys %{$lines_by_date_and_targettype{$key}}], $UPDATE_LIMIT)) {
				my $sql_case_by_orders = sql_case(OrderID => hash_cut($lines_by_date_and_targettype{$key}, $orders_chunk), default__dont_quote => 'goals_income');
				relaxed times => $SLEEP, sub {
					do_sql(PPC(shard => $shard), ["update bs_order_target_stat set goals_income = ", $sql_case_by_orders,
													where => {stat_date => $stat_date,
															  target_type => $target_type,
															  OrderID => $orders_chunk,
															  goals_income__is_null => 1}]);
				};

				$updated += scalar(@$orders_chunk);
				$updated_after_log += scalar(@$orders_chunk);
				if ($updated_after_log >= $UPDATE_LOG_LIMIT) {
					$log->out("locally updated $updated rows out of " . scalar(@$lines) . " (process_lines_from_file)");
					$updated_after_log = 0;
				}
			}
		}
	};
	$log->out("locally updated $updated rows out of " . scalar(@$lines) . " (process_lines_from_file)");
}
