#!/usr/bin/perl

use my_inc '..';

=head1 DEPLOY

# approved by zhur
# .migr
[
  {
    type => 'sql',
    webstop => "0",
    db => "ppcdict",
    when => 'before',
    time_estimate => "10 секунд",
    sql => "
CREATE TABLE `inc_hierarchical_multiplier_id` (
  `hierarchical_multiplier_id` bigint(20) unsigned NOT NULL AUTO_INCREMENT,
  PRIMARY KEY (`hierarchical_multiplier_id`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8
"
  },
  {
    type => 'sql',
    webstop => "0",
    db => "ppc:all",
    when => 'before',
    time_estimate => "10 секунд",
    sql => [
    "
CREATE TABLE `hierarchical_multipliers` (
  `hierarchical_multiplier_id` bigint(20) unsigned NOT NULL,
  `syntetic_key_hash` bigint(20) unsigned NOT NULL,
  `last_change` datetime NOT NULL DEFAULT CURRENT_TIMESTAMP,
  `cid` int(10) unsigned NOT NULL REFERENCES `campaigns`(`cid`),
  `pid` int(10) unsigned DEFAULT NULL REFERENCES `phrases`(`pid`),
  `type` enum('mobile_multiplier','demography_multiplier','retargeting_multiplier','distance_multiplier') NOT NULL,
  `multiplier_pct` smallint unsigned,
  `is_enabled` tinyint(4) unsigned NOT NULL DEFAULT '1',
  PRIMARY KEY (`hierarchical_multiplier_id`),
  UNIQUE KEY `cid_hash` (`cid`, `syntetic_key_hash`),
  UNIQUE KEY `cid` (`cid`,`pid`,`type`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8
    ",
    "
CREATE TABLE `demography_multiplier_values` (
  `demography_multiplier_value_id` bigint(20) unsigned NOT NULL,
  `last_change` datetime NOT NULL DEFAULT CURRENT_TIMESTAMP,
  `hierarchical_multiplier_id` bigint(20) unsigned NOT NULL REFERENCES hierarchical_multipliers(hierarchical_multiplier_id),
  `gender` enum('male','female') DEFAULT NULL,
  `age` enum('0-17','18-24','25-34','35-44','45-') DEFAULT NULL,
  `multiplier_pct` smallint unsigned NOT NULL,
  PRIMARY KEY (`demography_multiplier_value_id`),
  KEY `hierarchical_multiplier_id` (`hierarchical_multiplier_id`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8
    ",
    "
CREATE TABLE `retargeting_multiplier_values` (
  `retargeting_multiplier_value_id` bigint(20) unsigned NOT NULL,
  `last_change` datetime NOT NULL DEFAULT CURRENT_TIMESTAMP,
  `hierarchical_multiplier_id` bigint(20) unsigned NOT NULL REFERENCES hierarchical_multipliers(hierarchical_multiplier_id),
  `ret_cond_id` int(10) unsigned NOT NULL REFERENCES retargeting_conditions(ret_cond_id),
  `multiplier_pct` smallint unsigned NOT NULL,
  PRIMARY KEY (`retargeting_multiplier_value_id`),
  UNIQUE KEY `hierarchical_multiplier_id` (`hierarchical_multiplier_id`,`ret_cond_id`),
  KEY `ret_cond_id` (`ret_cond_id`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8
    "
    ]
  },
  {
    type => 'script',
    when => 'after',
    time_estimate => "8 часов на devtest",
    comment => "Если после запуска в логах появилось 'Error in shards', то нужно запустить повторно."
  }
]

=head1 OPTIONS

    --shard-id X - работать только с указанным шардом, опцию можно указывать несколько раз
    --only-groups - копировать данные только из group_params
    --only-camps - копировать данные только из camp_options
    --sleep-coef 2 - спать во столько раз больше времени после обработки каждого батча
    --report - ничего не делать, вывести отчёт о качестве уже проведённой миграции

=cut

use Direct::Modern;
use Yandex::DBTools;
use Yandex::DBShards;
use Yandex::Retry;
use Yandex::Overshard;

use Settings;
use ScriptHelper;
use HierarchicalMultipliers qw/mass_validate_and_maybe_add/;
use Try::Tiny;
use ShardingTools qw/foreach_shard_parallel_verbose ppc_shards/;
use PrimitivesIds;

my $ONLY_GROUPS = 0;
my $ONLY_CAMPS = 0;
my $SLEEP_COEF = 2;
my @SHARDS;
my $REPORT_ONLY;
extract_script_params(
    'only-groups' => \$ONLY_GROUPS,
    'only-camps' => \$ONLY_CAMPS,
    'sleep-coef=i' => \$SLEEP_COEF,
    'shard-id=i' => \@SHARDS,
    'report' => \$REPORT_ONLY,
);

if ($REPORT_ONLY) {
    for my $shard (ppc_shards()) {
        say "================================================================================";
        say "Report for shard $shard";

        my $camp_with_multipliers_count = get_one_field_sql(PPC(shard => $shard), "select count(*) from camp_options where mobile_multiplier_pct");
        say "There is $camp_with_multipliers_count campaigns which have mobile multiplier set";
        my $camp_delta = get_all_sql(PPC(shard => $shard), [
            "select abs(cast(ifnull(co.mobile_multiplier_pct, 0) as signed integer) - cast(h.multiplier_pct as signed integer)) as delta, count(*) as cnt",
            "from hierarchical_multipliers h left join camp_options co using (cid) ",
            "where h.pid is null group by abs(cast(ifnull(co.mobile_multiplier_pct, 0) as signed integer) - cast(h.multiplier_pct as signed integer))",
        ]);
        say "Hierarchical multipliers records - delta with campaigns:";
        say "      $_->{delta} = $_->{cnt}" for @$camp_delta;
        say "";

        my $group_with_multipliers_count = get_one_field_sql(PPC(shard => $shard), "select count(*) from group_params where mobile_multiplier_pct");
        say "There is $group_with_multipliers_count groups which have mobile multiplier set";
        my $group_delta = get_all_sql(PPC(shard => $shard), [
            "select abs(cast(ifnull(gp.mobile_multiplier_pct, 0) as signed integer) - cast(h.multiplier_pct as signed integer)) as delta, count(*) as cnt",
            "from hierarchical_multipliers h left join group_params gp using (pid) ",
            "where h.pid is not null group by abs(cast(ifnull(gp.mobile_multiplier_pct, 0) as signed integer) - cast(h.multiplier_pct as signed integer))",
        ]);
        say "Hierarchical multipliers records - delta with groups:";
        say "      $_->{delta} = $_->{cnt}" for @$group_delta;
    }
    exit(0);
}

$log->out('START');

@SHARDS = ppc_shards() if !@SHARDS;
my %ONLY_SHARDS = map { $_ => 1 } @SHARDS;

my $DO_GROUPS = 1;
my $DO_CAMPS = 1;
if ($ONLY_CAMPS) {
    ($DO_CAMPS, $DO_GROUPS) = (1, 0);
} elsif ($ONLY_GROUPS) {
    ($DO_CAMPS, $DO_GROUPS) = (0, 1);
}

sub save_batch {
    my ($by_clientid) = @_;
    for my $ClientID (keys %$by_clientid) {
        my $batch = $by_clientid->{$ClientID};
        mass_validate_and_maybe_add($ClientID, $batch);

        my $success_count = grep { exists $_->{value}{id} } @$batch;

        my %errors;
        for my $err (grep { $_ } map { $_->{value}{error} } @$batch) {
            $errors{$err->name}++;
        }

        my $error_msg = join(", ", map { "$_=$errors{$_}" } keys %errors);
        $log->out("Batch for client $ClientID: copied $success_count, errors: $error_msg");
    }
}

foreach_shard_parallel_verbose(
    $log, sub {
        my ($shard) = @_;
        return unless $ONLY_SHARDS{$shard};
        $log->out('Doing groups in shard $shard');
        my ($min, $max) = get_one_line_array_sql(PPC(shard => $shard), "select min(pid), max(pid) from group_params");
        while ($min <= $max) {
            my $relax = relaxed_guard times => $SLEEP_COEF;
            my $multipliers = get_all_sql(
                PPC(shard => $shard), [
                    "select pid, mobile_multiplier_pct from group_params",
                    "where mobile_multiplier_pct AND ",
                    {pid__ge => $min},
                    "order by pid limit 1000",
                ]
            );
            my $pid_to_cid = get_hash_sql(PPC(shard => $shard), ["select pid, cid from phrases", where => {pid => [map { $_->{pid} } @$multipliers]}]);
            $_->{cid} = $pid_to_cid->{$_->{pid}} for @$multipliers;
            my $camp_to_clientid = get_key2clientid(cid => [map { $_->{cid} } @$multipliers ]);

            my %by_clientid;
            for my $mult (@$multipliers) {
                push @{$by_clientid{$camp_to_clientid->{$mult->{cid}}}}, {
                    cid => $mult->{cid},
                    pid => $mult->{pid},
                    type => "mobile_multiplier",
                    value => {
                        multiplier_pct => $mult->{mobile_multiplier_pct},
                    },
                };
            }

            $log->out("got @{[scalar @$multipliers]} starting at $min");
            save_batch(\%by_clientid);
            if (@$multipliers) {
                $min = $multipliers->[-1]{pid} + 1;
            } else {
                last;
            }
        }
        $log->out('Done with groups in shard $shard');
    }
) if $DO_GROUPS;
$log->out("Past DO_GROUPS");

foreach_shard_parallel_verbose(
    $log, sub {
        my ($shard) = @_;
        return unless $ONLY_SHARDS{$shard};
        $log->out('Doing camps in shard $shard');
        my $relax = relaxed_guard times => $SLEEP_COEF;

        my ($min, $max) = get_one_line_array_sql(PPC(shard => $shard), "select min(cid), max(cid) from camp_options");
        while ($min <= $max) {
            my $multipliers = get_all_sql(
                PPC(shard => $shard), [
                    "select cid, null as pid, mobile_multiplier_pct from camp_options",
                    "where mobile_multiplier_pct AND ",
                    { cid__ge => $min },
                    "order by cid limit 1000",
                ]);
            my $camp_to_clientid = get_key2clientid(cid => [map { $_->{cid} } @$multipliers ]);

            my %by_clientid;
            for my $mult (@$multipliers) {
                push @{$by_clientid{$camp_to_clientid->{$mult->{cid}}}}, {
                    cid => $mult->{cid},
                    type => "mobile_multiplier",
                    value => {
                        multiplier_pct => $mult->{mobile_multiplier_pct},
                    },
                };
            }

            $log->out("got @{[scalar @$multipliers]} starting at $min");
            save_batch(\%by_clientid);

            if (@$multipliers) {
                $min = $multipliers->[-1]{cid} + 1;
            } else {
                last;
            }
        }
        $log->out('Done with camps in shard $shard');
    }
) if $DO_CAMPS;
$log->out("Past DO_CAMPS");


$log->out('FINISH');

