#!/usr/bin/perl

use my_inc '..';

=head1 DEPLOY

# approved by pankovpv
# .migr
{
  type => 'script',
  when => 'after',
  time_estimate => '~ 4 часов',
  comment => '
        деплой можно перезапускать
        ./deploy/20150825_set_camp_enable_cpc_hold.pl
        
        для запуска на определенных шардах  
        ./deploy/20150825_set_camp_enable_cpc_hold.pl --shard-id=1 --shard-id=4
        
        для уменьшения частоты update в БД
        ./deploy/20150825_set_camp_enable_cpc_hold.pl --sleep-coef=10
        
        для применения деплоя по заданным кампаниям
        ./deploy/20150825_set_camp_enable_cpc_hold.pl --order-ids-file=path_to_file
        
        path_to_file - текстовый файл, на каждой строке OrderID
        <OrderID>
        <OrderID>
        ....
        <OrderID>
        <OrderID> 
  ',
}

=cut


use Direct::Modern;
use feature "state";

use Settings;
use Yandex::DBTools;
use Yandex::DBShards;
use ScriptHelper;
use ShardingTools qw/foreach_shard_parallel_verbose ppc_shards/;
use BS::ResyncQueue qw/bs_resync/;
use Yandex::Retry qw/relaxed_guard/;
use List::MoreUtils qw/uniq/;

my $LIMIT = 100_000;
my $SLEEP_COEF = 2;
my $ORDER_IDS_FILE;
my @SHARDS;
extract_script_params(
    'sleep-coef=i' => \$SLEEP_COEF,
    'shard-id=i' => \@SHARDS,
    'order-ids-file=s' => \$ORDER_IDS_FILE 
);

@SHARDS = ppc_shards() if !@SHARDS;
my %ONLY_SHARDS = map { $_ => 1 } @SHARDS;

$log->out('START');

my $ORDER_IDS;
if ($ORDER_IDS_FILE) {
    $log->out("read order ids from $ORDER_IDS_FILE");
    $log->die("can't find file $ORDER_IDS_FILE") unless -e $ORDER_IDS_FILE;
    $ORDER_IDS = [];
    open IDS, $ORDER_IDS_FILE or $log->die($!);
    while (defined($_ = <IDS>)) {
        chomp $_;
        push @$ORDER_IDS, $_ if /^[0-9]+$/;   
    } 
    close IDS;
    $log->out(sprintf "read %d orders from $ORDER_IDS_FILE", scalar @$ORDER_IDS);
}

foreach_shard_parallel_verbose($log, sub {
    my $shard = shift;
    return unless $ONLY_SHARDS{$shard};

    open(F, '>>', "$Settings::LOG_ROOT/set_camp_enable_cpc_hold_resync_shard_$shard.tsv") or $log->die($!);
    my $camps = [];
    while ($camps) {
        {
            my $relax = relaxed_guard times => $SLEEP_COEF;
            
            $camps = get_next_camps($shard, $ORDER_IDS);
            if ($camps && @$camps) {
                my $cids = [map { $_->{cid} } @$camps];
                $log->out(sprintf "try updating cids %s", join ',', @$cids);
                my $affected = set_enable_cpc_hold($shard, $cids) + 0;
                if ($affected > 0) {
                    $log->out(sprintf "flag enable_cpc_hold set on %d camps", $affected);
                    for my $camp (@$camps) {
                        if ($camp->{resync}) {
                            print F join("\t", $camp->{cid}, 0, 0, ($camp->{statusActive} eq 'Yes' ? 94 : 80)), "\n";
                        }
                    }
                }
            }
        }
    }
    close(F) or $log->die($!);
});

$log->out('FINISH');

sub set_enable_cpc_hold {
    my ($shard, $cids) = @_;
    return do_sql(PPC(shard => $shard),
        ["UPDATE campaigns c JOIN camp_options co USING(cid)
            SET c.opts = IF(opts = '', 'enable_cpc_hold', CONCAT(opts, ',enable_cpc_hold'))
            WHERE FIND_IN_SET('enable_cpc_hold', c.opts) = 0 AND ",
                {'c.cid' => $cids}]);
}

sub get_next_camps {
    my ($shard, $order_ids) = @_;
    
    state $min_cid;
    state $max_cid;
    
    my %where;
    if ($order_ids) {
        unless (defined $min_cid) {
            $min_cid = 0;
            $max_cid = $min_cid + 1;    
        } 
        $where{'c.OrderID'} = $order_ids;
    } else {
        unless (defined $min_cid) {
            ($min_cid, $max_cid) = get_one_line_array_sql(PPC(shard => $shard), "SELECT MIN(cid), MAX(cid) FROM campaigns");
        }
        $where{'c.cid__between'} = [$min_cid, $min_cid + $LIMIT];
    }
    
    return undef if $max_cid <= $min_cid;
    
    my $camps = get_all_sql(PPC(shard => $shard),
        ["SELECT c.cid,
                (c.OrderID > 0 AND c.archived = 'No' AND IFNULL(co.strategy, '') != 'different_places' AND c.autobudget='No') AS resync,
                c.statusActive
            FROM campaigns c JOIN camp_options co USING(cid)
            WHERE c.type IN ('text','geo','mobile_content','performance','dynamic')
                AND FIND_IN_SET('enable_cpc_hold', c.opts) = 0 AND ", \%where]);
    $min_cid += $LIMIT + 1;
    return $camps;
}
