#!/usr/bin/perl

use my_inc '..';

=head1 DEPLOY

# approved by pankovpv
# .migr
{
  type => 'script',
  when => 'after',
  time_estimate => 'порядка 4 часов',
  comment => 'Дообъединяем и доконвертируем объединённых в Балансе клиентов. Скрипт можно перезапускать.',
}

=cut

use Direct::Modern;

use List::Util qw(first);

use Yandex::DBTools;
use Yandex::Balance qw(balance_get_all_equal_clients);
use Yandex::DBShards;
use Yandex::Log;
use Yandex::HashUtils;
use Yandex::TimeCommon;
use Yandex::Balance qw(balance_get_completion_history balance_get_orders_info);
use Yandex::ListUtils qw(xsort chunks);
use Yandex::HashUtils;

use Settings;
use ScriptHelper;
use RBACElementary;
use RBAC2::Extended;
use RBACDirect;
use Common;
use Campaign;
use Client;
use Client::NDSDiscountSchedule;
use Client::CurrencyTeaserData;
use Direct::Gearman::Currency::ConvertToRealMoney;
use CampAutoPrice::Common;
use Currencies;
use Currency::Rate;
use DBLog;
use Campaign::Types qw(get_camp_kind_types);
use geo_regions;

my $RBAC_OPERATOR_UID = 1;
my $MAX_RESHARD_ITERATIONS = 60;

$log->out('START');

$log->out('Fetching equal clients from Balance');
my $merged_clients = balance_get_all_equal_clients();
$log->out('Got ' . scalar(@$merged_clients) . ' merged client pairs');

my %clid2primary = map { $_->{CLIENT_ID} => $_->{CLASS_ID} } grep { $_->{CLIENT_ID} != $_->{CLASS_ID} } @$merged_clients;
foreach_shard ClientID => [keys %clid2primary], chunk_size => 1_000, sub {
    my ($shard, $clientids) = @_;

    # данные должны быть только на главном ClientID
    # с неглавных надо всё перевешивать
    $log->out('Fetching Direct data for non-primary clients:', $clientids);
    do_sql(PPC(shard => $shard), 'SET GROUP_CONCAT_MAX_LEN=999999');
    my $non_primary_clients = get_all_sql(PPC(shard => $shard), ['
        SELECT u.ClientID, GROUP_CONCAT(u.login) AS logins, GROUP_CONCAT(u.uid) AS uids
        FROM users u
     ', WHERE => {
            'u.ClientID' => $clientids,
        }, '
        GROUP BY u.ClientID
    ']);
    $log->out('Got ' . scalar(@$non_primary_clients) . ' non-primary clients');

    my $rbac = RBAC2::Extended->get_singleton( $RBAC_OPERATOR_UID );

    $log->out('Getting agencies info');
    my @clids = (keys(%clid2primary), values(%clid2primary));
    my $clid2chief_rep = rbac_get_chief_reps_of_agencies(\@clids);

    foreach my $non_primary_client (@$non_primary_clients) {
        my $clid = $non_primary_client->{ClientID};
        my $logins = $non_primary_client->{logins};
        my @uids = split /,/, $non_primary_client->{uids} // '';
        my $primary_clientid = $clid2primary{$clid};

        if (defined $clid2chief_rep->{$clid}) {
            $log->out("ClientID $clid is agency, skipping");
            next;
        } elsif (defined $clid2chief_rep->{$primary_clientid}) {
            $log->out("primary ClientID $primary_clientid is agency, skipping");
            next;
        }

        $log->out("Converting ClientIDs $clid ($logins) and $primary_clientid into currency if needed");
        _maybe_convert_clients($clid, $primary_clientid);

        # на всякий случай =)
        $rbac = RBAC2::Extended->get_singleton( $RBAC_OPERATOR_UID );

        $log->out("Merging ClientID $clid ($logins) with [primary] ClientID $primary_clientid");
        eval { _merge_clients($rbac, $clid, $primary_clientid, \@uids) };
        if ($@) {
            $log->out("ERROR merging $clid ($logins) => $primary_clientid: $@");
        }
    }
};

$log->out('FINISH');


=head3 _maybe_convert_clients

=cut

sub _maybe_convert_clients {
    my ($secondary_clientid, $primary_clientid) = @_;

    my $convert_msg_prefix = "[convert $secondary_clientid => $primary_clientid]";
    my $msg_prefix_guard = $log->msg_prefix_guard($convert_msg_prefix);

    my @clids = ($secondary_clientid, $primary_clientid);
    my $convert_data = get_all_sql(PPC(ClientID => \@clids), ['
        SELECT *
        FROM currency_convert_queue
     ', WHERE => {
            ClientID => SHARD_IDS,
        },
    ]);

    if (!@$convert_data) {
        $log->out('Both clients have not been converted yet, skipping');
        return;
    }

    my ($currency, $clid_to_convert, $start_convert_at);
    for my $convert (@$convert_data) {
        my $clid = $convert->{ClientID};
        my $state = $convert->{state};

        if($state eq 'NEW') {
            # удаляем необработанную заявку на конвертацию, чтобы не мешалась под ногами
            $log->out("Deleting uncompleted currency convert request for ClientID $clid:", $convert);
            do_delete_from_table(PPC(ClientID => $clid), 'currency_convert_queue', where => {ClientID => $clid});
            do_delete_from_table(PPC(ClientID => $clid), 'client_currency_changes', where => {ClientID => $clid});
        } elsif ($state eq 'DONE') {
            $currency = $convert->{new_currency};
            $start_convert_at = $convert->{start_convert_at};
            $clid_to_convert = first { $_ != $clid} @clids;

            if ($convert->{convert_type} ne 'MODIFY') {
                $log->die("COPY convert type is not supported");
            }
        }
    }

    if (!$currency || !$clid_to_convert) {
        $log->out('No completed convertation found, skipping');
        return;
    }

    $log->out("Checking current ClientID $clid_to_convert currency");
    my $client_currencies = get_client_currencies($clid_to_convert);
    my $cur_currency = $client_currencies->{work_currency};

    if ($cur_currency eq $currency) {
        $log->out("Currencies already match, skipping");
        return;
    } elsif ($cur_currency ne 'YND_FIXED') {
        $log->die("Currencies do not match, but current ClientID $clid_to_convert currency is $cur_currency which cannot be converted");
    }

    $log->out("Fetching campaigns to convert for ClientIDs " . join(', ', @clids));
    my $campaigns = get_all_sql(PPC(ClientID => \@clids), ['
        SELECT c.cid, u.ClientID
        FROM users u
        INNER JOIN campaigns c ON u.uid = c.uid
    ',  WHERE => {
        'u.ClientID' => SHARD_IDS,
        'c.currencyConverted' => 'No',
        'c.type' => get_camp_kind_types('currency_convert'),
        _TEXT => 'IFNULL(c.currency, "YND_FIXED") = "YND_FIXED"',
    }]);
    $log->out('Got ' . scalar(@$campaigns) . ' campaigns to fix:', $campaigns);

    for my $camp (@$campaigns) {
        my $cid = $camp->{cid};
        $camp->{new_currency} = $currency;
        $camp->{uid} = 1;
        $camp->{start_convert_at} = $start_convert_at;
        $camp->{convert_type} = 'MODIFY';
        $camp->{country_region_id} = $geo_regions::RUS;
        my $cid_msg_prefix_guard = $log->msg_prefix_guard("$convert_msg_prefix\t[cid_$cid]");
        $log->out('Converting campaign');
        _convert_campaign_inplace($camp);
    }
}

=head3 _convert_client_inplace

    копипаста из Direct::Gearman::Currency::ConvertToRealMoney::convert_client_inplace
    с изменённым началом

=cut

sub _convert_campaign_inplace {
    my ($camp) = @_;

    my $cid = $camp->{cid};
    my $client_id = $camp->{ClientID};
    my $data = hash_cut $camp, qw(ClientID uid convert_type state new_currency country_region_id email start_convert_at);
    my $currency = $data->{new_currency};

    my $client_msg_prefix = "[convert_client_inplace] [ClientID $client_id]";
    my $msg_prefix_guard = $log->msg_prefix_guard($client_msg_prefix);

    # дальше сплошная копипаста из Direct::Gearman::Currency::ConvertToRealMoney::convert_client_inplace

    my $rbac = Direct::Gearman::Currency::ConvertToRealMoney::_get_rbac_object();
    my $client_chief_uid = rbac_get_chief_rep_of_client($client_id);
    my $cond = {'c.uid' => $client_chief_uid,
                'c.type' => get_camp_kind_types('currency_convert'),
                'c.currencyConverted' => "No",
                _TEXT => 'IFNULL(c.currency, "YND_FIXED") = "YND_FIXED"',
                'c.cid' => $cid, # не копипаста, добавлено в миграции
    };

    my $campaigns_data = Common::get_user_camps_by_sql($cond, {shard => {ClientID => $client_id}, include_empty_campaigns => 1});
    my $campaigns = $campaigns_data->{campaigns};
    $log->out('Got ' . scalar(@$campaigns) . ' campaigns from DB to convert');

    my @cids = map {$_->{cid}} @$campaigns;
    clear_auto_price_queue(\@cids, error_str => 'currency convert');

    my $rate = convert_currency(1, 'YND_FIXED', $currency);
    my $rate_nds = convert_currency(1, 'YND_FIXED', $currency, with_nds => 1);

    my $migrate_datetime = Direct::Gearman::Currency::ConvertToRealMoney::_get_migrate_date($data->{start_convert_at}, $data->{convert_type});
    my $first_date_in_real_currency = mysql_round_day($migrate_datetime);

    $log->out('Fetching chips cost/spent from Balance to update in DB');
    for my $cids_chunk (chunks \@cids, 10) {
        my $cids_chunk_str = join(',', @$cids_chunk);
        $log->out("Fetching chips cost/spent for cids chunk: $cids_chunk_str");
        my $balance_info_chunk = balance_get_orders_info($cids_chunk);
        my @for_insert;
        for my $order (@$balance_info_chunk) {
            $log->out($order);
            my $cid = $order->{ServiceOrderID};
            my $chips_cost = $order->{CompletionFixedMoneyQty} // $order->{CompletionMoneyQty};
            my $chips_spent = $order->{CompletionFixedQty} // $order->{completion_qty} // 0;
            push @for_insert, [$cid, $chips_cost, $chips_spent];
        }
        $log->out("Updating chips cost/spent in DB for cids chunk $cids_chunk_str");
        do_mass_insert_sql(PPC(ClientID => $client_id), '
            INSERT INTO campaigns_multicurrency_sums
                (cid, chips_cost, chips_spent) VALUES %s
            ON DUPLICATE KEY UPDATE
                chips_cost = VALUES(chips_cost),
                chips_spent = VALUES(chips_spent)
        ', \@for_insert);
    }

    $log->out('Begin converting campaigns');

    my $campaigns_multicurrency_sums_data = get_hashes_hash_sql(PPC(ClientID => $client_id), ['SELECT cid, sum, chips_cost FROM campaigns_multicurrency_sums', WHERE => {cid => \@cids}]);
    for my $campaign (@$campaigns) {
        my $cid = $campaign->{cid};
        $log->msg_prefix("$client_msg_prefix [cid $cid]");

        my $converted_campaign_params = Client::ConvertToRealMoney::get_converted_campaign_params($campaign, $currency, $rate, keep_ProductID => 1, copy_last_bill_sums => 1, save_conversion_strategy => 1);

        # берём сохранённые суммы для кампании в валюте, которые получаем в нотификациях от Баланса
        my $campaign_multicurrency_sums_data = $campaigns_multicurrency_sums_data->{$cid};
        my $new_sum_spent = ($campaign_multicurrency_sums_data) ? $campaign_multicurrency_sums_data->{chips_cost} // 0 : 0;
        my $new_sum = ($campaign_multicurrency_sums_data) ? $campaign_multicurrency_sums_data->{sum} // 0 : 0;

        $converted_campaign_params->{campaigns}->{sum} = $new_sum;
        $converted_campaign_params->{campaigns}->{sum_spent} = $new_sum_spent;
        $converted_campaign_params->{campaigns}->{balance_tid} = 0;
        $log->out('Converted params:', $converted_campaign_params);

        $log->out("Logging new campaign sum $new_sum");
        DBLog::log_balance($data->{uid}, $cid, $new_sum, 0, 0, $campaign->{mediaType}, $campaign->{currency});

        $log->out('Converting campaign');

        do_in_transaction {
            # лочим валюту в campaigns. в других местах можно также выбрать кампанию с FOR UPDATE и добиться последовательного выполнения транзакции на построчном локе.
            my $campaign_currency = get_one_field_sql(PPC(ClientID => $client_id), ['SELECT currency FROM campaigns', WHERE => {cid => $cid}, 'FOR UPDATE']);
            $log->die("Campaign $cid is already in $campaign_currency. That's VERY strange =)") if $campaign_currency && $campaign_currency ne 'YND_FIXED';

            Client::ConvertToRealMoney::convert_campaign_bids([$cid], 'YND_FIXED', $currency);
            Client::ConvertToRealMoney::convert_campaign_secondary_options([$cid], 'YND_FIXED', $currency);
            Client::ConvertToRealMoney::convert_campaign_payments_info([$cid], 'YND_FIXED', $currency);

            if ($converted_campaign_params->{camp_options} && %{$converted_campaign_params->{camp_options}}) {
                do_update_table(PPC(ClientID => $client_id), 'camp_options', $converted_campaign_params->{camp_options}, where => {cid => $cid});
            }
            do_update_table(PPC(ClientID => $client_id), 'campaigns', {
                currencyConverted => 'Yes',
                # прогноз сконвертировали по курсу, но ставки могли вырасти (больше минимальная ставка), да и шаг торгов другой — надёжнее пересчитать
                autobudgetForecastDate => undef,
                statusBsSynced => 'No',
                %{$converted_campaign_params->{campaigns}}
            }, where => {cid => $cid});
            # записываем суммы до конвертации и после, чтобы показывать их на странице успеха
            my $money_before = $campaign->{total};
            my $money_after = $new_sum - $new_sum_spent;
            $log->out("Writing correspondence data for campaign $cid: money_before = $money_before, money_after = $money_after");
            do_insert_into_table(PPC(ClientID => $client_id), 'currency_convert_money_correspondence', {
                ClientID => $client_id,
                old_cid => $cid,
                new_cid => $cid,
                money_before => $money_before,
                money_after => $money_after,
                was_archived => ($campaign->{archived} && $campaign->{archived} eq 'Yes') ? 1 : 0,
            }, on_duplicate_key_update => 1, key => 'ClientID');

            # multicurrency: monitor.agency_clients_month_stat
            # multicurrency: ppc.stopped_camp_users
            # multicurrency: конвертировать данные в ppc.serviced_client_budget

            # конвертируем грубый прогноз приблизительно. через небольшое время он пересчитается в ppcCampSpentForecast.pl
            do_update_table(PPC(ClientID => $client_id), 'camp_rough_forecast', {'forecast_sum__dont_quote' => "forecast_sum*$rate", forecast_time__dont_quote => 'forecast_time'}, where => {cid => $cid});
        };

        if ($campaign->{OrderID} && $campaign->{OrderID} > 0) {
            $log->out("Converting non-detailed stat (OrderID = $campaign->{OrderID}) with rate = $rate_nds");
            Client::ConvertToRealMoney::convert_nondetailed_stat($campaign->{OrderID}, $rate_nds);
        }

        $log->out('Done converting campaign');
    }
    $log->msg_prefix($client_msg_prefix);
    $log->out('Done converting campaigns');

    $log->out('fixating new client currency');
    Client::ConvertToRealMoney::fixate_client_currency($client_id, $currency, $data->{country_region_id});
}

=head3 _merge_clients

=cut

sub _merge_clients {
    my ($rbac, $old_clientid, $new_clientid, $old_uids) = @_;

    my $main_client_id = $new_clientid;
    my $second_client_id = $old_clientid;

    my $msg_prefix_guard = $log->msg_prefix_guard("[merge $old_clientid => $new_clientid]");

    ############# Свозим клиентов в один шард
    _reshard_clients($second_client_id, $main_client_id);

    my $chief_rep_uid = rbac_get_chief_rep_of_client($main_client_id);

    do_in_transaction {
        my $errcode;

        my $tmp_rep_uid;
        if (!$chief_rep_uid) {
            # мы ничего не знаем про главный ClientID в Балансе, создаём его с временным представителем
            $chief_rep_uid = rbac_get_chief_rep_of_client($second_client_id);

            if (!$chief_rep_uid) {
                $log->die("No chief uid for ClientID $second_client_id known, skipping");
            }

            $log->out("Main ClientID $main_client_id has no chief rep, using chief uid $chief_rep_uid from non-primary ClientID $second_client_id");

            $tmp_rep_uid = 225023846; # более-менее рандомный uid, на котором ничего нет

            $log->out("Creating ClientID $main_client_id with main rep uid $tmp_rep_uid");
            $errcode = $rbac->Handle('RBAC_CreateClient', {
                  uid       => $tmp_rep_uid,
                  client_id => $main_client_id,
            });
            $log->die("RBAC_CreateClient(uid => $tmp_rep_uid, client_id => $main_client_id) failed: $errcode") if $errcode;
        }

        $log->out('merging clients in RBAC');
        $errcode = rbac_merge_clients($rbac, $main_client_id, $second_client_id, 1);
        $log->die("rbac_merge_clients(\$rbac, $main_client_id, $second_client_id) failed: $errcode") if $errcode;

        if ($tmp_rep_uid) {
            $log->out("Making chief uid $chief_rep_uid");
            $errcode = $rbac->Handle("RBAC_SwitchClientChiefRep", {
                client_id => $main_client_id,
                rep_uid => $chief_rep_uid,
            });
            $log->die("rbac_switch_client_chief(\$rbac, $chief_rep_uid, 1) failed: $errcode") if $errcode;

            $errcode = rbac_drop_client_rep($rbac, $main_client_id, $tmp_rep_uid, 1);
            $log->die("rbac_drop_client_rep(\$rbac, $main_client_id, $tmp_rep_uid, 1) failed: $errcode") if $errcode;
        }

        $log->out('moving direct data [set_main_client_id]');
        my @CLIENT_IDS = ($main_client_id, $second_client_id);
        _set_main_client_id($rbac, $main_client_id, \@CLIENT_IDS);
        foreach my $second_rep_uid (@$old_uids) {
            $log->out("moving direct data [change_uid]: uid $second_rep_uid => uid $chief_rep_uid");
            Common::change_uid($second_rep_uid, $chief_rep_uid);
        }

        $log->out('Commiting to RBAC');
        $rbac->Commit;
    };
}

=head3 _reshard_clients

    Привозит неглавного клиента в один шард с главным или убеждается, что это и так так

    _reshard_clients($old_clientid, $new_clientid);

=cut

sub _reshard_clients {
    my ($second_client_id, $main_client_id) = @_;

    my $main_client_shard = get_shard(ClientID => $main_client_id);
    my $second_client_shard = get_shard(ClientID => $second_client_id);

    if (!$main_client_shard) {
        $log->out("Saving shard $second_client_shard for unexisting ClientID $main_client_id");
        save_shard(ClientID => $main_client_id, shard => $second_client_shard);
        $main_client_shard = $second_client_shard;
    }

    $log->die("No shard for ClientID $main_client_id known") unless $main_client_shard;
    $log->die("No shard for ClientID $second_client_id known") unless $second_client_shard;

    if ($main_client_shard != $second_client_shard) {
        $log->out("Main ClientID $main_client_id is in $main_client_shard shard, but non-primary ClientID $second_client_id lives is in $second_client_shard. Scheduling resharding.");
        do_insert_into_table(PPCDICT, 'reshard_queue', {
            ClientID => $second_client_id,
            status => 'new',
            old_shard => $second_client_shard,
            new_shard => $main_client_shard,
            wanted_start_time__dont_quote => 'NOW()',
        });

        $log->out("Waiting for $second_client_id resharding");
        my $is_reshard_done = 0;
        my $iters = 0;
        do {
            sleep 30;
            $is_reshard_done = get_one_field_sql(PPCDICT, ['
                SELECT 1
                FROM reshard_queue
             ', WHERE => {
                    ClientID => $second_client_id,
                    status => 'done',
                },
            ]);
            $iters++;
        } while (!$is_reshard_done && $iters < $MAX_RESHARD_ITERATIONS);

        if ($is_reshard_done) {
            $log->out("Resharing of $second_client_id successed");
        } else {
            $log->die("Resharing of $second_client_id failed");
        }
    } else {
        $log->out("ClientID $main_client_id and ClientID $second_client_id are in same shard, skipping resharding");
    }
}

=head3 _get_tables_to_merge

    Возвращает ссылку на хеш с таблицами, где надо переносить данные со старого ClientID на новый
    Формат:
        table_name => table_ClientID_field

=cut

sub _get_tables_to_merge {
    return {
        'account_score' => 'ClientID',
        'agency_currencies' => 'ClientID',
        'agency_client_relations' => 'client_client_id',
        'agency_total_sums' => 'ClientID',
        'api_special_user_options' => 'ClientID',
        'banner_images_pool' => 'ClientID',
        'banner_images_process_queue' => 'ClientID',
        'clients' => 'ClientID',
        'client_brands' => 'ClientID',
        'client_currency_changes' => 'ClientID',
        'client_domains' => 'ClientID',
        'client_domains_stripped' => 'ClientID',
        'client_limits' => 'ClientID',
        'clients_allow_wallet' => 'ClientID',
        'clients_api_options' => 'ClientID',
        'clients_autoban' => 'ClientID',
        'clients_custom_options' => 'ClientID',
        'clients_geo_cluster_stat' => 'ClientID',
        'clients_geo_ip' => 'ClientID',
        'clients_needed_personal_manager_teaser' => 'ClientID',
        'clients_needy_stat' => 'ClientID',
        'clients_stat' => 'ClientID',
        'clients_to_fetch_nds' => 'ClientID',
        'currency_convert_money_correspondence' => 'ClientID',
        'eventlog' => 'ClientID',
        'feeds' => 'ClientID',
        'force_currency_convert' => 'ClientID',
        'infoblock_teasers_factors' => 'ClientID',
        'mds_metadata' => 'ClientID',
        'mobile_content' => 'ClientID',
        'mod_documents' => 'ClientID',
        'minus_words' => 'ClientID',
        'perf_creatives' => 'ClientID',
        'retargeting_conditions' => 'ClientID',
        'serviced_client_amnesty' => 'ClientID',
        'serviced_client_budget' => 'ClientID',
        'serviced_client_history' => 'ClientID',
        'serviced_client_status' => 'ClientID',
        'sitelinks_sets' => 'ClientID',
        'users' => 'ClientID',
    };
}

=head3 _set_main_client_id

    Сильно изменённая копипаста из старого кода

=cut

sub _set_main_client_id
{
    my ($rbac, $main_client_id, $client_ids) = @_;

    $log->out("_set_main_client_id: no clientids given") if ref($client_ids) ne 'ARRAY' || ! @$client_ids;

    for my $client_id (@$client_ids) {
        next if $client_id == $main_client_id;

        ############# нужно проапдейтить адреса в vcards и удалить дубликаты из addresses
        $log->out('Moving addresses');
        _merge_clients_addresses($main_client_id, $client_id);

        ############# Перенос clients_options
        $log->out('Moving clients_options');
        my $is_main_client_exist = get_one_field_sql(PPC(ClientID => $main_client_id), "select 1 from clients_options where ClientID = ?", $main_client_id);
        my $is_client_exist = get_one_field_sql(PPC(ClientID => $client_id), "select 1 from clients_options where ClientID = ?", $client_id);

        if ($is_main_client_exist && $is_client_exist) {
            # удаляем старые данные
            do_sql(PPC(ClientID => $client_id), "delete from clients_options where ClientID = ?", $client_id);
        } elsif(!$is_main_client_exist && $is_client_exist) {
            my $client_data = get_one_line_sql(PPC(ClientID => $client_id), "
                select balance_tid, overdraft_lim, debt,
                       nextPayDate, statusBalanceBanned, warned_nextPayDate, warned_interval,
                       discount, budget, border_next, discount_next, border_prev
                from clients_options
                where ClientID = ?", $client_id
            );
            $client_data->{ClientID} = $main_client_id;

            do_insert_into_table(PPC(ClientID => $main_client_id), 'clients_options', $client_data);
            do_sql(PPC(ClientID => $client_id), "delete from clients_options where ClientID = ?", $client_id);
        }

        ############# Перенос данных таблиц по ClientID
        my $TABLES_TO_MERGE = _get_tables_to_merge();
        for my $table (sort keys %$TABLES_TO_MERGE) {
            my $field = $TABLES_TO_MERGE->{$table};
            $log->out("Merging data for table $table from ClientID $client_id to $main_client_id using field $field");
            _merge_table($client_id, $main_client_id, $table, $field);
        }

        ############# графики НДС и скидок
        $log->out("Refreshing NDS schedule for ClientID $main_client_id");
        Client::NDSDiscountSchedule::sync_nds_schedule_for_clients([$main_client_id], log => $log, rbac => $rbac);

        $log->out("Refreshing discount schedule for ClientID $main_client_id");
        Client::NDSDiscountSchedule::sync_discount_schedule_for_clients([$main_client_id], log => $log, rbac => $rbac);

        $log->out("Deleting NDS and discount schedule for ClientID $client_id");
        for my $table (qw(client_nds client_discounts)) {
            do_delete_from_table(PPC(ClientID => $client_id), $table, where => {ClientID => $client_id});
        }

        ############# Страны-валюты
        $log->out("Updating country-currencies for ClientID $main_client_id");
        my $agency_id = Primitives::get_client_first_agency($main_client_id);
        Client::CurrencyTeaserData::fetch_client_multicurrency_teaser_data($main_client_id, $agency_id, balance_timeout => 320);

        ############# Шардирование
        $log->out("Updating shard chain $client_id => $main_client_id");
        update_shard_chain('ClientID', $client_id, $main_client_id);

        Yandex::Log
          ->new(%Yandex::Balance::BALANCE_CALLS_LOG_SETTINGS, msg_prefix => 'set_main_client_id:')
          ->out("new ClientID: $main_client_id, old ClientID: $client_id");
    }

    return 1;
}

=head3 _merge_table

=cut

sub _merge_table {
    my ($client_id, $main_client_id, $table, $field) = @_;

    my $old_data = get_all_sql(PPC(ClientID => $main_client_id), ["SELECT * FROM $table", WHERE => {$field => $client_id}]);
    $log->out("Old data for $client_id for table $table:", $old_data);

    do_update_table(PPC(ClientID => $main_client_id), $table, {$field => $main_client_id}, where => {$field => $client_id}, ignore => 1);
    do_delete_from_table(PPC(ClientID => $main_client_id), $table, where => {$field => $client_id});
}

=head3 _merge_clients_addresses

=cut

sub _merge_clients_addresses {
    my ($main_client_id, $second_client_id) = @_;

    my $all_addresses_data = get_all_sql(PPC(ClientID => $second_client_id), [
        'SELECT aid, address, ClientID FROM addresses', WHERE => { ClientID => [$main_client_id, $second_client_id] }
    ]);
    my %all_addresses;
    for my $row (@$all_addresses_data) {
        $all_addresses{ $row->{address} }->{ $row->{ClientID} } = $row->{aid};
    }
    for my $address (keys %all_addresses) {
        if (scalar( keys %{$all_addresses{$address}} ) == 1 && exists $all_addresses{$address}->{$main_client_id}) {
            # только один адрес и тот на новый ClientID, пропускаем
            next;
        } elsif (scalar( keys %{$all_addresses{$address}} ) == 1 && exists $all_addresses{$address}->{$second_client_id}) {
            # только один адрес и тот на старый ClientID, апдейтим только addresses
            do_update_table(PPC(ClientID => $second_client_id), 'addresses', {ClientID => $main_client_id},
                            where => {ClientID => SHARD_IDS, aid=>$all_addresses{$address}->{$second_client_id}});
        } elsif (scalar( keys %{$all_addresses{$address}} ) == 2) {
            # на старом и новом клиенте есть 2 одинаковых адреса, для старого удаляем из addresses и апдейтим vcards
            do_delete_from_table(PPC(ClientID => $second_client_id), 'addresses',
                            where => {ClientID => SHARD_IDS, aid=>$all_addresses{$address}->{$second_client_id}});
            do_update_table(PPC(ClientID => $second_client_id), 'vcards', {address_id => $all_addresses{$address}->{$main_client_id}},
                                                         where => {address_id => $all_addresses{$address}->{$second_client_id}});
        }
    }
}
