#!/usr/bin/perl

=head1 DEPLOY

# approved by pankovpv
# .migr
[
    {
        type => 'script',
        when => 'after',
        time_estimate => "1 минута",
        comment => "Чиним кампании привязанные не к главному представителю",
    },
    {
        type => 'manual',
        when => 'after',
        time_estimate => "несколько минут",
        text => "После завершения предыдущей миграции, запустить /var/www/ppc.yandex.ru/protected/one-shot/20160303_MODIFY-convert-unconverted-campaigns.pl"  
    }
]


=cut

use my_inc '..';

use Direct::Modern;

use Settings;
use ScriptHelper;
use RBAC2::Extended;
use RBACElementary;
use RBACDirect;
use Client::NDSDiscountSchedule;
use Client::CurrencyTeaserData;
use BS::ResyncQueue;

use Yandex::DBTools;
use Yandex::DBShards;
use Yandex::HashUtils;
use Yandex::Balance;

use List::MoreUtils qw/uniq/;

my $FILE_WITH_CLIENT_IDS_TO_PROCESS = $Settings::ROOT . '/deploy/20160223_fix_camp_owners_clients.data';
my $MAX_RESHARD_ITERATIONS = 60;

$log->out('START');

# получем список клиентов, кампаниям которых нужно проставить единого владельца ("главного представителя" по RBAC)
$log->out('Fetching clients to process');
open(my $fh, "<:encoding(UTF-8)", $FILE_WITH_CLIENT_IDS_TO_PROCESS) or $log->die("Error opening $FILE_WITH_CLIENT_IDS_TO_PROCESS:$!");
my @client_ids = grep { $_ && /^\d+$/ } map { chomp($_); $_ } <$fh>;

$log->out(sprintf('Fetched %s clients to process', scalar(@client_ids)));

my $rbac = eval { RBAC2::Extended->get_singleton(1) } or $log->die("Error initialising RBAC: $@");

CLIENT_LOOP: for my $ClientID (@client_ids) {
    my $lg = $log->msg_prefix_guard("[ClientID=$ClientID]");
    $log->out("start processing client $ClientID");
    
    my $rbac_chief_uid = RBACElementary::rbac_get_chief_rep_of_client($ClientID);
    if ($rbac_chief_uid) {
        $log->out("RBAC chief uid: $rbac_chief_uid");
    } else {
        $log->out("RBAC chief uid is undefined. Trying to get chief uid from balance.");
        my $balance_reps = balance_list_client_passports(0, $ClientID);
        my ($balance_main_rep_uid) = map { $_->{Uid} }  grep { $_->{IsMain} } @$balance_reps;
        if ($balance_main_rep_uid) {
            $log->out("Balance main uid: $balance_main_rep_uid");
            my $err = rbac_switch_client_chief($rbac, $balance_main_rep_uid);
            if ($err) {
                $log->out("Error while switching RBAC chief uid to balance_main_rep_uid=$balance_main_rep_uid: $err. Skipping client. Error");
                next;
            } else {
                $log->out("Chief uid succsessfully switched in RBAC to balance_main_rep_uid=$balance_main_rep_uid");
            }
            $rbac_chief_uid = $balance_main_rep_uid;
        } else {
            $log->out("Balance main uid is undefined. Skipping client. Error");
            next;
        }
    }

    unless (get_one_field_sql(PPC(ClientID => $ClientID), "select 1 from users where ClientID = ? and uid = ?", $ClientID, $rbac_chief_uid)) {
        $log->out("RBAC chief uid is not present in PPC, or attached to another client. Skipping client. Error");
        next;
    }

    my %rbac_client_rep_uids = map { $_ => 1 } @{rbac_get_client_uids_by_clientid($ClientID)};
    $log->out("Client reps in RBAC: " . join(',', keys %rbac_client_rep_uids));

    my $camps_to_fix = get_all_sql(PPC(ClientID => $ClientID), "select cid, uid, ManagerUID, AgencyUID
                                                                  from campaigns c 
                                                                  join users u using(uid) 
                                                                 where u.ClientID = ? and c.uid <> ?",
                                                                       $ClientID, $rbac_chief_uid);
    if (@$camps_to_fix) {
        $log->out("Camps to fix: ". join(',', map { $_->{cid} } @$camps_to_fix));

        my @problem_uids = uniq map { $_->{uid} } @$camps_to_fix;
        for my $puid (@problem_uids) {
            if ($rbac_client_rep_uids{$puid}) {
                $log->out("Rep uid=$puid exists on the same client");
            } else {
                my $puid_client_id = rbac_get_client_clientid_by_uid($puid);
                if ($puid_client_id) {
                    $log->out("Rep uid=$puid exists on another client: $puid_client_id");
                    if (scalar(@problem_uids) > 1) {
                        $log->out("There is more than one problem clients. Skipping client. Error");
                        next CLIENT_LOOP;
                    }
                    if (@{get_all_sql(PPC(shard => 'all'), "select uid from users where ClientID = ?", $puid_client_id)}) {
                        $log->out("There is user on PPC binded to client=$puid_client_id. Skipping client. Error");
                    }
                    my $err = rbac_merge_clients($rbac, $ClientID, $puid_client_id);
                    if ($err) {
                        $log->out("Error while merge of clients $puid_client_id -> $ClientID. Skipping client. Error");
                        next CLIENT_LOOP;
                    } else {
                        $log->out("Successful merge of clients $puid_client_id -> $ClientID in RBAC");
                        eval { _reshard_clients($ClientID, $puid_client_id) };
                        if ($@) {
                            $log->out("Error while _reshard_clients for clients $puid_client_id -> $ClientID: $@. Skipping client. Error");
                            next CLIENT_LOOP;
                        }
                        my $ok = eval { _set_main_client_id($rbac, $ClientID, [$puid_client_id]) };
                        if (!$ok || $@) {
                            my $err = $@ || '';
                            $log->out("Error while _set_main_client_id for clients $puid_client_id -> $ClientID: $err. Skipping client. Error");
                            next CLIENT_LOOP;
                        } else {
                            $log->out("Successful _set_main_client_id for clients $puid_client_id -> $ClientID");
                        }
                    }
                } else {
                    $log->out("Rep uid=$puid not exists. Skipping client. Error");
                    next CLIENT_LOOP;
                }
            }
        }

        for my $camp (@$camps_to_fix) {
            $log->out(hash_merge {}, $camp, {action => 'Start processing campaign'});
            # если представитель не котором висит кампания в PPC привязан к "правильному" клиенту в RBAC
            # то доступ у главного представителя клиента сам собой должен быть к кампании
            # иначе доступ должен появиться после "домержа" клиентов в RBAC
            # проверяем гипотезу, если это не так - пропускаем кампанию, потому что могут быть нюансы
            unless (rbac_is_owner_of_camp($rbac, $rbac_chief_uid, $camp->{cid})) {
                $log->out("Chief rbac rep is not an owner of camp. Skipping camp $camp->{cid}. Error");
                next;
            }

            do_update_table(PPC(ClientID => $ClientID), "campaigns", {uid => $rbac_chief_uid}, where => { cid => $camp->{cid} });
            bs_resync_camps([$camp->{cid}]);            
            $log->out("Camp $camp->{cid} successfully fixed");
        }
    } else {
        $log->out("No campaigns to fix. Skipping client");
        next;

    }
}

$log->out('FINISH');


=head3 _reshard_clients

    Привозит неглавного клиента в один шард с главным или убеждается, что это и так так

    _reshard_clients($old_clientid, $new_clientid);

=cut

sub _reshard_clients {
    my ($second_client_id, $main_client_id) = @_;

    my $main_client_shard = get_shard(ClientID => $main_client_id);
    my $second_client_shard = get_shard(ClientID => $second_client_id);

    if (!$main_client_shard) {
        $log->out("Saving shard $second_client_shard for unexisting ClientID $main_client_id");
        save_shard(ClientID => $main_client_id, shard => $second_client_shard);
        $main_client_shard = $second_client_shard;
    }

    $log->die("No shard for ClientID $main_client_id known") unless $main_client_shard;
    $log->die("No shard for ClientID $second_client_id known") unless $second_client_shard;

    if ($main_client_shard != $second_client_shard) {
        $log->out("Main ClientID $main_client_id is in $main_client_shard shard, but non-primary ClientID $second_client_id lives is in $second_client_shard. Scheduling resharding.");
        do_insert_into_table(PPCDICT, 'reshard_queue', {
            ClientID => $second_client_id,
            status => 'new',
            old_shard => $second_client_shard,
            new_shard => $main_client_shard,
            wanted_start_time__dont_quote => 'NOW()',
        });

        $log->out("Waiting for $second_client_id resharding");
        my $is_reshard_done = 0;
        my $iters = 0;
        do {
            sleep 30;
            $is_reshard_done = get_one_field_sql(PPCDICT, ['
                SELECT 1
                FROM reshard_queue
             ', WHERE => {
                    ClientID => $second_client_id,
                    status => 'done',
                },
            ]);
            $iters++;
        } while (!$is_reshard_done && $iters < $MAX_RESHARD_ITERATIONS);

        if ($is_reshard_done) {
            $log->out("Resharing of $second_client_id successed");
        } else {
            $log->die("Resharing of $second_client_id failed");
        }
    } else {
        $log->out("ClientID $main_client_id and ClientID $second_client_id are in same shard, skipping resharding");
    }
}

=head3 _get_tables_to_merge

    Возвращает ссылку на хеш с таблицами, где надо переносить данные со старого ClientID на новый
    Формат:
        table_name => table_ClientID_field

=cut

sub _get_tables_to_merge {
    return {
        'account_score' => 'ClientID',
        'additions_item_callouts' => 'ClientID',
        'agency_currencies' => 'ClientID',
        'agency_client_relations' => 'client_client_id',
        'agency_total_sums' => 'ClientID',
        'api_special_user_options' => 'ClientID',
        'banner_images_pool' => 'ClientID',
        'banner_images_process_queue' => 'ClientID',
        'bs_export_preprod_exceptions' => 'ClientID',
        'camp_copy_reports_queue' => 'ClientID',
        'clients' => 'ClientID',
        'client_brands' => 'ClientID',
        'client_currency_changes' => 'ClientID',
        'client_domains' => 'ClientID',
        'client_domains_stripped' => 'ClientID',
        'client_limits' => 'ClientID',
        'clients_allow_wallet' => 'ClientID',
        'clients_api_options' => 'ClientID',
        'clients_autoban' => 'ClientID',
        'clients_custom_options' => 'ClientID',
        'clients_geo_cluster_stat' => 'ClientID',
        'clients_geo_ip' => 'ClientID',
        'clients_needed_personal_manager_teaser' => 'ClientID',
        'clients_needy_stat' => 'ClientID',
        'clients_stat' => 'ClientID',
        'clients_to_fetch_nds' => 'ClientID',
        'clients_to_force_multicurrency_teaser' => 'ClientID',
        'client_teaser_data_lastupdate' => 'ClientID',
        'currency_convert_money_correspondence' => 'ClientID',
        'dbqueue_job_archive' => 'ClientID',
        'dbqueue_jobs' => 'ClientID',
        'eventlog' => 'ClientID',
        'feeds' => 'ClientID',
        'force_currency_convert' => 'ClientID',
        'infoblock_teasers_factors' => 'ClientID',
        'mds_metadata' => 'ClientID',
        'mobile_content' => 'ClientID',
        'mod_documents' => 'ClientID',
        'minus_words' => 'ClientID',
        'perf_creatives' => 'ClientID',
        'retargeting_conditions' => 'ClientID',
        'serviced_client_amnesty' => 'ClientID',
        'serviced_client_budget' => 'ClientID',
        'serviced_client_history' => 'ClientID',
        'serviced_client_status' => 'ClientID',
        'sitelinks_sets' => 'ClientID',
        'users' => 'ClientID',
    };
}

=head3 _set_main_client_id

    Сильно изменённая копипаста из старого кода

=cut

sub _set_main_client_id
{
    my ($rbac, $main_client_id, $client_ids) = @_;

    $log->out("_set_main_client_id: no clientids given") if ref($client_ids) ne 'ARRAY' || ! @$client_ids;

    for my $client_id (@$client_ids) {
        next if $client_id == $main_client_id;

        ############# нужно проапдейтить адреса в vcards и удалить дубликаты из addresses
        $log->out('Moving addresses');
        _merge_clients_addresses($main_client_id, $client_id);

        ############# Перенос clients_options
        $log->out('Moving clients_options');
        my $is_main_client_exist = get_one_field_sql(PPC(ClientID => $main_client_id), "select 1 from clients_options where ClientID = ?", $main_client_id);
        my $is_client_exist = get_one_field_sql(PPC(ClientID => $client_id), "select 1 from clients_options where ClientID = ?", $client_id);

        if ($is_main_client_exist && $is_client_exist) {
            # удаляем старые данные
            do_sql(PPC(ClientID => $client_id), "delete from clients_options where ClientID = ?", $client_id);
        } elsif(!$is_main_client_exist && $is_client_exist) {
            my $client_data = get_one_line_sql(PPC(ClientID => $client_id), "
                select balance_tid, overdraft_lim, debt,
                       nextPayDate, statusBalanceBanned, warned_nextPayDate, warned_interval,
                       discount, budget, border_next, discount_next, border_prev
                from clients_options
                where ClientID = ?", $client_id
            );
            $client_data->{ClientID} = $main_client_id;

            do_insert_into_table(PPC(ClientID => $main_client_id), 'clients_options', $client_data);
            do_sql(PPC(ClientID => $client_id), "delete from clients_options where ClientID = ?", $client_id);
        }

        ############# Перенос данных таблиц по ClientID
        my $TABLES_TO_MERGE = _get_tables_to_merge();
        for my $table (sort keys %$TABLES_TO_MERGE) {
            my $field = $TABLES_TO_MERGE->{$table};
            $log->out("Merging data for table $table from ClientID $client_id to $main_client_id using field $field");
            _merge_table($client_id, $main_client_id, $table, $field);
        }

        ############# графики НДС и скидок
        $log->out("Refreshing NDS schedule for ClientID $main_client_id");
        Client::NDSDiscountSchedule::sync_nds_schedule_for_clients([$main_client_id], log => $log, rbac => $rbac);

        $log->out("Refreshing discount schedule for ClientID $main_client_id");
        Client::NDSDiscountSchedule::sync_discount_schedule_for_clients([$main_client_id], log => $log, rbac => $rbac);

        $log->out("Deleting NDS and discount schedule for ClientID $client_id");
        for my $table (qw(client_nds client_discounts)) {
            do_delete_from_table(PPC(ClientID => $client_id), $table, where => {ClientID => $client_id});
        }

        ############# Страны-валюты
        $log->out("Updating country-currencies for ClientID $main_client_id");
        my $agency_id = Primitives::get_client_first_agency($main_client_id);
        Client::CurrencyTeaserData::fetch_client_multicurrency_teaser_data($main_client_id, $agency_id, balance_timeout => 320);

        ############# Шардирование
        $log->out("Updating shard chain $client_id => $main_client_id");
        update_shard_chain('ClientID', $client_id, $main_client_id);

        Yandex::Log
          ->new(%Yandex::Balance::BALANCE_CALLS_LOG_SETTINGS, msg_prefix => 'set_main_client_id:')
          ->out("new ClientID: $main_client_id, old ClientID: $client_id");
    }

    return 1;
}

=head3 _merge_table

=cut

sub _merge_table {
    my ($client_id, $main_client_id, $table, $field) = @_;

    my $old_data = get_all_sql(PPC(ClientID => $main_client_id), ["SELECT * FROM $table", WHERE => {$field => $client_id}]);
    $log->out("Old data for $client_id for table $table:", $old_data);

    do_update_table(PPC(ClientID => $main_client_id), $table, {$field => $main_client_id}, where => {$field => $client_id}, ignore => 1);
    do_delete_from_table(PPC(ClientID => $main_client_id), $table, where => {$field => $client_id});
}

=head3 _merge_clients_addresses

=cut

sub _merge_clients_addresses {
    my ($main_client_id, $second_client_id) = @_;

    my $all_addresses_data = get_all_sql(PPC(ClientID => $second_client_id), [
        'SELECT aid, address, ClientID FROM addresses', WHERE => { ClientID => [$main_client_id, $second_client_id] }
    ]);
    my %all_addresses;
    for my $row (@$all_addresses_data) {
        $all_addresses{ $row->{address} }->{ $row->{ClientID} } = $row->{aid};
    }
    for my $address (keys %all_addresses) {
        if (scalar( keys %{$all_addresses{$address}} ) == 1 && exists $all_addresses{$address}->{$main_client_id}) {
            # только один адрес и тот на новый ClientID, пропускаем
            next;
        } elsif (scalar( keys %{$all_addresses{$address}} ) == 1 && exists $all_addresses{$address}->{$second_client_id}) {
            # только один адрес и тот на старый ClientID, апдейтим только addresses
            do_update_table(PPC(ClientID => $second_client_id), 'addresses', {ClientID => $main_client_id},
                            where => {ClientID => SHARD_IDS, aid=>$all_addresses{$address}->{$second_client_id}});
        } elsif (scalar( keys %{$all_addresses{$address}} ) == 2) {
            # на старом и новом клиенте есть 2 одинаковых адреса, для старого удаляем из addresses и апдейтим vcards
            do_delete_from_table(PPC(ClientID => $second_client_id), 'addresses',
                            where => {ClientID => SHARD_IDS, aid=>$all_addresses{$address}->{$second_client_id}});
            do_update_table(PPC(ClientID => $second_client_id), 'vcards', {address_id => $all_addresses{$address}->{$main_client_id}},
                                                         where => {address_id => $all_addresses{$address}->{$second_client_id}});
        }
    }
}
