#!/usr/bin/perl

=head1 DEPLOY

# approved by pankovpv
# .migr
{
  type => 'script',
  when => 'after',
  time_estimate => "2 дня",
  comment => q{
    Перед запуском сделать:
    scp root@ppcdev6:/home/andy-ilyin/direct-utils/one-off-stat/20160224-DIRECT-51253-ads-with-mismatching-domains/bids.out /tmp/bids.out.20160229

    Скрипт можно убивать и перезапускать.

    При перезапуске стоит добавить параметр --min-bid=N, где N взять из строчки лога {"max_bid":"N"}

    На ТС выполнять не надо.
  },
  script => '/var/www/ppc.yandex.ru/deploy/20160229_fix_banner_domains.pl /tmp/bids.out.20160229',
}

=cut

use Direct::Modern;

use Getopt::Long;

use Yandex::ListUtils;
use Yandex::Validate;

use my_inc '..';

use BS::ResyncQueue;
use Direct::Banners;
use Direct::Model::BannerText::Manager;
use Direct::Model::BannerDynamic::Manager;
use Direct::Model::BannerMobileContent::Manager;
use RedirectCheckQueue;
use ScriptHelper;
use Settings;

# какого размера пачками загружать объявления из базы и обрабатывать
my $CHUNK_SIZE = 5_000;

# c каким приоритетом лениво переотправлять объявления в БК
my $BS_RESYNC_PRIORITY = 45;

my @BANNER_UPDATE_TIMES;

my $min_bid;
Getopt::Long::GetOptions(
    'min-bid=n' => \$min_bid,
) or die "Invalid options";

$log->out('START');

my ($bids_filename) = @ARGV;

die "no bids filename" unless defined $bids_filename;

$log->out( { bids_filename => $bids_filename } );

open my $fh_in, '<', $bids_filename or die "Cannot open $bids_filename: $!";

# проверка показала, что с файлом на 12 млн bid-ов и 120M весом структура будет занимать
# чуть меньше гигабайта; если файл будет больше, читать его целиком в память не выйдет,
# надо будет делать по-другому
my @bids;
my $skipped_bids_count = 0;
while ( my $line = <$fh_in> ) {
    chomp $line;

    next if $line eq '';
    unless ( is_valid_id($line) ) {
        $log->out( { invalid_bid => $line } );
        next;
    }

    if ( defined $min_bid && $line < $min_bid ) {
        $skipped_bids_count++;
        next;
    }

    push @bids, $line;
}

close $fh_in;

$log->out( { skipped_bids_count => $skipped_bids_count } );

@bids = nsort @bids;

$log->out( { bid_count => scalar(@bids) } );

for my $bids_chunk ( chunks( \@bids, $CHUNK_SIZE ) ) {
    my $banners_by_bid = Direct::Banners->get_by( banner_id => $bids_chunk )->items_by('id');

    my %updated_banners = map { $_ => [] } qw( text dynamic mobile );
    my @resync_data;

    for my $bid (@$bids_chunk) {
        my $banner = $banners_by_bid->{$bid};

        unless ($banner) {
            $log->out( { bid_not_found => $bid } );
            next;
        }

        $log->out( { bid => $banner->id, href => $banner->href, domain => $banner->domain } );

        my ( $canonical_domain, $need_check_redirect ) =
            RedirectCheckQueue::domain_need_check_redirect( { href => $banner->href, domain => $banner->domain } );

        if ($need_check_redirect) {
            # если у href домен счётчика, ставим в очередь на простукивание редиректов;
            # дальше процесс, который занимается этой очередью, сам поставит домен
            # и переотправит в БК как надо
            RedirectCheckQueue::push_banners( [ { bid => $bid } ] );
            $log->out( { bid_checking_redirect => $bid } );
        } else {
            # если у href не домен счётчика, какой же должен быть "правильный" домен?
            # если "правильный" домен отличается от домена в объявлении, надо поменять его в базе
            if ( $banner->domain ne $canonical_domain ) {
                my $original_domain = $banner->domain;

                $banner->domain($canonical_domain);
                push @{ $updated_banners{ $banner->banner_type } }, $banner;

                $log->out( {
                    bid_updated_domain => $bid,
                    domain => $original_domain,
                    canonical_domain => $canonical_domain,
                } );
            } else {
                $log->out( { bid_left_intact => $bid } );
            }

            # но независимо от того, меняли мы домен или нет, надо лениво переотправить в БК,
            # а то вдруг скрипт уже обрабатывал это объявление в прошлом запуске и теперь
            # домен там совпадает с "правильным"
            push @resync_data, { bid => $bid, cid => $banner->campaign_id, priority => $BS_RESYNC_PRIORITY };
        }

    }

    if ( my @banners = @{ $updated_banners{text} } ) {
        Direct::Model::BannerText::Manager->new( items => \@banners )->update;
    }

    if ( my @banners = @{ $updated_banners{dynamic} } ) {
        Direct::Model::BannerDynamic::Manager->new( items => \@banners )->update;
    }

    if ( my @banners = @{ $updated_banners{mobile} } ) {
        Direct::Model::BannerMobileContent::Manager->new( items => \@banners )->update;
    }

    $log->out( { updated => { map { $_ => scalar( @{ $updated_banners{$_} } ) } keys %updated_banners } } );

    $log->out( { resync_data => \@resync_data } );
    BS::ResyncQueue::bs_resync(\@resync_data);

    $log->out( { max_bid => $bids_chunk->[-1] } );
}

$log->out('FINISH');
