#!/usr/bin/perl

=head1 DEPLOY

# approved by pankovpv
# .migr
{
  type => 'script',
  when => 'after',
  time_estimate => "6 часов",
  comment => "Массовое рассервисирование клиентов для МКС. Запускать по отмашке Нади nadiano@ или Эльвиры galeeva@!"
}

=cut

use Direct::Modern;

use Yandex::DBShards;
use Yandex::DBTools;
use Yandex::ListUtils qw/chunks/;
use Yandex::Overshard;

use Path::Tiny;
use List::MoreUtils qw/uniq/;
use POSIX qw/strftime/;

use my_inc '..';

use Campaign::Types qw/get_camp_kind_types/;
use Campaign;
use Notification;
use Primitives;
use RBAC2::Extended;
use RBACDirect;
use ShardingTools qw/choose_shard_param/;

use ScriptHelper;
use Settings;

my $OPERATOR_UID = 1;
# количество логинов, рассервисируемых за итерацию
my $CHUNK_SIZE = 250;
# таймаут между итерациями, сек.
my $SLEEP_TIMEOUT = 10;

my $data_path = my_inc::path('./20160406_mass_client_unservicing.data');
my $rbac = RBAC2::Extended->get_singleton($OPERATOR_UID) or $log->die("Can't connect to RBAC: $@");


$log->out('START');

my @client_ids = path($data_path)->lines({ chomp => 1}) or $log->die("Can't open data file $data_path: $@");

$log->out('Got '. scalar @client_ids .' client_ids for unservicing');

foreach_shard ClientID => \@client_ids, chunk_size => 1_000, with_undef_shard => 1, sub {
    my ($shard, $client_ids_chunk) = @_;

    # ошибка, если не определили шард
    if (!$shard) {
        $log->msg_prefix("[shard=undef]");
        $log->out({ClientID => $client_ids_chunk});
        return;
    }

    $log->msg_prefix("[shard=$shard]");
    $log->out('Got '. scalar @$client_ids_chunk .' client_ids for unservicing');

    my $logins2ClientID = get_hash_sql(PPC(shard => $shard),
                                    ["SELECT distinct u.login, u.ClientID
                                      FROM users u
                                      JOIN campaigns c ON c.uid = u.uid",
                                      where => {
                                        'u.ClientID' => $client_ids_chunk,
                                        'c.ManagerUID__gt' => 0,
                                      }
                                    ]);

    foreach my $logins_chunk (chunks([keys %$logins2ClientID], $CHUNK_SIZE)) {
        foreach my $login (@$logins_chunk) {
            my $client_id = $logins2ClientID->{$login};
            $log->msg_prefix("[shard=$shard,ClientID=$client_id,login=$login]");
            $log->out("start unServicing");
            my $result =  _do_login_unservice($login);
            $log->out($result);
        }
        $log->msg_prefix("[shard=$shard]");
        $log->out("Sleeping between login chunks for $SLEEP_TIMEOUT sec.");
        sleep $SLEEP_TIMEOUT;
    }
    $log->out("Done");
};

$log->out('FINISH');


=head2 _do_login_unservice

    По сути это копипаста контроллера cmd_unServicingCamp, отличается:
    - Наложен патч из DIRECT-52813: Не рассервисировать МКБ при рассервисировании в Директе;
    - Рассервисируем чанками по 250 логинов c таймаутом 10 секунд на чанк;
    - Рассервисируем не форсированно;
    - Выкинута проверка на право тимлидера рассервисировать только кампании своих менеджеров;
    - Выкинуты send_alert-ы, ошибки пишем в лог.

=cut

sub _do_login_unservice
{
    my $login = shift;

    my $UID = $OPERATOR_UID;
    my $vars;

    my $supported_camp_types = get_camp_kind_types("web_edit_base");

    my %unserviced_clients;
    my $all_camps = [];
    my @errors;

    eval {
        my $campuid = get_uid_by_login($login);
        die \"логин \"$login\" не существует" if ! defined $campuid;

        my $sql_where;
        my %wallet_camp;

        $sql_where = { 'c.uid' => $campuid };

        my $empty_wallets = get_one_column_sql(PPC(uid => $campuid), [
            "SELECT wc.cid
            FROM campaigns wc
            LEFT JOIN campaigns c ON (c.wallet_cid = wc.cid AND c.uid = wc.uid)",
            WHERE => {
                'wc.uid' => $campuid,
                'wc.type' => 'wallet',
            },
            "GROUP BY wc.cid",
            "HAVING COUNT(c.cid) = 0",
        ]);

        for my $wallet_cid (@$empty_wallets) {
            my $manager_uid = rbac_is_scampaign($rbac, $wallet_cid);
            next if !$manager_uid;
            $wallet_camp{$wallet_cid} = { client_uid => $campuid };
        }

        my @shard = choose_shard_param($sql_where, [qw/cid uid/], set_shard_ids => 1);
        $sql_where->{'c.statusEmpty'} = 'No';
        $sql_where->{'c.type'} = $supported_camp_types;
        $all_camps = get_all_sql(PPC(@shard),[
                                       "select cid,
                                        c.name  as camp_name,
                                        u.uid   as client_uid,
                                        u.login as client_login,
                                        u.fio   as client_fio,
                                        u.email as client_email,
                                        u.valid as client_valid,
                                        u.ClientID, c.ManagerUID,
                                        c.wallet_cid,
                                        -- Подсчитаем кол-во сервисируемых кампаний под ОС под одним менеджером
                                        IF(c.wallet_cid > 0, (
                                            SELECT count(*) FROM campaigns sc WHERE sc.uid = c.uid AND sc.wallet_cid = c.wallet_cid AND sc.ManagerUID = c.ManagerUID
                                        ), 0) manager_serv_camps_in_wallet_count
                                        from campaigns c
                                          join users u on c.uid = u.uid
                                        ", where => $sql_where,
                                        ]);

        enrich_data($all_camps,
            using => 'ManagerUID',
            sub {
                my $manager_uids = shift;
                return get_hashes_hash_sql(PPC(uid => $manager_uids),
                    ["select u.uid as manager_uid, u.fio as manager_fio, u.email as manager_email, u.valid as manager_valid
                        from users u", where => { uid => SHARD_IDS }
                    ]
                );
            }
        );

        die \qq/кампании не найдены/ if ref($all_camps) ne 'ARRAY' || ! @$all_camps;

        my %serv_camps_in_wallet_count;

        CAMPAIGN:
        for my $row (@$all_camps) {
            my $cid = $row->{cid};
            my $wallet_cid = $row->{wallet_cid};
            my $client_uid = $row->{client_uid};
            my $manager_uid = $row->{manager_uid};

            $serv_camps_in_wallet_count{$wallet_cid}{$manager_uid} = $row->{manager_serv_camps_in_wallet_count} if $wallet_cid && $manager_uid && !exists $serv_camps_in_wallet_count{$wallet_cid}{$manager_uid};

            eval {
                my $manager_uid = rbac_is_scampaign($rbac, $cid);
                die \qq/кампания "$cid" - не на обслуживании менеджером/ unless $manager_uid;

                die \qq/кампания "$cid" - не принадлежит логину "$login"/
                    if get_one_field_sql(PPC(cid =>  $cid), "select count(*) from campaigns where statusEmpty = 'No' and cid = ? and uid = ?", $cid, $campuid) == 0;

                die \qq/перенос невозможен (RBAC)/ if rbac_move_scamp_to_nscamp($rbac, $cid);
                do_update_table(PPC(cid => $cid), 'campaigns', { statusBsSynced => 'No' }, where => { cid => $cid }); # send to BS
                campaign_manager_changed($rbac, $UID, $cid, 0, $manager_uid);

                $row->{done} = 1;
                if ($row->{ClientID}) {
                    $unserviced_clients{$row->{ClientID}} = 1;
                }

                if ($wallet_cid) {
                    $serv_camps_in_wallet_count{$wallet_cid}{$manager_uid}--;

                    # Проверим, должны ли мы рассервисировать или перевести на другого менеджера кампанию-кошелек
                    if (
                        $serv_camps_in_wallet_count{$wallet_cid}{$manager_uid} == 0 &&
                        rbac_is_scampaign($rbac, $wallet_cid)
                    ) {
                        $wallet_camp{$wallet_cid} = { client_uid => $client_uid};
                    }
                }
            }; # /eval

            if ($@) {
                if (ref($@) eq 'SCALAR') {
                    my $error = ${$@};
                    $log->out($error);
                    push @errors, $error;
                } else {
                    $log->out( Dumper $@ );
                    push @errors, 'Перенести не удалось. Обратитесь к разработчикам.';
                    # send_alert(Dumper(localtime() . "", $@, \%ENV) . '', "unServicing failed");
                    last CAMPAIGN;
                }
                next CAMPAIGN;
            }
        } # / for by camps

        # отдельно отцепляем кошельки
        WALLET:
        for my $wallet_cid (keys %wallet_camp) {
            eval {
                my $move_fail = rbac_move_scamp_to_nscamp($rbac, $wallet_cid);
                die \qq/перенос кошелька невозможен (RBAC, $move_fail)/ if $move_fail;

                do_update_table(PPC(cid => $wallet_cid), 'campaigns', { statusBsSynced => 'No' }, where => { cid => $wallet_cid }); # send to BS
                campaign_manager_changed($rbac, $UID, $wallet_cid);

                # Проверим, есть ли другие менеджеры под этим ОС
                my $client_uid = $wallet_camp{$wallet_cid}->{client_uid};
                my $new_manager_uid = get_one_field_sql(PPC(uid => $client_uid), [
                        q{SELECT c.ManagerUID FROM campaigns c},
                        WHERE => {
                                'c.uid' => $client_uid,
                                'c.archived' => 'No',
                                'c.wallet_cid' => $wallet_cid,
                                'c.ManagerUID__gt' => 0,
                                'c.type' => $supported_camp_types,
                            }, q{
                            GROUP BY c.ManagerUID
                            ORDER BY count(c.cid) DESC
                            LIMIT 1
                        }]);
                if ($new_manager_uid) {
                    Campaign::send_camp_to_service($rbac, $wallet_cid, $client_uid, $new_manager_uid, send_other_camps => 0, force => 1);
                }
            };
            if ($@) {
                if (ref($@) eq 'SCALAR') {
                    my $error = ${$@};
                    $log->out($error);
                    push @errors, $error;
                } else {
                    $log->out( Dumper $@ );
                    push @errors, 'Перенести не удалось. Обратитесь к разработчикам.';
                    # send_alert(Dumper(localtime() . "", $@, \%ENV) . '', "unServicing failed");
                }
                next WALLET;
            }
        }
        if (@errors) {
            my $error_str = join "\n", @errors;
            die \$error_str;
        }
    }; # / outer eval

    if ($@) {
        if (ref($@) eq 'SCALAR') {
            $vars->{error} = ${$@};
        } else {
            $vars->{error} = 'Перенести не удалось. Обратитесь к разработчикам.';
            # send_alert(Dumper(localtime() . "", $@, \%ENV) . '', "unServicing failed");
            $log->out($@);
        }
    }

    # result
    my %mailvars;
    for my $row (@$all_camps) {
        next unless $row->{done};

        for my $email_type (qw/manager_email client_email/) {

            if (exists $mailvars{ $row->{$email_type} }{vars}) {

                push @{ $mailvars{ $row->{$email_type} }{vars}{camps} }, {cid => $row->{cid}, camp_name => $row->{camp_name}};
                $mailvars{ $row->{$email_type} }{vars}{cids} .= ", $row->{cid}";

            } else {
                $mailvars{$row->{$email_type}}{vars} = {
                    cids          => $row->{cid},
                    date          => strftime("%d.%m.%Y", localtime),
                    camp_name     => $row->{camp_name},
                    client_uid    => $row->{client_uid},
                    client_fio    => $row->{client_fio},
                    client_login  => $row->{client_login},
                    client_id     => $row->{ClientID},
                    ClientID      => $row->{ClientID},
                    manager_uid   => $row->{manager_uid},
                    manager_fio   => $row->{manager_fio},
                    manager_email => $row->{manager_email},
                    camps         => [{cid => $row->{cid}, camp_name => $row->{camp_name}}],
                    $email_type   => 1,
                    forcedunservice => 0,
                };
            }
            $mailvars{$row->{$email_type}}{$email_type} = 1;
        }
    }

    for my $email (keys %mailvars) {
        $vars->{info_cids} = join ',', uniq map {$_->{cid}} @{ $mailvars{$email}{vars}{camps} } if $mailvars{$email}{client_email};
        add_notification($rbac, 'unservicing', $mailvars{$email}{vars});
    }

    return $vars;
}
