#!/usr/bin/perl

=head1 DEPLOY

# approved by ppalex
# .migr
[
  {
    type => 'sql',
    webstop => "0",
    db => "ppc:all",
    when => 'before',
    time_estimate => "несколько секунд",
    sql => "ALTER TABLE clients_options MODIFY COLUMN client_flags SET('no_text_autocorrection','no_display_hrefs','not_agreed_on_creatives_autogeneration','can_copy_ctr','not_convert_to_currency') DEFAULT NULL"
  },
  {
    type => 'script',
    when => 'after',
    time_estimate => "20 минут",
    comment => "Помечаем неконвертируемых клиентов специальным флагом not_convert_to_currency снимаем им флаг hidden"
  }
]

=cut

use Direct::Modern;

use Yandex::DBTools;
use Yandex::DBShards;

use List::MoreUtils qw/part/;
use Path::Tiny;

use my_inc '..';

use Client;
use PrimitivesIds qw/get_login2clientid/;
use ScriptHelper;
use Settings;

my $data_path = my_inc::path('./20161031_clients_options_not_convert_to_currency.data');

$log->out('START');

my @logins = path($data_path)->lines({ chomp => 1}) or $log->die("Can't open data file $data_path: $@");

$log->out('Got '. scalar @logins .' logins from file');

foreach_shard login => \@logins, chunk_size => 1000, with_undef_shard => 1, sub {
    my ($shard, $logins_chunk) = @_;

    if ($shard) {
        my $msg_prefix_quard = $log->msg_prefix_guard("[shard=$shard]");

        my $login2clientid = get_login2clientid(login => $logins_chunk);
        my $clientid2login = { reverse %$login2clientid };
        my $clids = [ keys %$clientid2login ];

        # Проверяем, что клиент в фишках
        my $clid2currencies = mass_get_client_currencies($clids);
        my ($good_currency_client_ids, $bad_currency_client_ids) = part { $clid2currencies->{$_}->{work_currency} eq 'YND_FIXED' ? 0 : 1 } @$clids;
        $log->out({bad_work_currency_logins => [ map {$clientid2login->{$_}} @$bad_currency_client_ids ]} );

        # Проверяем, что клиент не стоит в очереди на конвертацию
        my $clids_in_queue =get_hash_sql(PPC(shard => $shard), ["select ClientID, 1 from currency_convert_queue", where => {ClientID => $good_currency_client_ids}]);
        my $client_ids = [ map { $clids_in_queue->{$_} ? () : $_ } @$good_currency_client_ids ];
        $log->out({in_currency_convert_queue_logins => [ map {$clientid2login->{$_}} keys %$clids_in_queue ]} );

        # Оставляем только тех, у кого установлен флаг hidden
        my $clients2hidden = get_hash_sql(PPC(shard => $shard), ["select ClientID, IF(hidden = 'Yes', 1, 0) from users", where => {ClientID => $client_ids}]);
        my ($hidden_clients, $not_hidden_clients) = part { $clients2hidden->{$_} ? 0 : 1 } keys %$clients2hidden;
        $log->out({not_hidden_clients => [ map {$clientid2login->{$_}} @$not_hidden_clients ]} );

        # Обрабатываем оставшихся клиентов
        foreach my $clientid (@$hidden_clients) {
            my $msg_prefix_quard = $log->msg_prefix_guard("[shard=$shard,ClientID=$clientid,login=$clientid2login->{$clientid}]");
            $log->out('set not_convert_to_currency flag');
            create_update_client({client_data => {ClientID => $clientid, not_convert_to_currency => 1}});
        }
        # снимаем тизер мультивалютности
        $log->out({client_ids_to_delete_teaser_and_hidden_flag => $hidden_clients});
        $log->out('Delete clients_to_force_multicurrency_teaser');
        do_delete_from_table(PPC(shard => $shard), 'clients_to_force_multicurrency_teaser', where => {ClientID => $hidden_clients});
        # снимаем флаг hidden
        $log->out('Drop hidden flag');
        do_update_table(PPC(shard => $shard), 'users', {hidden => 'No'}, where => {ClientID => $hidden_clients, hidden => 'Yes'});
    } else {
        $log->out({not_found_logins => $logins_chunk});
    }
};

$log->out('FINISH');

