#!/usr/bin/perl

=head1 DEPLOY

# approved by ppalex
# .migr

[
  {
    type => 'script',
    when => 'after',
    time_estimate => 'больше 1 дня',
    comment => 'Проставляем значения в таблицу mod_mail_candidates и новые поля в mod_reasons. Можно перезапускать',
  },
]

=cut

use Direct::Modern;

use Yandex::DBTools;
use Yandex::DBShards;
use Yandex::ListUtils qw/chunks/;

use my_inc '..';

use ScriptHelper;
use Settings;
use ShardingTools qw/ppc_shards/;

my $SELECT_CHUNK_SIZE = 10000;
my $INSERT_CHUNK_SIZE = 1000;
my $RESHARD_CHUNK_SIZE = 50;

my %KEYS_DATA = (
    cid => {
        type => 'campaign',
        fields => 'c.cid, c.ClientID',
        joins => "LEFT JOIN campaigns c ON r.id = c.cid",
    },
    bid => {
        type => [qw/banner contactinfo sitelinks_set image display_href/],
        fields => 'b.cid, c.ClientID',
        joins => "LEFT JOIN banners b ON r.id = b.bid
                  LEFT JOIN campaigns c ON b.cid = c.cid",
    },
    pid => {
        type => 'phrases',
        fields => 'p.cid, c.ClientID',
        joins => "LEFT JOIN phrases p ON r.id = p.pid
                  LEFT JOIN campaigns c ON p.cid = c.cid",
    },
    image_id => {
        type => 'image_ad',
        fields => 'i.cid, c.ClientID',
        joins => "LEFT JOIN images i ON r.id = i.image_id
                  LEFT JOIN campaigns c ON i.cid = c.cid",
        shard_table => 'images',
    },
    creative_id => {
        type => 'perf_creative',
        fields => '0 as cid, pc.ClientID',
        joins => "LEFT JOIN perf_creatives pc ON r.id = pc.creative_id",
        shard_table => 'banners_performance',
    },
    banner_creative_id => {
        type => [qw/canvas video_addition/],
        fields => 'bp.cid, c.ClientID',
        joins => "LEFT JOIN banners_performance bp ON r.id = bp.banner_creative_id
                  LEFT JOIN campaigns c ON bp.cid = c.cid",
        shard_table => 'banners_performance',
    },
    additions_item_id => {
        type => 'callout',
        fields => '0 AS cid, aic.ClientID',
        joins => "LEFT JOIN additions_item_callouts aic ON r.id = aic.additions_item_id",
        shard_table => 'additions_item_callouts',
    },
    mobile_content_id => {
        type => 'mobile_content',
        fields => '0 AS cid, mc.ClientID',
        joins => "LEFT JOIN mobile_content mc ON r.id = mc.mobile_content_id",
        shard_table => 'mobile_content',
    },
    mbid => {
        type => 'media_banner',
        fields => 'mg.cid, c.ClientID',
        joins => "LEFT JOIN media_banners mb ON r.id = mb.mbid
                  LEFT JOIN media_groups mg ON mb.mgid = mg.mgid
                  LEFT JOIN campaigns c ON mg.cid = c.cid",
    },
    mgid => {
        type => 'media_group',
        fields => 'mg.cid, c.ClientID',
        joins => "LEFT JOIN media_groups mg ON r.id = mg.mgid
                  LEFT JOIN campaigns c ON mg.cid = c.cid",
        shard_table => 'media_groups',
    },
);


sub process_all_shards {
    my $resharded_cnt = 0;

    for my $shard (ppc_shards()) {
        my $guard = $log->msg_prefix_guard("[SHARD_$shard]");
        for my $key (keys %KEYS_DATA) {
            $resharded_cnt += process_key($shard, $key);
        }
    }

    return $resharded_cnt;
}


sub process_key {
    my ($shard, $key) = @_;

    $log->out("Processing $key");

    my $resharded_cnt = 0;
    my $chunk_num = 1;
    my $border_id;
    while (1) {
        my $data = get_data_chunk($shard, $key, $border_id);
        $log->out("Processing chunk $chunk_num of ".scalar(@$data)." entries");

        my @invalid_values;
        my %update_data;
        my %new_table_data;
        for my $item (@$data) {
            if ((! defined $item->{cid} || $item->{cid} == 0) && (! defined $item->{ClientID})) {
                push @invalid_values, $item;
                next;
            }

            $update_data{$item->{rid}} = {cid => $item->{cid}, ClientID => $item->{ClientID}};
            if ($item->{statusSending} ne "Yes") {
                $new_table_data{$item->{cid}} = [$item->{cid}, $item->{ClientID}, "New"];
            }
        }

        if (@invalid_values) {
            $resharded_cnt += process_sharded_entries($shard, $key, \@invalid_values);
        }
        do_in_transaction {
            if (%update_data) {
                update_mod_reasons($shard, \%update_data);
            }
            if (%new_table_data) {
                update_mod_mail_candidates($shard, [values %new_table_data]);
            }
        };

        last if (scalar @$data < $SELECT_CHUNK_SIZE);

        $chunk_num++;
        $border_id = $data->[-1]->{rid};
    }

    return $resharded_cnt;
}


sub get_data_chunk {
    my ($shard, $key, $border_id) = @_;

    return get_all_sql(PPC(shard => $shard),
            ["SELECT STRAIGHT_JOIN r.rid, r.id, r.statusSending,", $KEYS_DATA{$key}->{fields},
               "FROM mod_reasons r", $KEYS_DATA{$key}->{joins},
                WHERE => {
                    'r.type' => $KEYS_DATA{$key}->{type},
                    'r.ClientID' => 0,
                    'r.cid' => 0,
                    ($border_id ? ('r.rid__gt' => $border_id) : ())
                },
              "ORDER BY r.rid
              LIMIT $SELECT_CHUNK_SIZE"]);
}


sub update_mod_reasons {
    my ($shard, $rid2data) = @_;

    my $cnt = do_mass_update_sql(PPC(shard => $shard), 'mod_reasons', 'rid', $rid2data);
    $log->out("Updated $cnt entries in mod_reasons");
}


sub update_mod_mail_candidates {
    my ($shard, $mail_candidates) = @_;

    my $cnt = do_mass_insert_sql(PPC(shard => $shard), "INSERT IGNORE INTO mod_mail_candidates (cid, ClientID, status) VALUES %s", $mail_candidates);
    $cnt += 0;

    $log->out("Inserted $cnt new entries in mod_mail_candidates");
}


sub process_sharded_entries {
    my ($old_shard, $key, $entries_with_wrong_shard) = @_;

    my $cnt = scalar @$entries_with_wrong_shard;
    $log->out("Got $cnt entries with wrong shard with type $key");

    my %entries = map {($_->{id} => undef)} @$entries_with_wrong_shard;
    my @entries_ids = keys %entries;
    my $id2shard;
    if (exists $KEYS_DATA{$key}->{shard_table}) {
        $id2shard = get_shard_custom($key => \@entries_ids);
    } else {
        $id2shard = get_shard_multi($key => \@entries_ids)
    }

    my %shard2id;
    for my $id (keys %$id2shard) {
        if (!$id2shard->{$id}) {
            $log->out({msg => 'Cannot find shard for entry', shard => $old_shard, id => $id, key => $key});
            next;
        }
        if ($id2shard->{$id} == $old_shard) {
            $log->out({msg => 'Old shard is new shard for entry', shard => $old_shard, id => $id, key => $key});
            next;
        }

        push @{$shard2id{$id2shard->{$id}}}, $id;
    }

    my $resharded_cnt = 0;
    for my $shard (keys %shard2id) {
        for my $chunk (chunks($shard2id{$shard}, $RESHARD_CHUNK_SIZE)) {
            $log->out({msg => 'Keys to reshard', shard => $old_shard, data => $chunk});
            my $data = get_all_sql(PPC(shard => $old_shard),
                ["SELECT rid, id, type, statusSending, statusModerate, statusPostModerate, timeCreated, reason
                    FROM mod_reasons",
                   WHERE => {
                        type => $KEYS_DATA{$key}->{type},
                        id => $chunk,
                    }]);
            if ($data && @$data) {
                $resharded_cnt += do_mass_insert_sql(PPC(shard => $shard),
                    "INSERT IGNORE INTO mod_reasons (id, type, statusSending, statusModerate, statusPostModerate, timeCreated, reason) VALUES %s",
                    [map { [$_->{id}, $_->{type}, $_->{statusSending}, $_->{statusModerate}, $_->{statusPostModerate}, $_->{timeCreated}, $_->{reason} ] } @$data]
                );
                $log->out({msg => 'Deleting data', shard => $old_shard, new_shard => $shard, data => $data});
                do_delete_from_table(PPC(shard => $old_shard), 'mod_reasons',
                    where => {rid => [map {$_->{rid}} @$data]});
            }
        }
    }

    return $resharded_cnt;
}


sub get_shard_custom {
    my ($key, $ids) = @_;

    my $table = $KEYS_DATA{$key}->{shard_table};

    my %id2shard;
    for my $shard (ppc_shards) {
        my $data = get_one_column_sql(PPC(shard => $shard),
            [ "SELECT $key FROM $table", WHERE => { $key => $ids } ]);
        for my $id (@$data) {
            $id2shard{$id} = $shard;
        }
    }
    return \%id2shard;
}


$log->out('START');

$log->out('Processing first iteration');
my $resharded_num = process_all_shards();

if ($resharded_num > 0) {
    $log->out('Processing second iteration');
    process_all_shards();
}
$log->out('FINISH');
