#!/usr/bin/perl

=head1 DEPLOY

# approved by hrustyashko
# .migr
{
  type => 'script',
  when => 'after',
  time_estimate => "~2 часа",
  comment => "Нужно запустить на ppcscripts

    перед запуском привезти файл с визитками, баннеры которых нужно переотправить
    scp ppcdev4:/tmp/andreymak/vcards_to_resync /tmp/vcards_to_resync

    потом запустить миграцию такой командой:
    ./deploy/20170703_reresync_banners_with_permalinks.pl /tmp/vcards_to_resync

    Если скрипт упадет, это не очень страшно. Если упадет в начале работы, то можно просто перезапустить с тем же файлом.
    Если упадем в середине, можно по message логам посмотреть, какие строки уже были обработаны, и обрезать файл с того места
  "
}

=head1 DESCRIPTION

    Читает список визиток, которые изменились при запуске ppcUpdateVcardPermalinks 2017-06-27, и переотправляет активные
    баннеры, привязанные к этим визиткам, с большим приоритетом.

=cut

use Direct::Modern;

use JSON;

use Yandex::DBTools;
use Yandex::DBShards;

use my_inc '..';

use BS::ResyncQueue;
use ScriptHelper 'Yandex::Log' => 'messages';
use Settings;

use constant {
    CHUNK_SIZE => 1_000,
    RESYNC_PRIORITY => BS::ResyncQueue::PRIORITY_UPDATE_PERMALINKS,
};

my $vcards_file = $ARGV[0];
unless ($vcards_file) {
    die "need vcards file";
}
unless (-f $vcards_file) {
    die "can't read vcards file $vcards_file\n";
}

$log->out('START');

resync_banners_from_table($vcards_file);

=head2 resync_banners_from_table($diff_table)

    Принимает объект Yandex::YT::Table с таблицей
    Читает ее по кусочкам в CHUNK_SIZE строк, и передает для обработки в resync_banners_chunk

=cut

sub resync_banners_from_table {
    my ($vcards_file) = @_;

    open my $fh, "<:utf8", $vcards_file;

    my $i = 0;
    my $rows;
    while (($rows = read_chunk($fh, CHUNK_SIZE)) && @$rows > 0) {
        my $chunk_range = "[#$i:#".($i + @$rows)."]";
        $i += @$rows;
        $log->out({ message => "reading diff chunk $chunk_range" });
        resync_banners_chunk($rows);
    }

    close $fh;
}

=head2 read_chunk($fh, $chunk)
=cut
sub read_chunk {
    my ($fh, $chunk) = @_;

    my @rows;
    my $line;
    while ($chunk > 0 && defined($line = <$fh>)) {
        chomp($line);
        push @rows, from_json($line);
        --$chunk;
    }

    return \@rows;
}

=head2 resync_banners_chunk($rows)

    Принимает массив строк из таблицы с визитками, которые нужно обновить.
    ставит активные баннеры, привязанные к визиткам, на переотправку в БК

=cut

sub resync_banners_chunk {
    my ($rows) = @_;

    foreach_shard('cid' => $rows, sub {
        my ($shard_num, $shard_rows) = @_;

        my $vcard_ids = [map { $_->{vcard_id} } @$shard_rows];
        $log->out({ shard => $shard_num, message => "processing vcards", data => $vcard_ids });
        my $resync_data = get_all_sql(PPC(shard => $shard_num), [
                "SELECT b.cid, b.bid, ? AS priority
                FROM vcards v
                    JOIN campaigns c ON c.cid = v.cid
                    JOIN banners b ON b.cid = v.cid AND b.vcard_id = v.vcard_id",
                WHERE => {
                    'v.vcard_id' => $vcard_ids,
                    'c.statusActive' => "Yes",
                    'b.statusActive' => "Yes",
                },
            ], RESYNC_PRIORITY);
        $log->out({ shard => $shard_num, message => "fetched data to reresync", data => $resync_data });

        my $resend_cnt = bs_resync($resync_data);
        $log->out({ shard => $shard_num, message => "added $resend_cnt banners to BS resync queue" });
    });
}

$log->out('FINISH');
