#!/usr/bin/perl

=head1 DEPLOY

# approved by ppalex
# .migr
{
  type => 'script',
  when => 'after',
  time_estimate => "не более 5 минут",
}

=cut

use my_inc '..';
use Direct::Modern;

use Yandex::DBTools;
use Yandex::DateTime;

use JSON;
use Client;

use ShardingTools;
use ScriptHelper;
use Settings;

use List::MoreUtils qw/uniq/;

my $BREAK_START = "2017-08-15 17:20:00";
my $BREAK_END = "2017-08-17 12:12:59";
my $EVENT_TYPES = [3, 15];
my $MAX_SUM_DIFF = 1e-2;


$log->out('START');

for my $shard (ppc_shards()) {
    my $guard = $log->msg_prefix_guard("[SHARD_$shard]");
    $log->out('Start processing shard');

    my $events = get_all_sql(PPC(shard => $shard),
        ["SELECT id, eventtime, cid, params
            FROM eventlog",
           WHERE => {
                type => $EVENT_TYPES,
                eventtime__between => [$BREAK_START, $BREAK_END],
            }]);
    my %cids = map {$_->{cid} => undef} @$events;

    my $cid2client = get_hash_sql(PPC(shard => $shard),
        ["SELECT cid, IF(agencyid = 0, clientid, agencyid)
            FROM campaigns",
           WHERE => {
                cid => [keys %cids],
            }]);
    my @client_ids = uniq values %$cid2client;
    my $clientid2nds = mass_get_client_NDS(\@client_ids);

    my $log_entries = get_all_sql(PPCLOG,
        ["SELECT cid, logtime, sum_delta, currency
            FROM logbalance",
           WHERE => {
                sum_delta__gt => 0,
                logtime__between => [$BREAK_START, $BREAK_END],
                cid => [keys %cids],
            }]);

    my %cid2entries;
    for my $entry (@$log_entries) {
        push @{ $cid2entries{$entry->{cid}} }, $entry;
    }

    my %update_data;
    for my $event (@$events) {
        my $old_param = from_json($event->{params});
        my $nds = $clientid2nds->{$cid2client->{$event->{cid}}};
        my $log_entry = get_closest_log_entry($event->{eventtime}, $old_param->{sum_payed}, $nds, $cid2entries{$event->{cid}});
        if (!defined $log_entry) {
            $log->out({event => $event, error => "No log entry found"});
            next;
        }

        if (abs($old_param->{sum_payed} - $log_entry->{sum_delta}) < $MAX_SUM_DIFF) {
            $log->out({event => $event, log_entry => $log_entry, error => "Sum is already fixed"});
            next;
        }

        my %new_param = (
            currency => $log_entry->{currency},
            sum_payed => $log_entry->{sum_delta}
        );
        $log->out({event => $event, log_entry => $log_entry, new_param => \%new_param});
        $update_data{$event->{id}} = {params => to_json(\%new_param)};
    }

    my $num = scalar keys %update_data;
    $log->out("Going to update $num entries");

    if (%update_data) {
        $log->out('Performing update');
        do_mass_update_sql(PPC(shard => $shard), 'eventlog', 'id', \%update_data);
    }
}

$log->out('FINISH');

sub get_closest_log_entry {
    my ($eventtime, $sum_payed, $nds, $log_entries) = @_;

    my $log_entry;
    my $current_td;
    my @times;
    for my $entry (@$log_entries) {
        if (!can_be_same_sum($entry->{sum_delta}, $sum_payed, $nds // 0)) {
            next;
        }

        my $timedelta = get_delta($eventtime, $entry->{logtime});
        push @times, $timedelta;

        if ($timedelta == 0) {
            $log_entry = $entry;
            last;
        } elsif (!defined $log_entry || $current_td > $timedelta) {
            $log_entry = $entry;
            $current_td = $timedelta;
        }
    }

    @times = sort {$a <=> $b} grep {$_ <= 5 && $_ >= 0} @times;
    return $log_entry;
}

sub can_be_same_sum {
    my ($sum_log, $sum_event, $nds) = @_;

    return abs($sum_event - $sum_log / (1 + $nds/100)) <= $MAX_SUM_DIFF ? 1 : 0;
}

sub get_delta {
    my ($d1, $d2) = @_;
    if ($d1 eq $d2) {
        return 0;
    }

    my $dt1 = datetime($d1);
    my $dt2 = datetime($d2);
    return abs($dt1->epoch() - $dt2->epoch());
}
