#!/usr/bin/perl

=head1 DEPLOY

# approved by hrustyashko
# .migr
{
  type => 'script',
  when => 'after',
  time_estimate => 'примерно 1 час',
  comment => ''
}

=cut

use Direct::Modern;

use my_inc '..';

use BS::ResyncQueue qw/ bs_resync_camps /;
use Settings;
use ScriptHelper;

use Yandex::DBShards qw/ sharded_chunks /;
use Yandex::DBTools qw/ get_one_column_sql do_update_table /;
use Yandex::ListUtils qw/ nsort /;
use Yandex::Retry qw/ relaxed /;

my $CHUNK_SIZE = 1000;

my $SLEEP_COEF = 0.5;
my $VERBOSE    = 0;
my $DRYRUN     = 0;
my $SHARD      = undef;

extract_script_params(
    'sleep-coef=f' => \$SLEEP_COEF,
    'verbose'      => \$VERBOSE,
    'dry'          => \$DRYRUN,
    'shard=i'      => \$SHARD,    
);



$log->out('START');

my $campaign_cids = get_incorrect_campaign_cids( $SHARD );

my $chunk_count             = 0;
my $total_updated_campaigns = 0;

my @chunks = sharded_chunks( cid => $campaign_cids, chunk_size => $CHUNK_SIZE, with_undef_shard => 1 );
for my $chunk ( @chunks ) {
    $chunk_count++;

    my $shard = $chunk->{shard} // -1;
    my $cids  = [ nsort @{ $chunk->{cid} } ];

    $log->out( sprintf( 'chunk %s, shard %s, size %s' => $chunk_count, $shard, scalar @$cids ) );

    unless ( $shard ) {
        $log->out( sprintf( "can't guess shard for cids %s - clients might be under resharding, skip" => join( ', ' => @$cids ) ) );
        next;
    }

    if ( $shard == -1 ) {
        $log->out( sprintf( "can't guess shard for cids %s - no entries in ppcdict, skip" => join( ', ' => @$cids ) ) );
        next;
    }

    my $updated_cids = switch_off_adf_for_campaings( $shard, $cids );

    $total_updated_campaigns += scalar( @$updated_cids );

    if ( scalar( @$updated_cids ) ) {
        resend_to_bs( $shard, $updated_cids );
    }
}

$log->out( sprintf( 'adf disabled for %s campaigns' => $total_updated_campaigns ) );

$log->out('FINISH');



sub get_incorrect_campaign_cids {
    my ( $shard ) = @_;

    $shard //= 'all';

    $log->out( sprintf( 'looking for incorrect campaigns in %s', ( $shard ne 'all' ) ? "$shard shard" : 'all shards' ) );

    my $cids = get_one_column_sql( PPC(shard => $shard), [
        'SELECT cid FROM camp_options',
        WHERE => { broad_match_flag => 'Yes', is_search_stop => 1 }
    ] );

    $log->out( sprintf( '%s incorrect campaigns found', scalar( @$cids ) ) );

    return $cids;
}

sub switch_off_adf_for_campaings {
    my ( $shard, $cids ) = @_;

    die "No shard given!\n" unless $shard;

    $log->out( sprintf( 'switch off adf for campaings: %s', join( ', ' => @$cids ) ) );

    my $to_update_cnt = scalar( @$cids );

    my $updated_rows = 0;
    my $updated_cnt  = 0;
    my $updated_cids;
    unless ( $DRYRUN ) {
        relaxed times => $SLEEP_COEF, sub {
            $updated_rows += do_update_table( PPC( shard => $shard ), 'campaigns c JOIN camp_options co USING (cid)',
                { 'co.broad_match_flag' => 'No', 'c.LastChange__dont_quote' => 'NOW()' },
                where => { 'c.cid' => $cids, 'co.broad_match_flag' => 'Yes', 'co.is_search_stop' => 1 });
        };

        $log->out( sprintf( '%s rows updated in two tables', $updated_rows ) );

        $updated_cids = get_one_column_sql( PPC( shard => $shard ), [
            'SELECT cid FROM camp_options',
            WHERE => { cid => $cids, broad_match_flag => 'No', is_search_stop => 1 }
        ] );

        $updated_cnt = scalar( @$updated_cids );

        if ( $to_update_cnt != $updated_cnt ) {
            $log->out( sprintf( 'ATTENTION: number of updated campaigns %s not equal to number of campaigns planned to update %s', $updated_cnt, $to_update_cnt ) );
        }
    }

    $log->out( sprintf( '%s campaigns updated' => $updated_cnt ) );

    return $updated_cids;
}

sub resend_to_bs {
    my ( $shard, $cids ) = @_;

    die "No shard given!\n" unless $shard;

    $log->out( 'put campaigns to bs resync queue' );
    
    my $not_archived_cids = get_one_column_sql( PPC(shard => $shard),
        ['SELECT cid FROM campaigns', WHERE => { cid => $cids, archived => 'No' }]
    );

    if ( scalar( @$cids ) != scalar( @$not_archived_cids ) ) {
        my %all = map { $_ => undef } @$cids;
        delete @all{ @$not_archived_cids };
        $log->out( sprintf( 'filter out archived campaigns: %s', join( ', ' => keys %all ) ) );
    }

    if ( ! scalar( @$not_archived_cids ) ) {
        $log->out( 'nothing to put to bs resync queue' );
        return;
    }

    unless ( $DRYRUN ) {
        bs_resync_camps( $not_archived_cids );

        $log->out( sprintf( 'campaigns put to bs resync queue: %s', join( ', ' => @$not_archived_cids ) ) );
    }

    return;
}
