#!/usr/bin/perl

=head1 DEPLOY

# approved by liosha
# .migr
{
  type => 'script',
  when => 'after',
  time_estimate => "на devtest примерно 10 часов",
  comment => "перебирает все записи banners и sitelinks_links и приписывает к href протокол http://, если его нет
    скрипт можно прибивать и перезапускать, а также указывать ограничение на количество обрабатываемых строк.

    Скрипт генерирует очень много логов, поэтому лучше запускать его с ограничением на количество обрабатываемых строк:
        ./deploy/20170928_fix_banners_sitelinks_hrefs.pl --row-limit=10000000
    После того как эта итерация отработает, можно посмотреть, нужно ли снова ее запускать, для этого нужно
    сделать tail -n 1 на логи скрипта по всем шардам, например на бете это получается сделать так
        tail -n 1 protected/logs/20170928_fix_banners_sitelinks_hrefs_shard*
    Если у всех шардов строчки содержат 'finished processing sitelinks_links (no more rows to process)', значит больше
    скрипт запускать не нужно.
    Если у каких-то шардов строчки будут содержать 'finished processing sitelinks_links (row_limit was hit)', значит
    нужна еще одна итерация.
    Если у каких-то шардов строчки не будут содержать 'finished processing', нужно сообщить разработчику и дальше
    действовать по ситуации.
    Итераций потребуется 4 или 5. Сложно определить точно, т.к. в YT нельзя посчитать по шардам

    После каждой итерации нужно пожать получившиеся логи, и увезти их на ppcdev4:/tmp/andreymak/href_fixes_logs/
  "
}

=head1 DESCRIPTION

    Миграция для проставления отсутствуеющего протокола в banners.href и sitelinks_links.href. Перебирает все
    записи фулсканом, не больше чем о 100_000 за раз, выбирает из них те, у которых в ссылке нет протокола, и
    приписывает им http:// в начало.

    Скрипт обрабатывает все шарды параллельно силами foreach_shard_parallel_verbose
    В каждом шарде сначала обрабатывается вся таблица banners, а потом вся таблица sitelinks_links.

    По-умолчанию в каждой итерации чтения таблицы обрабатывается 1_000 записей за раз, после чего итерация спит с
    коэффициентом 1. Каждые 50 итераций, или в конце работы, последний обработанный ID записывается в ppc_properties
    в записи под именем FIX_BANNERS_SITELINKS_${table_name}_SHARD_${shard}_MIN_ID

    Работой скрипта можно управлять следующими именованными параметрами:
        --shard-id=7    - обрабатывать только указанный шард. Иначе - все шарды обрабатывать параллельно
        --row-limit=10  - в каждой таблице обрабатывать не более указанного количества строк. Считаются только те
                          строки, у которых не был выставлен протокол. По-умолчанию ограничения нет
        --chunk-size=5  - в каждой итерации обрабатывать не более указанного количества строк. По-умолчанию 1_000
        --no-banners    - не обрабатывать таблицу banners
        --no-sitelinks  - не обрабатывать таблицу sitelinks_links

    На бетах также можно указать диапазон banners.bid или sitelinks_links.sl_id, которые нужно обрабатывать.
    Если указать диапазон из banners но не указать диапазон из sitelinks_links, таблица с сайтлинками не будет
    обрабатываться.
    Также при указании диапазона ID выключается механизм продолжения работы после перезапуска скрипта (не сохраняется
    последний обработанный ID)
        --min-bid=1234      - при обработке banners, читать строки начиная с указанного bid
        --max-bid=1235      - при обработке banners, читать строки до указанного bid включительно
        --min-sl-id=1234    - при обработке sitelinks_links, читать строки начиная с указанного sl_id
        --max-sl-id=1235    - при обработке sitelinks_links, читать строки до указанного sl_id включительно

=head1 USAGE

    # обработать по 10_000_000 строк в каждой таблице в каждом шарде и выйти
    ./deploy/20170928_fix_banners_sitelinks_hrefs.pl --row-limit=10000000

    # обработать баннеры из диапазона 123456 - 123506 включительно на шарде 7 с выводом логов на экран
    LOG_TEE=1 ./deploy/20170928_fix_banners_sitelinks_hrefs.pl --min-bid=123456 --max-bid=123506 --shard-id=7

=cut

use Direct::Modern;

use JSON;

use Yandex::DBTools;
use Yandex::DBShards;
use Yandex::Log;
use Yandex::Retry qw/relaxed_guard/;

use my_inc '..';

use EnvTools;
use Property;
use ScriptHelper;
use ShardingTools qw/foreach_shard_parallel_verbose/;
use Settings;

my $SLEEP_COEF = 1;
my $DEFAULT_CHUNK_SIZE = 1_000;
my $MAX_ROWS_SCAN = 100_000;
my $MAX_SL_ID = 165_433_713; # https://yql.yandex-team.ru/Operations/WSyOKgcJUf3o2Oe8MtfTS6fP3C5mrXllJ4jdZ-J7XTw=
my $ITERATIONS_PER_CHECKPOINT = 50;

my $chunk_size = $DEFAULT_CHUNK_SIZE;
my $process_banners = 1;
my $process_sitelinks = 1;

# опции только для бет
my %BETA_OPTIONS = (
    'min-bid=i' => \my $min_bid,
    'max-bid=i' => \my $max_bid,
    'min-sl-id=i' => \my $min_sl_id,
    'max-sl-id=i' => \my $max_sl_id,
);

extract_script_params(
    'shard-id=i' => \my $only_shard_id,
    'row-limit=i' => \my $row_limit,
    'chunk-size=i' => \$chunk_size,
    'banners!' => \$process_banners,
    'sitelinks!' => \$process_sitelinks,
    %BETA_OPTIONS,
);

if (!is_beta()) {
    for my $key (keys %BETA_OPTIONS) {
        my $val = ${$BETA_OPTIONS{$key}};
        my $opt = '--'.(split(/=/, $key))[0];
        if ($val) {
            die "$opt option is beta only!";
        }
    }
}

my $have_banners_range = defined($min_bid) || defined($max_bid);
my $have_sitelinks_range = defined($min_sl_id) || defined($max_sl_id);
if ($have_banners_range && !$have_sitelinks_range) {
    $process_sitelinks = 0;
}
if ($have_sitelinks_range && !$have_banners_range) {
    $process_banners = 0;
}

$log->out('START');

foreach_shard_parallel_verbose($log, sub {
    my $shard = shift;

    if (defined($only_shard_id) && $only_shard_id != $shard) {
        return;
    }

    my $log = Yandex::Log->new(
            date_suf => "%Y%m%d", auto_rotate => 1, tee => $ENV{LOG_TEE},
            log_file_name => get_script_name(shardid => $shard),
            msg_prefix => "[shard $shard]",
        );

    my %base_params = (
        ($row_limit ? (row_limit => $row_limit) : ()),
        chunk_size => $chunk_size,
    );

    if ($process_banners) {
        my %params = (
            %base_params,
            query_cond => "banner_type != 'mobile_content'",
            has_last_change => 1,
            min_id => $min_bid,
            max_id => $max_bid,
            with_checkpoints => int(!$have_banners_range),
        );
        fix_table($log, $shard, 'banners', 'bid', %params);
    }

    if ($process_sitelinks) {
        my %params = (
            %base_params,
            min_id => $min_sl_id,
            max_id => $max_sl_id // $MAX_SL_ID,
            with_checkpoints => int(!$have_sitelinks_range),
        );
        fix_table($log, $shard, 'sitelinks_links', 'sl_id', %params);
    }
});

$log->out('FINISH');


sub fix_table {
    my ($log, $shard, $table_name, $primary_key, %params) = @_;

    $params{min_id} //= 0;
    $params{max_id} //= get_one_field_sql(PPC(shard => $shard), "SELECT MAX($primary_key) FROM $table_name");

    my $min_id_prop = Property->new(get_min_id_prop_name($table_name, $shard));
    my $continuation_note;
    if ($params{with_checkpoints} && $min_id_prop->get()) {
        $params{min_id} = $min_id_prop->get();
        $continuation_note = "continuing from ".$params{min_id};
    }

    $log->out("start processing $table_name, params: ".to_json(\%params, {canonical => 1}));
    if ($continuation_note) {
        $log->out($continuation_note);
    }

    my ($min_id, $max_id, $chunk_size, $row_limit) = @params{qw/min_id max_id chunk_size row_limit/};

    my $iterations = 0;
    my $row_limit_hit = 0;
    while ($min_id <= $max_id) {
        my $relaxed_guard = relaxed_guard(times => $SLEEP_COEF);

        ++$iterations;
        my $next_min_id = $min_id + $MAX_ROWS_SCAN;
        if ($next_min_id > $max_id) {
            $next_min_id = $max_id + 1;
        }

        $log->out("scanning rows with ids [$min_id - ".($next_min_id - 1)."]");

        my $fetch_limit = $chunk_size;
        if ($row_limit && $fetch_limit > $row_limit) {
            $fetch_limit = $row_limit;
        }

        my $rows = get_all_sql(PPC(shard => $shard), "
                SELECT $primary_key, href
                FROM $table_name
                WHERE $primary_key >= $min_id AND $primary_key < $next_min_id AND
                    href IS NOT NULL AND
                    LOWER(href) NOT LIKE 'http://%' AND
                    LOWER(href) NOT LIKE 'https://%' AND
                    ".($params{query_cond} ? $params{query_cond} : 1)."
                ORDER BY $primary_key
                LIMIT $fetch_limit
            ");
        if (@$rows == $fetch_limit) {
            $next_min_id = $rows->[-1]{$primary_key} + 1;
        }
        $min_id = $next_min_id;

        next unless @$rows;

        $log->out($rows);
        $log->out("last fetched id: ".$rows->[-1]{$primary_key});

        my @ids = map { $_->{$primary_key} } @$rows;
        my $updated = 0 + do_update_table(PPC(shard => $shard), $table_name,
                {
                    href__dont_quote => "CONCAT('http://', href)",
                    ($params{has_last_change} ? (LastChange__dont_quote => 'LastChange') : ()),
                },
                where => [
                    $primary_key => \@ids,
                    href__is_not_null => 1,
                    _TEXT => "LOWER(href) NOT LIKE 'http://%'",
                    _TEXT => "LOWER(href) NOT LIKE 'https://%'",
                ],
            );
        $log->out("updated $updated rows");

        if ($row_limit) {
            $row_limit -= $updated;
            if ($row_limit <= 0) {
                $row_limit_hit = 1;
                last;
            }
        }

        if ($params{with_checkpoints} &&
            ($iterations % $ITERATIONS_PER_CHECKPOINT == 0)
        ) {
            $min_id_prop->set($min_id);
        }
    }

    if ($params{with_checkpoints}) {
        $min_id_prop->set($min_id);
    }

    if ($row_limit_hit) {
        $log->out("finished processing $table_name (row_limit was hit)");
    } else {
        $log->out("finished processing $table_name (no more rows to process)");
    }
}

sub get_min_id_prop_name {
    my ($table_name, $shard) = @_;

    return "FIX_BANNERS_SITELINKS_${table_name}_SHARD_${shard}_MIN_ID";
}
