#!/usr/bin/perl

=head1 DEPLOY

# .migr
{
  tasks => [
    {
      type => 'sql',
      db => 'ppc:all',
      when => 'before',
      time_estimate => 'меньше секунды',
      sql => "ALTER TABLE `manager_hierarchy`
            ADD COLUMN `chiefs_client_id` JSON DEFAULT NULL AFTER `supervisor_uid`,
            ADD COLUMN `chiefs_uid` JSON DEFAULT NULL AFTER `chiefs_client_id`,
            ADD COLUMN `subordinates_client_id` JSON DEFAULT NULL AFTER `chiefs_uid`,
            ADD COLUMN `subordinates_uid` JSON DEFAULT NULL AFTER `subordinates_client_id`"
    },
    {
      type => 'script',
      when => 'after',
      time_estimate => '30 sec',
      comment => 'заполняет новые поля в manager_hierarchy'
    }
  ],
  approved_by => 'zhur'
}

=cut

use Direct::Modern;

use JSON qw/to_json/;

use my_inc '..';

use ScriptHelper;
use Settings;

use Yandex::DBShards qw/ sharded_chunks /;
use Yandex::DBTools qw/ get_hashes_hash_sql do_mass_update_sql sql_case /;
use Yandex::ListUtils qw/ nsort /;
use Yandex::Retry qw/ relaxed /;


my $CHUNK_SIZE = 1_000;

my $SLEEP_COEF = 0.5;
my $DRYRUN     = 0;

extract_script_params(
    'sleep-coef=f' => \$SLEEP_COEF,
    'dry'          => \$DRYRUN,
);

$log->out('START');

$log->out('seek for all managers ...');

my $managers = get_hashes_hash_sql( PPC(shard =>'all'),
    'SELECT
        manager_client_id,
        manager_uid,
        supervisor_client_id,
        supervisor_uid,
        subrole
    FROM
        manager_hierarchy
            JOIN clients ON (manager_client_id = ClientId AND role = "manager")');

$log->out( sprintf( '%s managers found', scalar keys %$managers ) );


$log->out('gather all chiefs and subordinates for each manager ...');

my %chiefs_client_id;
my %chiefs_uid;
my %subordinates_client_id;
my %subordinates_uid;
for my $manager_client_id ( keys %$managers ) {

    my $manager    = $managers->{ $manager_client_id };
    my $supervisor = $managers->{ $manager->{supervisor_client_id} };
    
    while ( $supervisor ) {
        my $supervisor_client_id = $supervisor->{manager_client_id};

        # добавляем текущему менеджеру шефа
        push @{ $chiefs_client_id{ $manager_client_id } //= [] }, $supervisor_client_id;
        push @{ $chiefs_uid{ $manager_client_id } //= [] }, $supervisor->{manager_uid};

        # добавляем шефу текущего менеджера
        push @{ $subordinates_client_id{ $supervisor_client_id } //= [] }, $manager_client_id;
        push @{ $subordinates_uid{ $supervisor_client_id } //= [] }, $manager->{manager_uid};

        $supervisor = $managers->{ $supervisor->{supervisor_client_id} };
    }
}

$log->out('done');


$log->out('update managers in db');

my $chunk_count   = 0;
my $total_updated = 0;

my @chunks = sharded_chunks( ClientID => [ keys %$managers ], chunk_size => $CHUNK_SIZE, with_undef_shard => 1 );
for my $chunk ( @chunks ) {
    $chunk_count++;

    my $shard      = $chunk->{shard} // -1;
    my @client_ids = nsort @{ $chunk->{ClientID} };

    $log->out( sprintf( "\twork on chunk %s, shard %s, size %s" => $chunk_count, $shard, scalar @client_ids ) );

    unless ( $shard ) {
        $log->out( sprintf( "\tCan't guess shard for ClientID's %s - managers might be under resharding, skip" => join( ', ' => @client_ids ) ) );
        next;
    }

    if ( $shard == -1 ) {
        $log->out( sprintf( "\tCan't guess shard for ClientID's %s - no entries in ppcdict, skip" => join( ', ' => @client_ids ) ) );
        next;
    }
    
    my %values;
    for my $client_id ( @client_ids ) {
        my %client_vals;

        my $chiefs_client_id = $chiefs_client_id{ $client_id };
        if ( $chiefs_client_id ) {
            $client_vals{chiefs_client_id} = to_json([ map { int } @$chiefs_client_id ]);
        }

        my $chiefs_uid = $chiefs_uid{ $client_id };
        if ( $chiefs_uid ) {
            $client_vals{chiefs_uid} = to_json([ map { int } @$chiefs_uid ]);
        }

        my $subordinates_client_id = $subordinates_client_id{ $client_id };
        if ( $subordinates_client_id ) {
            $client_vals{subordinates_client_id} = to_json([ map { int } @$subordinates_client_id ]);
        }

        my $subordinates_uid = $subordinates_uid{ $client_id };
        if ( $subordinates_uid ) {
            $client_vals{subordinates_uid} = to_json([ map { int } @$subordinates_uid ]);
        }

        $values{ $client_id } = \%client_vals if %client_vals;
    }

    my $updated = 0;
    unless ( $DRYRUN ) {
        relaxed times => $SLEEP_COEF, sub {
            $updated += do_mass_update_sql( PPC( shard => $shard ), 'manager_hierarchy', 'manager_client_id', \%values );
        };

        if ( scalar @client_ids != $updated ) {
            $log->out( sprintf( "\tNumber of updated rows %s not equal to number of rows planned for update %s", $updated, scalar @client_ids ) );
        }
    }

    $log->out( sprintf( "\t%s managers updated" => $updated ) );

    $total_updated += $updated;
}

$log->out( sprintf( 'total %s managers updated' => $total_updated ) );

$log->out('FINISH');
