#!/usr/bin/perl

=head1 DEPLOY

# .migr
{
  tasks => [
    {
      type => 'sql',
      db => 'ppc:all',
      when => 'before',
      time_estimate => '10 sec',
      sql => "CREATE TABLE `agency_managers`(
                `agency_client_id` int(10) unsigned NOT NULL,
                `manager_client_id` int(10) unsigned NOT NULL,
                `manager_uid` bigint(20) NOT NULL,
                PRIMARY KEY (`agency_client_id`, `manager_client_id`),
                INDEX(`manager_client_id`),
                INDEX(`manager_uid`)
              )"
    },
    {
      type => 'script',
      when => 'after',
      time_estimate => '15 sec',
      comment => 'заполняет новую таблицу agency_managers'
    }
  ],
  approved_by => 'zhur'
}

=cut

use Direct::Modern;

use my_inc '..';

use List::MoreUtils qw/ uniq /;

use PrimitivesIds qw/ get_clientids get_key2clientid /;

use RBAC2::Extended;
use RBACDirect qw/ rbac_get_all_agencies rbac_get_all_managers_of_agencies_clientids /;

use ScriptHelper;
use Settings;

use Yandex::DBShards qw/ sharded_chunks /;
use Yandex::DBTools qw/ do_mass_insert_sql /;
#use Yandex::ListUtils qw/ nsort /;

my $CHUNK_SIZE = 1_000;

my $SLEEP_COEF = 0.5;
my $DRYRUN     = 0;
my $AGENCY_UID = undef;

extract_script_params(
    'sleep-coef=f' => \$SLEEP_COEF,
    'dry'          => \$DRYRUN,
    'agencyuid'    => \$AGENCY_UID,
);

$log->out('START');

my $rbac = RBAC2::Extended->get_singleton(1);

$log->out('seek for agencies ...');
my $agencies_uid = $AGENCY_UID ? [ $AGENCY_UID ] : rbac_get_all_agencies( $rbac );
my $all_agencies_client_id = get_clientids( uid => \@$agencies_uid );
$log->out( sprintf("%d agencies found", scalar @$all_agencies_client_id) );

my $chunk_count   = 0;
my $total_updated = 0;

for my $chunk ( sharded_chunks(ClientID => $all_agencies_client_id, chunk_size => $CHUNK_SIZE, with_undef_shard => 1 ) ) {

    my $shard              = $chunk->{shard} // -1;
    my @agencies_client_id = sort @{ $chunk->{ClientID} };

    unless ( $shard ) {
        $log->out( sprintf( "Can't guess shard for ClientIDs %s - agencies might be under resharding, skip" => join( ', ' => @agencies_client_id ) ) );
        next;
    }

    if ( $shard == -1 ) {
        $log->out( sprintf( "Can't guess shard for ClientIDs %s - no entries in ppcdict, skip" => join( ', ' => @agencies_client_id ) ) );
        next;
    }

    my $msg_prefix_guard = $log->msg_prefix_guard("[shard $shard]");

    $log->out( sprintf( "\tchunk %s of size %s" => $chunk_count++, scalar @agencies_client_id ) );

    my $managers_uid_for_agency = rbac_get_all_managers_of_agencies_clientids( $rbac, \@agencies_client_id );

    my $manager_client_id_by_uid = get_key2clientid( uid => [ uniq map { @$_ } values %$managers_uid_for_agency ] );

    my @data;
    for my $agency_client_id ( @agencies_client_id ) {

        my $managers_uid = $managers_uid_for_agency->{ $agency_client_id };
        unless ( $managers_uid ) {
            $log->out( sprintf( "No managers found for agency with ClientID %s" => $agency_client_id ) );
            next;
        }

        for my $manager_uid ( @$managers_uid ) {
            my $manager_client_id = $manager_client_id_by_uid->{ $manager_uid };
            unless ( $manager_client_id ) {
                $log->out( sprintf( "No ClienbtID found for manager with uid %s" => $manager_uid ) );
                next;
            }
            
            push @data, [ $agency_client_id, $manager_client_id, $manager_uid ];
        }
    }

    $log->out( sprintf( "\t%s managers for %s agenies found" => scalar( @data ), scalar( @agencies_client_id ) ) );

    $total_updated += do_mass_insert_sql(
        PPC(shard => $shard),
        'insert into `agency_managers` (`agency_client_id`, `manager_client_id`, `manager_uid`) values %s' ,
        \@data,
        { sleep_coef => $SLEEP_COEF }
    );
}

$log->out( sprintf( "%s rows inserted total" => $total_updated ) );

$log->out('FINISH');

