#!/usr/bin/perl

=head1 DEPLOY

# .migr
{
    tasks => [
                {
                    type => 'script',
                    when => 'after',
                    time_estimate => "около 3,5 часов для каждого шарда",
                    comment => "Данные отбираются и удаляются в порядке возрастания bid.
                    Скрипт можно перезапускать.
                    По умолчанию данные обрабатываются с момента, на котором прервалась работа в последнем запуске.
                    Параметры:
                        --shard {номер шарда}           - обработка только одного шарда.
                        --reset                         - обработка начинается с самого начала.
                        --bid bids1 [--bid bids2 ...]   - обработка только переданных баннеров."
                }
    ],
    approved_by => 'gukserg'
}

=cut

use Direct::Modern;

use Yandex::DBTools;
use Yandex::ListUtils qw/chunks/;

use my_inc '..';

use ScriptHelper;
use Settings;
use ShardingTools qw/foreach_shard_parallel_verbose ppc_shards/;
use Models::Banner;

my $SHARD;
my $RESET;
my @ONLY_BIDS;
my $PROP_NAME_SUFFIX = "_LAST_MIN_BID";

extract_script_params(
    "shard=i" => \$SHARD,
    "reset" => \$RESET,
    "bid=i@" => \@ONLY_BIDS,
);

$log->out('START');

my $CHUNK = 10_000;

foreach_shard_parallel_verbose($log, sub {
    my $shard = shift;
    return if defined $SHARD && $SHARD != $shard;

    my $log_prefix_guard = $log->msg_prefix_guard("[shard $shard]:");

    my $min_bid_prop_name = get_script_name(shardid => $shard) . $PROP_NAME_SUFFIX;

    if ($RESET) {
        my $temp_property = Property->new($min_bid_prop_name);
        $temp_property->delete();
    }

    my $last_min_bid_prop = Property->new($min_bid_prop_name);

    my $MAX_BID = get_one_field_sql(PPC(shard => $shard), "SELECT max(bid) FROM banners");
    my $cur_bid = $last_min_bid_prop->get();

    if( !defined($cur_bid) ) {
        $cur_bid = get_one_field_sql(PPC(shard => $shard), "SELECT min(bid) FROM banners");
        $last_min_bid_prop->set($cur_bid);
    }

    $log->out("Processing banners in interval [$cur_bid .. $MAX_BID]") if (!@ONLY_BIDS);

    my $count_del_rows = 0;

    while ($cur_bid <= $MAX_BID) {
        my $from = $cur_bid;
        my $to = $cur_bid + $CHUNK - 1;

        my $log_prefix_guard_with_banners;

        my $sql = "SELECT b.bid, b.cid, b.pid
                        FROM banners b
                        LEFT JOIN phrases p ON b.pid = p.pid";
        my %condition = ('p.pid__is_null' => 1);
        if (@ONLY_BIDS) {
            $log->out("Processing only these banners [" . join(',', @ONLY_BIDS) . "]");
            $condition{'b.bid'} = \@ONLY_BIDS;
        } else {
            $log_prefix_guard_with_banners = $log->msg_prefix_guard("[shard $shard] Progress [$from..$to]:");
            $condition{'b.bid__between'} = [$from, $to];
        }

        my $corrupt_banners = get_all_sql(PPC(shard => $shard), [$sql, where => \%condition]);

        if (@$corrupt_banners) {
            my @bids_to_del;
            my %banners_to_del;
            for my $banner (@$corrupt_banners) {
                push @bids_to_del, $banner->{bid};
                push @{ $banners_to_del{$banner->{cid}} }, { bid => $banner->{bid}, pid => $banner->{pid} };
            }

            my @all_related_tables = (
                                        "banners",
                                        "banners_performance",
                                        "banner_images",
                                        "images",
                                        "banners_additions",
                                        "camp_turbolanding_metrika_counters",
                                        "banner_turbolandings",
                                        "banner_turbolanding_params",
                                        "banner_resources",
                                        "banner_display_hrefs"
                                     );

            my $LIMIT_FOR_LOG = 1_000;

            for my $chunk_bids (chunks(\@bids_to_del, $LIMIT_FOR_LOG)) {
                my @corrupt_banners_info = ();
                for my $table_name (@all_related_tables) {
                    my $sql = "SELECT * FROM $table_name";
                    my $banners_info = get_all_sql(PPC(shard => $shard), [ $sql, where => { bid => $chunk_bids }]);
                    push @corrupt_banners_info, {$table_name => $banners_info} if @$banners_info;
                }

                $log->out({"Bids" => $chunk_bids, "Corrupted banners info: " => \@corrupt_banners_info}) if @corrupt_banners_info;
            }

            for my $cid (keys %banners_to_del) {
                my @delete_banners = @{ $banners_to_del{$cid} };
                my ($del_banners_num, $error) = Models::Banner::delete_banners($cid, \@delete_banners, {skip_checks => 1});

                $log->out("Models::Banner::delete_banners error: $error") if $error;
                $count_del_rows += $del_banners_num;
            }
        }

        last if (@ONLY_BIDS);

        $cur_bid += $CHUNK;
        $last_min_bid_prop->set($cur_bid);
    }

    $log->out("Deleted $count_del_rows rows");
    return 1;
});

if (!@ONLY_BIDS) {
    for my $shard (ppc_shards()) {
        next if (defined $SHARD && $SHARD != $shard);

        my $property_name = get_script_name(shardid => $shard) . $PROP_NAME_SUFFIX;
        my $property = Property->new($property_name);
        my $property_value = $property->get();

        $log->out("Delete unused property value " . $property_name . "=" . $property_value);
        $property->delete();
    }
}
$log->out('FINISH');
