#!/usr/bin/env perl

=head1 DEPLOY

# .migr
{
    approved_by => 'ppalex',
    tasks => [
        {
            type => 'script',
            when => 'after',
            time_estimate => "15 часов на девтесте",
            comment => 'можно перезапускать'
        }
    ]
}

=cut

use Direct::Modern;

use Yandex::Balance;
use Yandex::DBTools;
use Yandex::DBShards;
use Yandex::ListUtils qw/nsort/;

use my_inc '..';

use Campaign qw/send_camp_to_service campaign_manager_changed/;
use Campaign::Types;
use Client qw//;
use Primitives;
use PrimitivesIds;
use RBACElementary;
use RBACDirect;
use ScriptHelper;
use Settings;

use Path::Tiny;

{
    no warnings 'redefine';
    *RBACDirect::log_rbac_update = sub {};
}

$log->out('START');

my ($DRY_RUN, $LIMIT);
extract_script_params(
    'dry-run' => \$DRY_RUN,
    'limit=i' => \$LIMIT,
);

my $type = ['wallet', 'billing_aggregate', @{get_camp_kind_types('web_edit_base')}];

my $count = 0;
my @lines = path(my_inc::path('20190416_mass_change_managers.data'))->lines({chomp => 1});

my ($clientid2manager_login, $manager_login2info);
my (@skipped_clients, @success_clients, @partial_success_clients, @failure_clients);

for my $line (@lines) {
    next unless $line =~ /^\d/;

    my ($clientid, $manager_login, $dont_do_it) = split /\t/, $line;
    if ($dont_do_it) {
        $log->out("[ClientID $clientid]\tSKIPPING: Client has more than one manager (marked in the input)");
        push @skipped_clients, $clientid;
        next;
    }
    $clientid2manager_login->{$clientid} = $manager_login;
    $manager_login2info->{$manager_login} = '';

    if ($LIMIT) {
        $count++;
        last if $count == $LIMIT;
    }
}

# про клиента
my @clientids = nsort keys %$clientid2manager_login;
my $agencies_count = count_agencies(\@clientids);
my $agencies_count2 = Primitives::mass_get_client_first_agency(\@clientids);
my $clientid2chiefuid = rbac_get_chief_reps_of_clients(\@clientids);

# про менеджера
my @manager_logins = keys %$manager_login2info;
my $login2uid = get_login2uid(login => \@manager_logins);
my $login2clientid = get_login2clientid(login => \@manager_logins);
my $uid2balanceinfo = balance_mass_get_managers_info([values %$login2uid], timeout => 10);


for my $clientid (@clientids) {
    my $manager_login = $clientid2manager_login->{$clientid};
    check_manager_login($manager_login);
}

for my $clientid (@clientids) {
    my $g = $log->msg_prefix_guard("[ClientID $clientid]");

    my $manager_login = $clientid2manager_login->{$clientid};
    $log->out("Attempting to transfer the client to the manager $manager_login");

    if (!$clientid2chiefuid->{$clientid}) {
        push @skipped_clients, $clientid;
        $log->out("SKIPPING: Client has no chief uid");
        next;
    }

    if ($agencies_count->{$clientid} || $agencies_count2->{$clientid}) {
        $log->out("SKIPPING: Client is bound to an agency or has agency campaigns");
        push @skipped_clients, $clientid;
        next;
    }

    if ($manager_login2info->{$manager_login} ne 'good') {
        $log->out("SKIPPING: New manager for client was not found in Direct and/or Balance");
        push @skipped_clients, $clientid;
        next;
    }

    my $manageruid = $login2uid->{$manager_login};
    my $cid2manageruid = get_hash_sql(PPC(ClientID => $clientid), ["SELECT cid, manageruid FROM campaigns",
        where => {clientid => $clientid, type => $type, archived => 'No', statusEmpty => 'No', _OR => [manageruid__is_null => 1, manageruid__ne => $manageruid]}]);

    my $manageruid2cids = manageruid2cids($cid2manageruid);
    if ((scalar grep {$_} keys %$manageruid2cids) > 1) {
        push @skipped_clients, $clientid;
        $log->out("SKIPPING: Client has more than one manager");
        next;
    }

    if (!%$manageruid2cids) {
        $log->out("Client has no camps to change manager; binding the manager explicitly");
        if (!$DRY_RUN) {
            Client::bind_manager($clientid, $manageruid);
        }
        push @success_clients, $clientid;
        $log->out("SUCCESS: Client transferred to the manager $manager_login");
        next;
    }

    $log->out({DATA => {manageruid2cids => $manageruid2cids}});

    # закончили проверять, назначаем менеджера
    my %totals = (success => 0, failure => 0);
    for my $old_manager_uid (keys %$manageruid2cids) {
        if ($old_manager_uid == 0) { # приём кампаний на сервисирование
            eval {
                send_to_servicing($manageruid2cids->{$old_manager_uid}->[0], $clientid2chiefuid->{$clientid}, $manageruid);
            };
            if ($@) {
                $log->out("FAILURE: Didn't send camps to servicing, $@");
                $totals{failure}++;
            } else {
                $totals{success}++;
            }
        } elsif ($old_manager_uid != $manageruid) { # перенос кампаний на другого менеджера
            my $failed_cids = change_manager($manageruid2cids->{$old_manager_uid}, $clientid, $old_manager_uid, $manageruid);
            if (@$failed_cids) {
                $log->out("FAILURE: Failed to change a manager for some cids");
                $log->out({DATA => {"failed cids" => $failed_cids}});
                $totals{failure}++;
            }
            if (scalar @$failed_cids < scalar @{$manageruid2cids->{$old_manager_uid}}) { # не всё плохо
                $totals{success}++;
            }
        }
    }
    if ($totals{success}) {
        if ($totals{failure}) {
            push @partial_success_clients, $clientid;
            $log->out("SUCCESS: PARTIAL success");
        } else {
            push @success_clients, $clientid;
            $log->out("SUCCESS: total success");
        }
    } else {
        push @failure_clients, $clientid;
        $log->out("FAILURE: Nothing is changed for a client");
    }
}

# Выводим сводку
$log->out('Successes => ' . (scalar @success_clients) . ", Partial successes => " . (scalar @partial_success_clients) . ", Failures => " . (scalar @failure_clients) . ", Skipped => " . (scalar @skipped_clients));

$log->out('FINISH');

sub count_agencies {
    my $clientids = shift;
    return get_hash_sql(PPC(ClientID => $clientids), ["SELECT clientid, COUNT(DISTINCT agencyid) FROM campaigns",
        where => {clientid => SHARD_IDS, type => $type, agencyid__ne => 0}, 'GROUP BY clientid']);
}

sub manageruid2cids {
    my $cid2manageruid = shift;
    my $result = {};
    for my $cid (keys %$cid2manageruid) {
        if (my $manageruid = $cid2manageruid->{$cid}) {
            push @{$result->{$manageruid}}, $cid;
        } else {
            push @{$result->{0}}, $cid;
        }
    }
    return $result;
}

sub send_to_servicing {
    my ($cid, $chief_uid, $manageruid) = @_;
    $log->out("Sending camps to servicing");
    if (!$DRY_RUN) {
        send_camp_to_service(undef, $cid, $chief_uid, $manageruid, force => 1, send_other_camps => 1);
    }
    $log->out("Camps sent to servicing");
}

sub change_manager {
    my ($cids, $clientid, $old_manager, $new_manager) = @_;
    $log->out("Changing manager from $old_manager to $new_manager");
    my @failed_cids;
    for my $cid (@$cids) {
        next if $DRY_RUN;
        my $error = rbac_change_manager(undef, $old_manager, $new_manager, $cid);
        if ($error) {
            push @failed_cids, $cid;
            next;
        } else {
            campaign_manager_changed(undef, 0, $cid, $new_manager);
            Direct::TurboLandings::mass_refresh_metrika_grants_for_all_client_counters(
                [$clientid], undef
            );
        }
    }
    $log->out("Done changing manager");
    return \@failed_cids;
}

sub check_manager_login {
    my ($manager_login) = @_;
    if (!$manager_login2info->{$manager_login}) {
        my $g = $log->msg_prefix_guard("[Manager $manager_login check]");
        my $manageruid = $login2uid->{$manager_login};
        if (!$manageruid || !$login2clientid->{$manager_login}) {
            $manager_login2info->{$manager_login} = 'bad';
            $log->out("FAILURE: No $manager_login login in Direct");
        } elsif(!defined $uid2balanceinfo->{$manageruid}) {
            $manager_login2info->{$manager_login} = 'bad';
            $log->out("FAILURE: No $manager_login login in Balance");
        } else {
            $manager_login2info->{$manager_login} = 'good';
        }
    }
}
