#!/usr/bin/perl

=head1 DEPLOY

# .migr
{
  tasks => [
    {
      type => 'script',
      when => 'after',
      time_estimate => "около 7 часов",
      comment => "если заметите, что упало, скажите, пожалуйста andreymak@"
    }
  ],
  approved_by => 'aleran'
}

=head1 DESCRIPTION

    Выставляет захардкоженные погодные корректировки захардкоженным клиентам.
    Корректировки выставляются на список групп, вычисляемый в рантайме по условиям:
        _find_groups(login => 'velostrana',
            'c.name__like' => '%ВЕЛОСТРАНА%',
            'p.adgroup_type' => $WEATHER_ADGROUP_TYPES,
        ),
        _find_groups(login => 'pro-vseinstrumenti',
            'c.cid' => [
                37906665,37906021,37905689,37905479,37905225,37904974,37904793,37906582,37905926,37905626,
                37905404,37905160,37904893,37904614,37906356,37904587,37906277,37904576
            ],
            'p.adgroup_type' => $WEATHER_ADGROUP_TYPES,
        ),

    Если на группе уже есть погодная корректировка, она не перезатирается.
    В продакшене скрипт нужно запустить без параметров.

    Для отладки на бете, можно запускать с параметрами
    LOG_TEE=1 ./deploy/20190926_convert-old-weather-bidmodifiers.pl [--client-id=NNNN] [--cid=MMMM] [--undo] [--sleep-coef=YYY]
        --client-id и --cid позволяют ограничить список групп по клиенту и номеру кампании

        --undo  позволяет откатить ранее выставленные миграцией корректировки.
                Айдишники берутся из message лога в кликхаусе с позавчерашней даты, поэтому спешите.

        --sleep-coef    определяет, сколько скрипт спит после обработки каждой группы. По умолчанию 0.2

select cid, count(*) from campaigns c join phrases using(cid) where ClientID = 577312 and name like '%ВЕЛОСТРАНА%' and archived = 'No' and adgroup_type in ('base', 'cpm_banner', 'cpm_video', 'cpm_outdoor', 'performance', 'cpm_audio') and exists (select 1 from hierarchical_multipliers hm where hm.cid = c.cid and hm.pid is not null) group by 1 order by 2  limit 10;

=cut

use Direct::Modern;

use JSON;

use Yandex::DBTools;
use Yandex::Retry;

use my_inc '..';

use HierarchicalMultipliers;
use PrimitivesIds qw/get_clientid/;
use ScriptHelper;
use Settings;
use Tools qw/get_clickhouse_handler/;
use Yandex::Clone qw/yclone/;

# типы групп, где поддерживаются корректировки на погоду
my $WEATHER_ADGROUP_TYPES = [qw/
    base
    performance
    cpm_banner
    cpm_video
    cpm_outdoor
    cpm_audio
/];

# У клиента velostrana одна корректировка на все кампании
my %MULTIPLIERS_BY_LOGIN = (
    velostrana => from_json('
        {
          "conditions": [
            {
              "expression": [
                [{"operation": "ge", "parameter": "temp", "value": 20}],
                [{"operation": "le", "parameter": "temp", "value": 35}]
              ],
              "multiplier_pct": "150"
            },
            {
              "expression": [
                [{"operation": "eq", "parameter": "prec_strength", "value": 0}]
              ],
              "multiplier_pct": "115"
            },
            {
              "expression": [
                [{"operation": "eq", "parameter": "cloudness", "value": 0}, {"operation": "eq", "parameter": "cloudness", "value": 25}]
              ],
              "multiplier_pct": "115"
            }
          ],
          "is_enabled": "1"
        }
    ')
);

my $subzero_template = '
    {
      "conditions": [
        {
          "expression": [
            [{"value": -50, "operation": "ge", "parameter": "temp"}], [{"value": -1, "operation": "le", "parameter": "temp"}]
          ],
          "multiplier_pct": "%d"
        }
      ],
      "is_enabled": "1"
    }
';
my $subzero_150 = from_json(sprintf($subzero_template, 150));
my $subzero_125 = from_json(sprintf($subzero_template, 125));

my $snow_template = '
    {
      "conditions": [
        {
          "expression": [
            [
                {"value": 25, "operation": "eq", "parameter": "prec_strength"},
                {"value": 50, "operation": "eq", "parameter": "prec_strength"},
                {"value": 75, "operation": "eq", "parameter": "prec_strength"},
                {"value": 100, "operation": "eq", "parameter": "prec_strength"}
            ],
            [{"value": -50, "operation": "ge", "parameter": "temp"}],
            [{"value": 0, "operation": "le", "parameter": "temp"}]
          ],
          "multiplier_pct": "%d"
        }
      ],
      "is_enabled": "1"
    }
';
my $snow_150 = from_json(sprintf($snow_template, 150));
my $snow_125 = from_json(sprintf($snow_template, 125));

# клиент pro-vseinstrumenti просит выставить корректировки в этих кампаниях
my %MULTIPLIERS_BY_CAMPAIGN_ID = (
    (map { $_ => $subzero_150 } 37906665, 37906021, 37905689, 37905479, 37905225, 37904974, 37904793),
    (map { $_ => $subzero_125 } 37906582, 37905926, 37905626, 37905404, 37905160, 37904893, 37904614),
    (map { $_ => $snow_150 } 37906356, 37904587),
    (map { $_ => $snow_125 } 37906277, 37904576),
);

my ($ERROR_COUNT, $SUCCESS_COUNT) = (0, 0);

my ($ONLY_CID, $ONLY_CLIENT_ID, $UNDO, $SLEEP_COEF);
extract_script_params(
    'client-id=i' => \$ONLY_CLIENT_ID,
    'cid=i' => \$ONLY_CID,
    'undo' => \$UNDO,
    'sleep-coef=f' => \$SLEEP_COEF,
);
$SLEEP_COEF //= 0.2;

$log->out(
    'START',
    {only_cid => $ONLY_CID, only_client_id => $ONLY_CLIENT_ID, undo => $UNDO, sleep_coef => $SLEEP_COEF}
);

if ($UNDO) {
    _find_inserted_ids();
}

# {login => abcdef, client_id => 321, campaign_id => 456, adgroup_id => 123, campaign_type => 'text'}
my @groups = (
    _find_groups(login => 'velostrana',
        'c.name__like' => '%ВЕЛОСТРАНА%',
        'p.adgroup_type' => $WEATHER_ADGROUP_TYPES,
    ),
    _find_groups(login => 'pro-vseinstrumenti',
        'c.cid' => [
            37906665,37906021,37905689,37905479,37905225,37904974,37904793,37906582,37905926,37905626,
            37905404,37905160,37904893,37904614,37906356,37904587,37906277,37904576
        ],
        'p.adgroup_type' => $WEATHER_ADGROUP_TYPES,
    ),
);
# можем пофильтровать по client-id или cid, указанным в параметрах к скрипту
@groups = _filter_groups(\@groups);

$log->out("found ".scalar(@groups)." groups");

for my $group (@groups) {
    my $relax_guard = relaxed_guard(times => $SLEEP_COEF);

    if ($UNDO) {
        _undo_group($group);
    } else {
        _process_group($group);
    }
}

$log->out({error => $ERROR_COUNT, success => $SUCCESS_COUNT, group_count => scalar(@groups)});
$log->out('FINISH');

=head1 FUNCTIONS

=head2 _find_groups(login => $login, %sql_where)

    Ищет группы в неархивных кампаниях клиента $login,
    и добавляет в WHERE условие из %sql_where

    Возвращает список хешиков вида
    # {login => abcdef, client_id => 321, campaign_id => 456, adgroup_id => 123, campaign_type => 'text'}

=cut
sub _find_groups {
    my %O = @_;

    my $login = delete $O{login};
    my %sql_where = %O;

    my $client_id = get_clientid(login => $login);
    my $rows = get_all_sql(PPC(ClientID => $client_id), ["
        SELECT c.ClientID AS client_id,
            c.cid AS campaign_id,
            c.type AS campaign_type,
            p.pid AS adgroup_id,
            ? AS login
        FROM
            campaigns c
            JOIN phrases p ON p.cid = c.cid",
        WHERE => {
            'c.ClientID' => $client_id,
            'c.archived' => 'No',
            %sql_where
        }
    ], $login);

    return @$rows;
}

=head2 _filter_groups(\@groups)

    Фильтрует группы по $ONLY_CLIENT_ID и $ONLY_CID.
    Возвращает список

=cut
sub _filter_groups {
    my ($groups) = @_;

    if ($ONLY_CLIENT_ID) {
        $log->out("working only with client $ONLY_CLIENT_ID");
    }
    if ($ONLY_CID) {
        $log->out("working only with campaign $ONLY_CID");
    }

    return grep {
        ($ONLY_CLIENT_ID ? $_->{client_id} == $ONLY_CLIENT_ID : 1) &&
        ($ONLY_CID ? $_->{campaign_id} == $ONLY_CID : 1)
    } @$groups;
}

=head2 _process_group(\%group)

    Основная обработка

    Читает корректировки на группе, и если там ещё нет погодной, добавляет ту,
    которая указана в $MULTIPLIERS_BY_CAMPAIGN_ID{$group->{campaign_id}} или $MULTIPLIERS_BY_LOGIN{$group->{login}};

    После добавления перечитывает корректировки на группе, чтобы записать новый айдишник в базу,
    и чтобы проверить, что добавилось именно то, что мы ожидали, и ничего не удалилось.

=cut
sub _process_group {
    # {login => abcdef, client_id => 321, campaign_id => 456, adgroup_id => 123, campaign_type => 'text'}
    my ($group) = @_;
    my $prefix_guard = $log->msg_prefix_guard('["adgroup_id":'.$group->{adgroup_id}.']');

    my $old_modifiers = _get_group_modifiers($group);
    $log->out({old_modifiers => $old_modifiers});

    if ($old_modifiers->{weather_multiplier}) {
        $log->out("group already has weather multiplier, skipping");
        return;
    }

    my $weather_multiplier = $MULTIPLIERS_BY_CAMPAIGN_ID{$group->{campaign_id}} // $MULTIPLIERS_BY_LOGIN{$group->{login}};
    if (!$weather_multiplier) {
        $log->die("can't find multiplier for login ".$group->{login}.", campaign_id ".$group->{campaign_id});
    }
    my %new_modifiers = (
        weather_multiplier => $weather_multiplier,
        %$old_modifiers,
    );
    $log->out({new_modifiers => \%new_modifiers});
    _update_group_modifiers($group, \%new_modifiers);

    my $updated_modifiers = _get_group_modifiers($group);

    _check_modifiers_equal($updated_modifiers, \%new_modifiers);

    if ($updated_modifiers->{weather_multiplier}) {
        $log->out({inserted_modifier_id => $updated_modifiers->{weather_multiplier}{hierarchical_multiplier_id}});
    }
    ++$SUCCESS_COUNT;
}

=head2 _update_group_modifiers(\%group, \%new_modifiers)

    Выставляет корректировки на группу.
    В %new_modifiers должны быть все корректировки, не только погодные.

=cut
sub _update_group_modifiers {
    my ($group, $new_modifiers) = @_;

    HierarchicalMultipliers::save_hierarchical_multipliers(
        $group->{campaign_id},
        $group->{adgroup_id},
        $new_modifiers
    );
    return 1;
}

=head2 _get_group_modifiers(\%group)

    Читает все корректировки группы.
    Возвращает HashRef

=cut
sub _get_group_modifiers {
    my ($group) = @_;

    return HierarchicalMultipliers::get_hierarchical_multipliers(
        $group->{campaign_id},
        $group->{adgroup_id}
    );
}

=head2 _check_modifiers_equal(\%updated_modifiers, \%expected_modifiers)

    Проверяет, что корректировки %updated_modifiers, прочитанные из базы, совпадают с ожидаемыми.
    Над погодной при этом производятся манипуляции: удаляются айдишники и last_change,
    но манипуляции производятся на склонированном объекте.

    Если есть несовпадение, об этом пишется в лог и увеличивается счётчик $ERROR_COUNT

=cut
sub _check_modifiers_equal {
    my ($updated_modifiers, $expected_modifiers) = @_;

    my $patched_updated_modifiers = yclone($updated_modifiers);
    if ($patched_updated_modifiers->{weather_multiplier}) {
        my $wm = $patched_updated_modifiers->{weather_multiplier};
        delete $wm->{hierarchical_multiplier_id};
        delete $wm->{last_change};
        for (@{$wm->{conditions}}) {
            delete $_->{weather_multiplier_value_id};
        }
    }
    my $expected_modifiers_json = to_json($expected_modifiers, {canonical => 1});
    my $actual_modifiers_json = to_json($patched_updated_modifiers, {canonical => 1});

    if ($actual_modifiers_json ne $expected_modifiers_json) {
        $log->out(
            "ERROR: actual new modifiers don't match expected: $actual_modifiers_json ne $expected_modifiers_json"
        );
        ++$ERROR_COUNT;
    }
}

#============= ниже поддерка UNDO, можно отпилить при желании


# adgroup_id => { <modifier_id_1> => 1, <modifier_id_2> => 1 }
my %INSERTED_MODIFIER_IDS;
=head2 _find_inserted_ids()

    Заполняет хеш %INSERTED_MODIFIER_IDS из лога миграции из кликхауса.
    Умеет определять, что добавленную корректировку уже откатили ранее.

=cut
sub _find_inserted_ids {
    my $clh = get_clickhouse_handler('logs');
    $clh->query_format('JSON');

    my $rows = $clh->query("
            SELECT log_time, prefix, message
            FROM messages
            PREWHERE
                log_date >= yesterday()
                AND service = 'direct.script'
                AND method = '20190926_convert-old-weather-bidmodifiers'
            WHERE
                prefix LIKE '%adgroup_id%'
                AND (message LIKE '%inserted_modifier_id%' OR message LIKE '%removed_modifier_id%')
            ORDER BY log_time
        ")->json()->{data};

    my ($found_inserted, $found_undone) = (0, 0);
    for my $row (@$rows) {
        my $group = from_json('{'.$row->{prefix}.'}');
        my $msg = from_json($row->{message});

        my $imids = $INSERTED_MODIFIER_IDS{$group->{adgroup_id}} //= {};
        if ($msg->{inserted_modifier_id}) {
            $imids->{$msg->{inserted_modifier_id}} = 1;
            ++$found_inserted;
        }

        if ($msg->{removed_modifier_id} && $imids->{$msg->{removed_modifier_id}}) {
            delete $imids->{$msg->{removed_modifier_id}};
            ++$found_undone;
        }
    }

    $log->out("_find_inserted_ids: found $found_inserted inserted, $found_undone undone");
}

=head2 _undo_group(\%group)

    Удаляет погодную корректировку, если её айдишник найден в %INSERTED_MODIFIER_IDS

=cut
sub _undo_group {
    # {login => abcdef, client_id => 321, campaign_id => 456, adgroup_id => 123, campaign_type => 'text'}
    my ($group) = @_;
    my $prefix_guard = $log->msg_prefix_guard('["adgroup_id":'.$group->{adgroup_id}.']');

    my $inserted_modifier_ids = $INSERTED_MODIFIER_IDS{$group->{adgroup_id}};
    if (!$inserted_modifier_ids || scalar(keys %$inserted_modifier_ids) == 0) {
        return;
    }

    my $old_modifiers = _get_group_modifiers($group);
    $log->out({old_modifiers => $old_modifiers});

    if (!$old_modifiers->{weather_multiplier}) {
        return;
    }
    my $old_multiplier_id = $old_modifiers->{weather_multiplier}{hierarchical_multiplier_id};
    if (!$inserted_modifier_ids->{$old_multiplier_id}) {
        return;
    }

    $log->out("undoing weather_multiplier");
    delete $old_modifiers->{weather_multiplier};

    _update_group_modifiers($group, $old_modifiers);

    $log->out({removed_modifier_id => $old_multiplier_id});
    ++$SUCCESS_COUNT;
}
