package Beta;

=head1 NAME

quasi-make::Beta - правила для разработческих бет

=head1 DESCRIPTION

Правила для разработческих рабочих копий.

=head1 COMMENTS

    Мотивация

    вообще надо систематически упростить генерацию всего, что бывает нужно на
    бете Директа: сетей, конфигов apache, Settings, pdf-приложений, переводов,
    переводов для js, регионов, рубрик каталога, городов и стран для саджеста,
    тегов и т.п. Запуск юнит-тестов и перезапуск apache тоже можно считать
    "генерацией".

    Сейчас эти действия разрбосаны по сборке пакетов,
    direct-create-beta,
    direct-sch,
    beta_httpd_conf.pl
    direct-mk,
    а надо, чтобы были вместе, в каком-то одном предсказуемом, удобном и понятном файле. И чтобы выполнять действия можно было независимо друг от друга.

    О реализации: makefile легко написать, но неудобно передавать параметры для
    действия. Скрипты хорошо принимают параметры, но писать надо больше, чем в
    makefil-е + сложнее указывать зависимости между действиями. Нужен какой-то
    гибрид/компромис.

    из фич: reload или restart в зависимости от того, запущен апач или нет; лог
    не удаляется, а дописывается в конец error.log.1; если действие не удалось
    - пишет в stdout ошибки из error.log

=cut

use Direct::Modern;

use Cwd;
use Carp qw/croak/;
use Getopt::Long;
use List::MoreUtils qw/uniq any none/;
use Time::HiRes qw//;
use POSIX qw/strftime/;
use File::Slurp;
use File::Touch qw( touch );
use File::chdir;
use JSON;
use IO::Socket::INET;
use Sys::Hostname;
use Net::Domain;
use Path::Tiny;
$| = 1;

use Yandex::Shell;
use Yandex::HTTP qw/http_fetch/;

use lib::abs '.';
use Beta::Service;

$Beta::Service::CONFIG = "etc/beta/service.yaml";

use utf8;

our $BETA_PORT = _beta_port();
our $BETA_NUMBER = _beta_number();

our $BETA_SETTINGS_FILE = _beta_dir() . "/beta-settings.json";
my $beta_settings = _beta_settings();

# в protected/run по идее хранится информация, относящаяся к перловым процессам, но более подходящего места нет.
our $DOCKER_DB_CONTAINER_ID_FILE = "protected/run/docker_db.container_id";


our $JDK_VERSION = $beta_settings->{jdk_version};
our $REAL_JDK_VERSION = $JDK_VERSION // 11;
our $JAVA_PATH = {
    8 => '/usr/bin/java',
    11 => '/usr/local/yandex-direct-jdk11/bin/java',
    17 => '/usr/local/yandex-direct-jdk17/bin/java',
};
our @JAVA_DEFAULT_APPS = (
    'api5',
    'intapi',
    'web',
    'canvas-backend',
);

my $CANVAS_DOMAIN_DNA = "qanvas-devtest.qams";
if ($beta_settings->{canvas_ui_domain}) {
    $CANVAS_DOMAIN_DNA = $beta_settings->{canvas_ui_domain};
}

my $CANVAS_HOST;
if ($BETA_PORT) {
    $CANVAS_HOST = sprintf("canvas-%d.beta%d.direct.yandex.ru", $BETA_PORT, $BETA_NUMBER);
}

our %JAVA_APP_CFG = (
    logviewer => {
        path => "direct/logviewer",
        main_class => 'ru.yandex.direct.logviewer.LogViewerApp',
        classpath => "./yandex-direct-logviewer",
        props => {
            'jetty.host' => '127.0.0.11',
        },
    },
    api5 => {
        path => "direct/api5",
        main_class => 'ru.yandex.direct.api.v5.YandexDirectApiV5Application',
        classpath => "./yandex-direct-api5",
        props => {
            'jetty.host' => '127.0.0.12',
        },
    },
    intapi => {
        path => "direct/intapi",
        main_class => 'ru.yandex.direct.intapi.IntapiApp',
        classpath => "./yandex-direct-intapi",
        props => {
            'jetty.host' => '127.0.0.13',
        },
    },
    'canvas-backend' => {
        path => "direct/canvas",
        main_class => 'ru.yandex.canvas.Application',
        classpath => "./yandex-direct-canvas-backend",
        props => {
            'spring.profiles.active' => 'beta',
            'profile' => 'beta',
            'server.address' => '127.0.0.40',
            'port' => _beta_port(),
            'DIRECT_URL' => sprintf("http://%d.beta%d.intapi.direct.yandex.ru", $BETA_PORT, $BETA_NUMBER),
            'CANVAS_HOST' => $CANVAS_HOST,
            'CANVAS_VA_API_BASE_URL' => 'https://'.$CANVAS_HOST.'/rest/video-additions/',
            # адрес для редиркта в видео конструктор на питоне, нужен для поиска по видео, чтобы работало придется поднять в докере
            'VIDEO_ADDITIONS_API_URL' => sprintf('http://127.0.0.32:%d/',_beta_port()),
            'SANDBOX_BASE_URL' => 'https://sandbox.yandex-team.ru/api/v1.0',
            'java.security.egd' => 'file:/dev/./urandom',
            'java.net.preferIPv6Addresses' => 'true',
            'java.net.preferIPv4Stack' => 'false',
            'java.net.preferIPv6Addresses' => 'true',
        },
    },
    web => {
        path => "direct/web",
        main_class => 'ru.yandex.direct.web.DirectWebApp',
        classpath => "./yandex-direct-web",
        props => {
            'jetty.host' => '127.0.0.16',
            'canvas.ui_domain' => $CANVAS_DOMAIN_DNA,
        },
    },
    jobs => {
        path           => "direct/jobs",
        main_class     => 'ru.yandex.direct.jobs.DebugJobRunner',
        classpath      => "./direct-jobs",
        debug_runners  => {
            receive_moderation => {
                main_class     => 'ru.yandex.direct.jobs.moderation.debug.DebugReceiveModerationJobRunner',
                default_params => {
                    job => 'ReceiveModerationResponseJob',
                    ess_tag => $BETA_PORT
                }
            }
        }
    },
    'ess_chain' => {
        path => "direct/apps/event-sourcing-system/full-ess-chain",
        main_class => 'ru.yandex.direct.ess.fulltest.DebugEssRunner',
        classpath => "./full-ess-chain",
    },
    'oneshot' => {
        path => "direct/oneshot",
        main_class => 'ru.yandex.direct.oneshot.OneshotApp',
        classpath => "./direct-oneshot",
    },
);

our $INFOBLOCK_HOST = '127.0.0.14';

our $JAVA_YT_CACHE_CLUSTER = 'hahn.yt.yandex.net';

our $JAVA_YT_CACHE_FORDER = '//home/direct/tmp/ya-cache';

Beta::Service::load();

our $BETA_CTL = '/usr/local/bin/beta-ctl';

our ($mroot) = getcwd() =~ m!^(/var/www/[^/]+\.[0-9]+)!;

our %BASE_ACTIONS = (
    DEFAULT => [
        'echo',
        '[ -n "$DIRECT_MK_ORIG_DIR" ] && echo $DIRECT_MK_ORIG_DIR || pwd',
        sub {
            if (!_is_arc()) {
                yash_system q!svn info | grep URL | perl -lpe 'use Term::ANSIColor; s/(svn.yandex.ru\/\w+\/)(.*)/$1.colored(["blue"], $2)/ge'!;
            }
        },
        q!perl -I perl/settings/ -le 'use Term::ANSIColor; use Settings; print "configuration: ", colored ["red"], $Settings::CONFIGURATION'!,
        'echo java_services: '.(join(", ", _java_apps()) || '-'),
        'echo'
    ],
    st => [
        'action:DEFAULT',
        'betas $(basename $(pwd)) -mm -nh',
        "echo",
        ],
    # apache и его конфиги
    rr => {
        todo => [
        "action:apache_logs_clear",
        "action:apache_restart",
        ],
    },
    apache_check => \&apache_check,
    apache_logs_clear => "rm -f ./apache/logs/*",
    apache_logs_move => 'cd apache/logs ; for i in *.log ; cat $i >> $i.1 ; echo "" > $i ; done',
    apache_start => "./apache/init.sh start",
    apache_start_smart => {
        todo => [
        "./apache/init.sh start",
        "action:apache_check",
        ],
    },
    apache_wait_stop => \&apache_wait_stop,
    apache_stop => "./apache/init.sh stop",
    apache_restart => "./apache/init.sh restart",
    apache_restart_smart => \&apache_restart_smart,
    apache_reloader => "./bin/apache_reloader.pl",
    apache_reload => "./apache/init.sh reload",
    apache_reload_smart => {
        todo => [
            'action:apache_reload',
            'action:apache_check',
        ],
    },

    'api5-java-proxy-service-operations' => [
        sub {
            my ($args) = @_;
            local $Yandex::Shell::PRINT_COMMANDS = 1;
            yash_system( $BETA_CTL, 'set', 'api5-java-proxy-service-operations=' . join( ' ', @$args ) );
        },
        'action:apache_restart',
    ],
    'api5-java-proxy-service-operations-reset' => [
        "$BETA_CTL unset api5-java-proxy-service-operations",
        'action:apache_restart',
    ],
    httpd_force_stop => \&httpd_force_stop,
    "httpd-conf" => [
         "./protected/maintenance/beta_httpd_conf.pl",
         "'$BETA_CTL' restart js-templater --only-if-up",
    ],
    "httpd-conf-pure" => "./protected/maintenance/beta_httpd_conf.pl",
    "httpd-conf-init" => "./protected/maintenance/beta_httpd_conf.pl --init",
    "httpd-conf-sandbox" => "./protected/maintenance/beta_httpd_conf.pl --sandbox",
    "httpd-conf-no-intapi" => "./protected/maintenance/beta_httpd_conf.pl --no-intapi",
    "httpd-conf-api5-coverage" => "./protected/maintenance/beta_httpd_conf.pl --api5-coverage",

    messages_log => \&messages_log,

    # Действия при создании, обновлении, удалении беты
    full_stop => [
        'action:canvas_stop',
        sub {
            perform_action('java_stop') if _java_apps();
        },
        './apache/init.sh stop',
        'action:nginx_stop',
        sub {
            perform_action('dna_stop') if _dna_enabled();
        },
        "''$BETA_CTL'' stop all",
        sub {
            system('/usr/local/bin/direct-mk service ALL stop'),
        },
        sub {
            _stop_docker_db() if -f $DOCKER_DB_CONTAINER_ID_FILE;
        },
    ],
    beta_prerm => 'action:full_stop',
    beta_status => [
        'action:service ALL status',
    ],
    beta_postupdate => [
        'mkdir -p tmp/api-reports',
        'action:translations_compile',
        'action:bem_make',
        'action:internal_networks',
        'action:beta_api_wsdls',
        'action:httpd-conf-init',
        "'$BETA_CTL' restart js-templater --only-if-up",
        'action:make_yaml_settings',
    ],
    beta_postcreate => {
        todo => [
        'M=`umask`; umask 0; mkdir protected/logs; umask $M',
        'mkdir -p tmp/api-reports',
        'action:bem_make',
        'action:beta_api_wsdls',
        sub {
            execute(join(" ", './protected/maintenance/beta_httpd_conf.pl --init', @{$_[0]}));
        },
        'action:settings_devtest',
        "'$BETA_CTL' setup js-templater",
        ],
    },
    beta_destroy_all_changes_and_reinit => [
        'action:beta_prerm',
        'svn-reset --quiet .',
        'action:beta_postcreate',
    ],

    # при наличии переменной окружения DIRECT_BETA_NO_BEM не собирает и не пересобираем bem
    # это быстрое решение, чтобы беты для тестирования API создавались быстрее и надежнее, оч надо, пока не ускорена сборка data3
    bem_make => ($ENV{DIRECT_BETA_NO_BEM} ? [] : [
        'action:translations_compile',
        './protected/maintenance/export_consts_to_js.pl',
        './protected/maintenance/export_gettext_to_js.pl',

        # На декабрь 2017 кажется, что кеширование сборки съедает больше ресурсов, чем сама сборка
        # (оверхед: скопировать файлы на hdd, посчитать чексуммы, скопировать архив, распаковать; плюс сборка на hdd медленнее, чем на ssd)
        # поэтому вместо bem-builder возвращаем просто make
        # если нужен продакшен-режим, то использовать bem_make_production
        #'bem-builder --rootdir ./ --conffile etc/bem-builder-data3.conf make',
        'cd data3 && make all-for-robots',

        'action:static_file_hashsums',
    ]),
    # копия bem_make, но с YENV=production
    bem_make_production => ($ENV{DIRECT_BETA_NO_BEM} ? [] : [
        'action:translations_compile',
        './protected/maintenance/export_consts_to_js.pl',
        './protected/maintenance/export_gettext_to_js.pl',
        'cd data3 && YENV=production make all-for-robots',

        'action:static_file_hashsums',
    ]),

    bem_make_usercache => [
        'action:translations_compile',
        './protected/maintenance/export_consts_to_js.pl',
        './protected/maintenance/export_gettext_to_js.pl',
        'bem-builder --rootdir ./ --conffile etc/bem-builder-data3.conf --cache user make',
        'action:static_file_hashsums',
    ],

    # подсчет hash сумм файлов статики
    static_file_hashsums => [
        # несуществующие директории пропускаются, поэтому не страшно иметь директории, которые есть не на каждой бете
        './protected/maintenance/hash_data2_files.pl data3/desktop.bundles/direct data3/desktop.bundles/morda data3/desktop.bundles/head data3/touch.bundles/direct dna/build/static/css dna/build/static/js -o data3/hashsums.json',
    ],

    dev_server_tests => sub {
        execute("cd data3; TARGET_BLOCK=$_[0]->[0] BETA_PORT=" . _beta_port() . " make dev-server-tests")
    },

    dev_server_sandbox => sub {
        execute("cd data3; TARGET_BLOCK=$_[0]->[0] BETA_PORT=" . _beta_port() . " make dev-server-sandbox")
    },

    error_pages => 'cd data3/utils/error-pages/ && ./generate-error-pages.sh',

    'js-templater-start' => [
        'beta-ctl start js-templater',
    ],

    'js-templater-stop' => [
        'beta-ctl stop js-templater',
    ],

    'js-templater-restart' => [
        'beta-ctl restart js-templater',
    ],

    reload  => 'action:apache_reload',
    restart => 'action:apache_restart',

    # Разное
    'prepare-for-test' => [
        'mkdir -p protected/logs',
        './protected/prebuild/mk_api_wsdl.pl',
    ],
    test => './unit_tests/runtests.pl --skip-tag perl --skip-tag templates --skip-tag smoke',
    "test-full" => [
        'action:prepare_for_test',
        "./unit_tests/runtests.pl --skip-tag smoke",
    ],
    "test-smoke" => [
        "./unit_tests/runtests.pl --only-tag smoke",
    ],
    "test-full-heavy" => [
        'echo -e "\n\n\n!!! test-full-heavy deprecated. use test-full instead of !!!\n\n\n"',
        'action:test-full',
    ],
    "test-dependencies" => "./unit_tests/runtests.pl --no-db ./unit_tests/packages/dependencies_installed.t",
    "test-frontend" => "cd data3/ && make tests.standalone",

    tags => "direct-ctags",

    # Settings
    # TODO соответствие между названиями БД и Settings'ов хранить где-то отдельно,
    # там же хранить короткие алиасы для direct-sql
    settings_devtest => [
        'beta-ctl use-db devtest',
        sub { switch_settings_local("DevTest") },
    ],
    settings_dev7 => [
        'beta-ctl use-db dev7',
        sub { switch_settings_local("Dev7") },
    ],
    settings_roprod => [
        'beta-ctl use-db roprod',
        sub { switch_settings_local("ROProd") },
    ],
    settings_test => [
        'beta-ctl use-db test',
        sub { switch_settings_local("Test") },
    ],
    settings_test2 => [
        'beta-ctl use-db test2',
        sub { switch_settings_local("Test2") },
    ],
    settings_testload => [
        'beta-ctl use-db testload',
        sub { switch_settings_local("TestLoad") },
    ],
    settings_sandbox => [
        'beta-ctl use-db sandbox',
        sub { switch_settings_local("Sandbox") },
    ],
    settings_sandbox_devtest => [
        'beta-ctl use-db sandboxdevtest',
        sub { switch_settings_local("SandboxDevTest") },
    ],
    settings_sandbox_dev7 => [
        'beta-ctl use-db sandboxdev7',
        sub { switch_settings_local("SandboxDev7") },
    ],
    settings_sandbox_test => [
        'beta-ctl use-db sandboxtest',
        sub { switch_settings_local("SandboxTest") },
    ],
    settings_sandbox_test2 => [
        'beta-ctl use-db sandboxtest2',
        sub { switch_settings_local("SandboxTest2") },
    ],
    settings_prod => [
        'beta-ctl use-db roprod',
        sub { switch_settings_local("") },
    ],
    settings_dockerdb => [
        'beta-ctl use-db dockerdb',
        sub { switch_settings_local("DockerDB") },
    ],
    # conf_* генерируются автоматически на основе settings_*, см. actions

    # особо полензные конфигурации снабжаем короткими именами
    devtest => "action:conf_devtest",
    dev7 => "action:conf_dev7",
    prod => "action:conf_prod",
    roprod => "action:conf_roprod",

    # запуск скриптов ./protected/ppc*
    servicing_events => './protected/ppcServicingCampEvents.pl',

    news => './protected/getNews.pl --target rss --target db --target json',
    intapi_coverage_collect => './protected/maintenance/intapi_coverage/collect_coverage.pl',
    api_stop_and_collect_coverage => './protected/maintenance/api_coverage/stop_and_collect_coverage.pl',

    nginx_stop => './nginx/init.sh stop',
    nginx_start => './nginx/init.sh start',
    nginx_restart => [ './nginx/init.sh stop || true', 'sleep 1', 'action:nginx_start' ],
    nginx_reload => './nginx/init.sh reload',

    service => \&Beta::Service::service,

    up => [
        sub {
            if (_is_arc()) {
                warn "up is not supported for arc, checkout manually, and do beta-postupdate";
                return;
            }
            execute('direct-svn-up.pl --no-restart . ' . (join " ", map {yash_quote($_)} @{$_[0]}));
            perform_action('beta_postupdate');
            perform_action('java_up') if _java_apps();
            perform_action('dna_up') if _dna_enabled();
        },
    ],

    dna_up => [
        sub {
            require Startrek::Client::Easy;
            my $st = Startrek::Client::Easy->new();
            # TODO возможность указать версию напрямую
            $_[0]->[0] =~ /^startrek:([\w_-]+)$/;
            if (defined $1) {
                my $st_index = $1;
                $st_index = -1 if $st_index eq 'last';
                $st_index = -2 if $st_index eq 'before-last';

                die("Wrong startrek release index value, it should be 'last', 'before-last', or integer < 0\n") unless $st_index =~ /^-?[0-9]+$/ && $st_index < 0;
                die("Startrek release index should be >= -10\n") if $st_index < -10;

                my @last_10_releases = @{ $st->get(query => qq/Queue: DIRECT Type: Release Components: "App: dna" "Sort By": Key DESC/, limit => 10) || [] };
                my $release = $last_10_releases[-$st_index - 1];
                my $branch = $release->{branch} or die ("empty branch field in release ticket " . $release->{key} . " , stop");
                warn "removing old dna\n";
                yash_system 'rm', '-rf', 'dna';
                warn "setting up dna from branch $branch\n";
                yash_system $BETA_CTL, 'setup', 'dna', '--branch', $branch, '--fast', '1';
            } else {
                my $commit_before_update = _dna_commit();

                yash_system $BETA_CTL, 'update-dna';

                my $commit = _dna_commit();

                if ($commit ne $commit_before_update) {
                    my $beta = _beta_hostname();
                    my $comment = "Бета https://$beta обновлена." . "Коммит ((https://github.yandex-team.ru/direct/dna/commit/$commit $commit))";

                    $st->do(
                        key     => _dna_ticket(),
                        comment => $comment,
                    );
                }
            }
        },
    ],

    dna_stop => [
        sub {
            yash_system $BETA_CTL, 'stop', 'dna';
        }
    ],

    dna_enable_hashsums_override => [
        sub {
            yash_system $BETA_CTL, 'enable-dna-override';
        }
    ],

    dna_disable_hashsums_override => [
        sub {
            yash_system $BETA_CTL, 'disable-dna-override';
        }
    ],

    dna_record_port => [
        sub {
            require Startrek::Client::Easy;
            my $st = Startrek::Client::Easy->new();

            my $tags = $st->get(key=>_dna_ticket())->{tags};
            push $tags, "BETA_PORT_$BETA_PORT";

            $st->do(
                key  => _dna_ticket(),
                tags => $tags,
            );
        },
    ],

   beta_api_wsdls => [
        "./protected/prebuild/mk_api_wsdl.pl --createall --beta --hostname=" . _beta_hostname() . ":14443",
   ],

   canvas_start =>[ "'$BETA_CTL' start canvas" ],
   canvas_stop => \&canvas_stop,
   canvas_setup => [ "'$BETA_CTL' setup canvas" ],

   # TODO изучить возможность переноса в beta-service
   docker_db_start => sub {
       $Yandex::Shell::PRINT_COMMANDS = 1;
       if (-f $DOCKER_DB_CONTAINER_ID_FILE) {
           die "can't start container: file $DOCKER_DB_CONTAINER_ID_FILE already exists, is container running?\n";
       }
       my $beta_port = _beta_port();
       my $image_id = 'ppccloud-registry-test.yandex.ru:5000/direct_empty_db_mysql_v0.5';
       my $container_id = yash_qx('docker', 'run', '-d', '-p', "127.0.0.10:${beta_port}:3306", $image_id);
       chomp $container_id;
       write_file($DOCKER_DB_CONTAINER_ID_FILE, $container_id);
       print "container $container_id started, id stored in $DOCKER_DB_CONTAINER_ID_FILE\n";

       # если запускать скрипты сразу, то получаем ошибку
       # ERROR 2013 (HY000): Lost connection to MySQL server at 'reading initial communication packet', system error: 0
       # поэтому sleep
       sleep 5;

       my @scripts = (
           "tpage --define port=$beta_port etc/db-config.dockerdb.json.tmpl > etc/db-config.dockerdb.json",
           ['direct-populate-sandbox.pl', '--host', '127.0.0.10', '--port', $beta_port, '--user', 'adiuser', glob('db_schema/*')],
           'protected/maintenance/create_test_users.pl',
           ['protected/mk_regions.pl', '--db', '--timezones'],
           'protected/ppcFetchCountryCurrencies.pl',
       );

       {
           local $ENV{MYSQL_PWD} = 'utro';
           local $ENV{SETTINGS_LOCAL_SUFFIX} = 'DockerDB';
           for my $script (@scripts) {
               my $ok = eval { yash_system(ref $script eq 'ARRAY' ? @$script : $script); 1 };
               if (! $ok) {
                   my $error = $@;
                   print $error;
                   print "stopping container...\n";
                   _stop_docker_db();
                   exit 1;
               }
           }
       }
       $Yandex::Shell::PRINT_COMMANDS = 0;
   },
   docker_db_stop => \&_stop_docker_db,

   run_java_job => sub {
       my $args = shift;
       die "job name required; usage: mk run_java_job HotPhrasesSynchronizer -- --shard 3" unless $args->[0];
       my $job_name = $args->[0];
       my $job_params = [@$args[1..(@$args-1)]];
       _run_java_job($job_name, $job_params);
   },
   run_ess_chain => sub {
       my $args = shift;
       _run_ess_chain($args);
   },
   run_receive_moderation => sub {
       my $args = shift;
       _run_receive_moderation($args);
   },
   run_oneshot => sub {
       my $args = shift;
       _run_oneshot($args);
   },
   java_ya_make => sub {
       my $args = shift;

       my @apps = _java_apps();
       @apps = ($args->[0]) if $args && $args->[0];
       die "uknown java apps: " . join(",", @apps) if any { !exists $JAVA_APP_CFG{$_} } @apps;

       if (@apps) {
           local $CWD = _beta_dir();
           my $yatool = "./arcadia/ya";
           yash_system("ulimit -v 100000000 && $yatool make --ignore-recurses --stat --yt-store -o java ".join(" ", map {"arcadia/$JAVA_APP_CFG{$_}->{path}"} @apps));
       }
   },
   java_init => [
    sub {
        my %opt;
        my @apps;
        my $patch_review_id;
        my $no_sandbox;
        my $launch_id;
        Getopt::Long::GetOptionsFromArray($_[0],
            'b|branch=s' => sub { $opt{'--branch'} = '/arc/branches/direct/task/' . $_[1] . '/arcadia' },
            'a|app=s' => sub {
                my $apps = $_[1];
                for my $app (split /,/, $apps) {
                    if (!exists $JAVA_APP_CFG{$app}) {
                        die "Unknow java app: $app";
                    }
                    push @apps, $app;
                }
            },
            'r|revision=s' => sub { $opt{'--revision'} = $_[1] },   # sub, а не \$opt{'--revision'} , чтобы не автооживлялся ключ
            'rp|review-patch=s' => \$patch_review_id,
            'l|launch-id=s' => \$launch_id,
            'ns|no-sandbox' => \$no_sandbox,
        ) || die "can't parse options, stop";
        if (@{$_[0]}) {
            die "Unexpected free arguments: ".join(" ", map {"'$_'"} @{$_[0]});
        }
        if (!@apps) {
            die "Java applications not specified.\nUsage: direct-mk java-init -- -a APP1 -a APP2 ...\n where APPn is one of: ".join(' ', sort keys %JAVA_APP_CFG)."\n";
        }

        yash_system($BETA_CTL, 'set-from-json', "java_apps=".to_json(\@apps));
        local $Yandex::Shell::PRINT_COMMANDS = 1;

        if (!$no_sandbox && _is_only_sandboxed_java_apps(@apps)) {
            _update_sandbox_cli();
            my $sb_cli = _sb_cli();

            my $branch = $patch_review_id ? "\\\"pr:$patch_review_id\\\"" : "\\\"trunk\\\"";
            my $launch_substr = substr($launch_id // '', 0, 12);
            my $search_string = $launch_id ? "--hints LAUNCH:$launch_substr" : "--input-params env_vars=BRANCH=$branch";

            if (`$sb_cli locate -t YA_MAKE_2 $search_string`) {
                {
                    local $CWD = _beta_dir();
                    yash_system("$sb_cli download-artifact -t YA_MAKE_2 $search_string -a BUILD_OUTPUT -O java-web-build.tgz");
                    yash_system("tar", "zxf", "java-web-build.tgz");
                    yash_system("rm", "-rf", "java");
                    yash_system("mv", "build", "java");
                    yash_system("svn", "checkout", "svn+ssh://arcadia.yandex.ru/arc/trunk/arcadia/direct/libs-internal/config/src/main/resources", "java_resources/");
                    yash_system($BETA_CTL, 'set-from-json', "java_resources_path=\"/java_resources/\"");
                    yash_system($BETA_CTL, 'set-from-json', "java_from_sandbox=1");
                }

                # для sandbox-сборки позволяем обновлять приложение через java-init, поэтому сейчас может быть запущено другое приложение
                perform_action('java_stop');

                return;
            }
        }

        if (!_is_arc()) {
            {
                # run ya clone from svn cat
                open my $svn_cat_fh, '-|', 'svn', 'cat', 'svn+ssh://arcadia.yandex.ru/arc/trunk/arcadia/ya' or die "can't svn cat ya: $!";
                open my $py_fh, '|-', 'python', '-', 'clone', '--no-junk', 'arcadia', %opt or die "can't start python: $!";
                local $SIG{PIPE} = sub { die "write to pipe failed" };
                print $py_fh $_ while (<$svn_cat_fh>);
                close $py_fh or die $! ? "can't close pipe: $!" : "ya clone exited with status " . ($? >> 8);
            }

            local $CWD = _beta_dir();
            my $yatool = "./arcadia/ya";
            yash_system($yatool, 'make', '-j0', '--checkout', map {"arcadia/".$JAVA_APP_CFG{$_}->{path}} @apps);
        }

        if ($patch_review_id) {
            local $CWD = _beta_dir();
            yash_system("cd ./arcadia && ./ya pr checkout $patch_review_id");
        }

        perform_action('java_ya_make')
    },
    'action:java_start',
    'action:httpd-conf',
   ],
   java_stop => [
    (map "action:${_}_stop", grep { $_ !~ 'jobs|ess_chain|oneshot' } keys %JAVA_APP_CFG),
   ],
   java_start => [
       sub {
           for my $app (grep { $_ !~ 'jobs|ess_chain|oneshot' } _java_apps()) {
               perform_action($app."_start");
           }
       }
   ],
   java_cleanup => [
    'action:java_stop',
    sub {
       my $root = _beta_dir();
       local $Yandex::Shell::PRINT_COMMANDS = 1;

       yash_system('rm', '-rf', "$root/java");
       yash_system('rm', '-rf', "$root/java-resources");
       yash_system('rm', '-f', "$root/java-web-build.tgz");
       if (!_is_arc()) {
           yash_system('rm', '-rf', "$root/arcadia");
       }

       yash_system($BETA_CTL, unset => "java_apps");
       yash_system($BETA_CTL, unset => "java_resources_path");
       yash_system($BETA_CTL, unset => "java_from_sandbox");
    },
    'action:httpd-conf',
   ],
   (
    map {
        my $app = $_;
        "${app}_start" => sub {my $detach = 1; GetOptions('detach!' => \$detach) || die "can't parse options, stop"; _start_java_service($app, $detach)},
        "${app}_stop" => sub {_stop_java_service($app)},
    } grep { $_ !~ 'jobs|ess_chain|oneshot' } keys %JAVA_APP_CFG
   ),
   java_up => [
    sub {
        if (_is_arc()) {
            warn "java-up is not supported for arc, checkout manually and do java-ya-make, java-stop, java-start";
            return;
        }
        if ($beta_settings->{java_from_sandbox}) {
            warn "java inited from sandbox, skipped java-up\n";
            return;
        }
        if (!-d 'arcadia') {
            warn "no arcadia checkout, skipped\n";
            return;
        };
        my ($revision) = @{ $_[0] || [] };
        my $yatool = "./arcadia/ya";
        yash_system('svn', 'up', 'arcadia', $revision ? ('--revision' => $revision) : ());
        yash_system($yatool, 'make', '--checkout', '-j0', map {"arcadia/".$JAVA_APP_CFG{$_}->{path}} _java_apps());

        perform_action('java_ya_make');
        perform_action('java_stop');
        perform_action('java_start');
    },
   ],
   java_patch => [
    sub {
        my ($args) = @_;
        my $REV;
        my $DRAFT;
        Getopt::Long::GetOptionsFromArray($args,
            'h|help' => sub {
                print "Usage: mk java-patch -- OPTIONS REVIEW\n"
                    ."\n"
                    ."apply patch from given REVIEW to java-code in dir 'arcadia'\n"
                    ."\n"
                    ."Options:\n"
                    ." -h, --help - show help message\n"
                    ." -r, --revision - target revision for applying patch\n"
                    ." -d, --draft - use review draft \n"
                    ."\n"
                    ."Examples:\n"
                    ."    mk java-patch -- --revision 12345 https://a.yandex-team.ru/review/123456/details\n"
                    ."    mk java-patch -- --draft https://a.yandex-team.ru/review/123456/details\n"
                    ."    mk java-patch -- https://a.yandex-team.ru/review/123456/details\n"
                    ;
                exit;
            },
            "r|revision=i" => \$REV,
            "d|draft" => \$DRAFT,
        ) || die "can't parse options, stop";

        if (@$args > 1) {
            die "Supported only one argument - filter pattern";
        }
        my ($URL) = @$args;

        if (!_is_arc() && !-d 'arcadia') {
            warn "no arcadia checkout, skipped\n";
            return;
        };
        if (!defined $URL ||
              !($URL =~ qr{^(https://rb\.yandex-team\.ru/arc/r/|https://a\.yandex-team\.ru/(arc/)?review/)?(\d+).*})) {
            die "Usage: mk java_patch [ https://a\.yandex-team\.ru/review/NNNNNN | NNNNNN ]";
        }

        my $patch_review_id = $3;
        my $yatool = _beta_dir() . "/arcadia/ya";

        if (_is_arc() && $REV) {
            die "--revision is not supported for arc, use 'arc pr co --dirty', 'arc co r$REV'";
        }
        if (_is_arc() && $DRAFT) {
            warn "--draft has no effect with arc, draft revision is checked out by default";
        }

        if (_is_arc()) {
            yash_system("arc pr checkout --force $patch_review_id");
        } else {
            yash_system('svn-reset', './arcadia');
            yash_system('svn-reset', './arcadia'); # workaround - добавленные файлы удаляются только со второй попытки
            yash_system('svn', 'up', './arcadia', ($REV ? ('--revision' => $REV) : ()));

            yash_system("cd ./arcadia && $yatool pr checkout $patch_review_id".($DRAFT ? (' --draft') : ()));

            yash_system($yatool, 'make', '--checkout', '-j0', map {"arcadia/".$JAVA_APP_CFG{$_}->{path}} _java_apps());
        }
    },
    sub { local @ARGV; perform_action('java_ya_make') }, # чтобы в java_ya_make не попадал наш args
    'action:java_stop',
    'action:java_start',
   ],

   update_banner_storage_dict => sub {

        yash_system("$mroot/protected/getBannerStorageDicts.pl");
   },

   make_yaml_settings => sub {
        my $cwd = cwd();
        yash_system("$cwd/protected/maintenance/make_yaml_settings.pl");
   },
   ###
);


# TODO генерацию правил для конфигураций по списку Settings'ов;


sub actions
{
    my %ACTIONS = %BASE_ACTIONS;

    for my $settings_action ( grep {/^settings_/} keys %ACTIONS ){
        (my $conf_action = $settings_action) =~ s/^settings_/conf_/;
        $ACTIONS{$conf_action} = [
        # ($conf_action =~ /^conf_(test|test2|testload|sandbox|sandbox_test|sandbox_test2|prod)$/
        ($conf_action =~ /^conf_(test|sandbox|sandbox_test|prod)$/
            ? qq{echo "\n\nПользоваться конфигурацией $1 на бете не рекомендуется.\n\n"}
            : ()
        ),
        "action:$settings_action",
        'action:make_yaml_settings',
        "action:apache_restart_smart",
        ];
    }

    return \%ACTIONS;
}

sub apache_restart_smart
{
    my $port = _beta_port();
    if (_connect_port_ok($port)) {
        # сразу после запуска апача он не реагирует на stop, нужно сначала дожаться чтоб он заработал
        apache_check();
        system "./apache/init.sh stop";
        apache_wait_stop();
    }
    system "./apache/init.sh start";
    apache_check();
}

sub apache_wait_stop
{
    my $port = _beta_port();
    my $cnt = 0;
    while (_connect_port_ok($port)) {
        # print '~';
        if ($cnt++ > 200) {
            print "FAILED to stop apache\n";
            last;
        }
        Time::HiRes::sleep 0.5;
    }
}

# спасибо icenine@ и /opt/home/icenine/bin/r.pl
sub apache_check
{
    my $port = _beta_port();

    my $error_log;
    my $connect_ok = 0;
    my $url = "http://127.0.0.2:$port/DirectConfiguration";
    for (0..15) {
        Time::HiRes::sleep 0.5;
        eval {
            http_fetch GET => $url;
        };
        next if $@;

        $connect_ok = 1;
        last;
    }
    if ($connect_ok){
        print "OK\n";
    } else {
        $error_log = read_file('apache/logs/error.log');
        $error_log =~ s/\\n/\n/g;
        $error_log = substr($error_log, -1000, 1000);
        print "\ncan't connect to port $port\nerror.log says:\n$error_log\n";
    }
}

# убить все apache и nginx этой беты
sub httpd_force_stop
{
    my $user = [getpwuid($<)]->[0];
    my $port = _beta_port();
    my @apache_pids = map { (split /\s+/)[1] } split /\n/, `ps axuww | grep $user | grep $port | grep apache2 | grep -v grep`;
    my @nginx_pids  = map { (split /\s+/)[1] } split /\n/, `ps axuww | grep $user | grep $port | grep nginx | grep -v grep`;
    my @pids = (@apache_pids, @nginx_pids);
    if (!@pids) {
        print STDERR "force_stop: nothing to terminate\n";
        return;
    }
    print STDERR "about to terminate pids: [".(join ", ", @pids)."]\n";
    kill 15 => @pids;

    sleep 1;
    @apache_pids = map { (split /\s+/)[1] } split /\n/, `ps axuww | grep $user | grep $port | grep apache2 | grep -v grep`;
    @nginx_pids  = map { (split /\s+/)[1] } split /\n/, `ps axuww | grep $user | grep $port | grep nginx | grep -v grep`;
    @pids = (@apache_pids, @nginx_pids);
    return unless (@pids);
    sleep 2;
    print STDERR "about to kill pids: [".(join ", ", @pids)."]\n";
    kill '-KILL' => @pids;
}

sub _connect_port_ok
{
    my $s = new IO::Socket::INET(
        PeerAddr => '127.0.0.2', # адрес для беты веб-интерфейса из etc/frontend/apache2/ppc.yandex.ru.conf
        PeerPort => $_[0],
        Proto => 'tcp',
    );
    return !! $s;
}

# спасибо zhur@ и direct-sch
sub switch_settings_local
{
    my ($suff) = @_;

    my $cwd = cwd();

    my $local = "$cwd/perl/settings/SettingsLocal.pm";
    if (!-e $local) {
        print "old link: empty\n";
    } elsif (!-l $local) {
        die "SettingsLocal is not symlink";
    } else {
        my $link = readlink $local;
        print "old link: $link\n";
    }

    my $source = "$cwd/perl/settings/Settings$suff.pm";

    unlink($local);
    if ($suff){
        symlink("$source", $local);
        print "new link: $source\n";
    } else {
        print "new link: empty\n"
    }

    yash_system("$cwd/protected/maintenance/beta_httpd_conf.pl");

    for my $app (_java_apps()) {
        # процедура perform_action доступна только если этот модуль подключается из quasi-make, поэтому костылим, чтобы этот кусок "компилировался" во время выполнения
        sub {
            perform_action("${app}_stop");
            perform_action("${app}_start");
        }->();
    }
}

sub _beta_port
{
    state $cwd = getcwd();
    # в пути к каталогу находим первое вхождение элемента beta.(что-то без точек).(число), и считаем, что (число) -- это порт беты
    (my $port = $cwd) =~ s!^.*?/beta\.[^\.]+\.([0-9]+)(?:$|/.*)!$1!;
    return $port;
}

sub _beta_number {
    state $hostname = Sys::Hostname::hostname();
    $hostname =~ /^ppcdev(\d+)$/;
    return $1;
}

sub _beta_hostname {
    return _beta_port() . ".beta" . _beta_number . ".direct.yandex.ru";
}

sub _is_arc
{
    state $is_arc;
    if (defined $is_arc){
        return $is_arc
    }

    my $path = getcwd();
    $path =~ s!(^.*?/beta\.[^\.]+\.[0-9]+)(?:$|/.*)!$1!;
    $is_arc = ( -d "$path/arcadia" ) && ( -d "$path/store" );

    return $is_arc;
}

sub _beta_dir
{
    state $root;
    if (!defined $root) {
        ($root) = getcwd() =~ m!^(/var/www/[^/]+\.[0-9]+)!;
    }
    return $root;
}

sub _is_only_sandboxed_java_apps {
    my @apps = @_;

    return (scalar(@apps) eq 1) && ($apps[0] eq "web");
}


sub _dna_commit {
    return `git -C dna log --pretty=format:'%h' -n 1`;
}

sub _dna_branch {
    return `git -C dna rev-parse --abbrev-ref HEAD`;
}

sub _dna_ticket {
    my ($ticket) = _dna_branch() =~ /(DIRECT-\d+)/;
    return $ticket;
}

sub _stop_docker_db {
    if (! -f $DOCKER_DB_CONTAINER_ID_FILE) {
        die "can't stop the container: file $DOCKER_DB_CONTAINER_ID_FILE not found\n";
    }
    my $container_id = read_file($DOCKER_DB_CONTAINER_ID_FILE);
    chomp $container_id;
    yash_system('docker', 'stop', $container_id);
    yash_system('docker', 'rm', $container_id);
    unlink $DOCKER_DB_CONTAINER_ID_FILE;
}

sub _run_java_job {
    my ($job_name, $job_params) = @_;

    print "run java job $job_name\n";
    my $dir = _beta_dir() . "/java/$JAVA_APP_CFG{jobs}{path}";
    unless (-d "$dir/run") {
        mkdir "$dir/run" or die "can't create directory '$dir/run': $!";
    }
    unless (-d "$dir/logs") {
        mkdir "$dir/logs" or die "can't create directory '$dir/logs': $!";
    }

    my $classpath = $JAVA_APP_CFG{jobs}->{classpath};
    my %args = (
        'yandex.environment.type' => _configuration(),
        'LOG_ROOT' => "$dir/logs",
        'java.library.path' => $classpath,
    );

    my $yatool = _beta_dir() . "/arcadia/ya";
    my $beta_settings = _beta_settings();
    my @java_args = (
        ($beta_settings->{java_agent_path} ? "-agentpath:$beta_settings->{java_agent_path}" : ()),
        ($classpath ? ('-cp' => "$classpath/*:") : ()),
        (map { "-D$_" . '=' . $args{$_} } keys %args),
        "-Xmx2g",
        "-Xms512m",
        "-XX:ParallelGCThreads=2",
        "-XX:CICompilerCount=2",
    );

    if (my $props = $beta_settings->{java_properties}) {
        croak "java_properties in $BETA_SETTINGS_FILE must be hash" unless ref $props eq 'HASH';
        push @java_args, map {yash_quote("-D$_=$props->{$_}")} keys %$props;
    }

    my @app_args = (
        "--log-configs-directory" => _beta_dir() . "/arcadia/direct/libs-internal/config/src/main/resources/",
    );

    my $java_cmd = join " ", map {yash_quote($_)} $JAVA_PATH->{$REAL_JDK_VERSION}, @java_args, $JAVA_APP_CFG{jobs}->{main_class}, $job_name, @app_args, @$job_params;
    yash_system("ulimit -v 100000000 && cd $dir && exec $java_cmd");
}

sub _run_ess_chain {
    my ($ess_chain_params) = @_;
    print "run ess chain\n";

    my $dir = _beta_dir() . "/java/$JAVA_APP_CFG{ess_chain}{path}";
    unless (-d "$dir/run") {
        mkdir "$dir/run" or die "can't create directory '$dir/run': $!";
    }
    unless (-d "$dir/logs") {
        mkdir "$dir/logs" or die "can't create directory '$dir/logs': $!";
    }
    my $classpath = $JAVA_APP_CFG{ess_chain}->{classpath};
    my %args = (
        'yandex.environment.type' => _configuration(),
        'LOG_ROOT' => "$dir/logs",
        'java.library.path' => $classpath,
    );
    my $beta_settings = _beta_settings();
    my @java_args = (
        ($beta_settings->{java_agent_path} ? "-agentpath:$beta_settings->{java_agent_path}" : ()),
        ($classpath ? ('-cp' => "$classpath/*:") : ()),
        (map { "-D$_" . '=' . $args{$_} } keys %args),
        "-Xmx2g",
        "-Xms512m",
        "-XX:ParallelGCThreads=2",
        "-XX:CICompilerCount=2",
    );

    if (my $props = $beta_settings->{java_properties}) {
        croak "java_properties in $BETA_SETTINGS_FILE must be hash" unless ref $props eq 'HASH';
        push @java_args, map {yash_quote("-D$_=$props->{$_}")} keys %$props;
    }

    my @app_args = (
        "--log-configs-directory" => _beta_dir() . "/arcadia/direct/libs-internal/config/src/main/resources/",
    );

    unless (any { $_ eq '--ess-tag' } @$ess_chain_params) {
        my $ess_tag = $beta_settings->{java_properties}->{"tracing.tags.ess"};
        push @app_args, "--ess-tag" => yash_quote("$ess_tag") if $ess_tag;
    }

    my $java = $JAVA_PATH->{$REAL_JDK_VERSION};
    my $java_cmd = join " ", map {yash_quote($_)} $java, @java_args, $JAVA_APP_CFG{ess_chain}->{main_class}, @app_args, @$ess_chain_params;
    yash_system("ulimit -v 100000000 && cd $dir && exec $java_cmd");

}

sub _run_receive_moderation {
    my ($moderation_params) = @_;
    print "run receive moderation\n";

    my $dir = _beta_dir() . "/java/$JAVA_APP_CFG{jobs}{path}";
    my $runner_info = $JAVA_APP_CFG{jobs}->{debug_runners}->{receive_moderation};
    unless (-d "$dir/run") {
        mkdir "$dir/run" or die "can't create directory '$dir/run': $!";
    }
    unless (-d "$dir/logs") {
        mkdir "$dir/logs" or die "can't create directory '$dir/logs': $!";
    }
    my $classpath = $JAVA_APP_CFG{jobs}->{classpath};
    my %args = (
        'yandex.environment.type' => _configuration(),
        'LOG_ROOT' => "$dir/logs",
        'java.library.path' => $classpath,
    );
    my $beta_settings = _beta_settings();
    my @java_args = (
        ($beta_settings->{java_agent_path} ? "-agentpath:$beta_settings->{java_agent_path}" : ()),
        ($classpath ? ('-cp' => "$classpath/*:") : ()),
        (map { "-D$_" . '=' . $args{$_} } keys %args),
        "-Xmx2g",
        "-Xms512m",
        "-XX:ParallelGCThreads=2",
        "-XX:CICompilerCount=2",
    );

    if (my $props = $beta_settings->{java_properties}) {
        croak "java_properties in $BETA_SETTINGS_FILE must be hash" unless ref $props eq 'HASH';
        push @java_args, map {yash_quote("-D$_=$props->{$_}")} keys %$props;
    }

    my @app_args = (
        "--log-configs-directory" => _beta_dir() . "/arcadia/direct/libs-internal/config/src/main/resources/",
    );

    if (none { $_ eq '--ess-tag' } @$moderation_params) {
        my $ess_tag = $runner_info->{default_params}->{ess_tag};
        push @app_args, "--ess-tag" => yash_quote("$ess_tag") if $ess_tag;
    }

    if (none { $_ eq '--job' } @$moderation_params) {
        my $default_job = $runner_info->{default_params}->{job};
        push @app_args, "--job" => yash_quote("$default_job") if $default_job;
    }

    my $java = $JAVA_PATH->{$REAL_JDK_VERSION};
    my $java_cmd = join " ", map {yash_quote($_)} $java, @java_args, $runner_info->{main_class}, @app_args, @$moderation_params;
    yash_system("ulimit -v 100000000 && cd $dir && exec $java_cmd");

}

sub _run_oneshot {
    my ($oneshot_params) = @_;
    print "run oneshot\n";

    my $dir = _beta_dir() . "/java/$JAVA_APP_CFG{oneshot}{path}";
    unless (-d "$dir/run") {
        mkdir "$dir/run" or die "can't create directory '$dir/run': $!";
    }
    unless (-d "$dir/logs") {
        mkdir "$dir/logs" or die "can't create directory '$dir/logs': $!";
    }
    my $classpath = $JAVA_APP_CFG{oneshot}->{classpath};
    my %args = (
        'yandex.environment.type' => _configuration(),
        'LOG_ROOT' => "$dir/logs",
        'java.library.path' => $classpath,
    );
    my $beta_settings = _beta_settings();
    my @java_args = (
        ($beta_settings->{java_agent_path} ? "-agentpath:$beta_settings->{java_agent_path}" : ()),
        ($classpath ? ('-cp' => "$classpath/*:") : ()),
        (map { "-D$_" . '=' . $args{$_} } keys %args),
        "-Xmx6g",
        "-Xms512m",
        "-XX:ParallelGCThreads=2",
        "-XX:CICompilerCount=2",
    );

    if (my $props = $beta_settings->{java_properties}) {
        croak "java_properties in $BETA_SETTINGS_FILE must be hash" unless ref $props eq 'HASH';
        push @java_args, map {yash_quote("-D$_=$props->{$_}")} keys %$props;
    }

    my @app_args = (
        "--log-configs-directory" => _beta_dir() . "/arcadia/direct/libs-internal/config/src/main/resources/",
    );

    my $java = $JAVA_PATH->{$REAL_JDK_VERSION};
    my $java_cmd = join " ", map {yash_quote($_)} $java, @java_args, $JAVA_APP_CFG{oneshot}->{main_class}, @app_args, @$oneshot_params;
    yash_system("ulimit -v 100000000 && cd $dir && exec $java_cmd");

}

sub _start_java_service {
    my ($name, $detach) = @_;

    my $conf = _configuration();

    my $app_cfg = $JAVA_APP_CFG{$name};

    my %O = %{$app_cfg->{props}};

    print "starting service $name\n";
    my $dir = _beta_dir() . "/java/$JAVA_APP_CFG{$name}{path}";
    unless (-d "$dir/run") {
        mkdir "$dir/run" or die "can't create directory '$dir/run': $!";
    }
    unless (-d "$dir/logs") {
        mkdir "$dir/logs" or die "can't create directory '$dir/logs': $!";
    }

    my $classpath = $app_cfg->{classpath};
    $O{'jetty.port'} //= _beta_port();
    $O{'jetty.ssl_port'} //= 0;
    $O{'jetty.thread_pool.max_threads'} //= 20;
    $O{'yandex.environment.type'} //= $conf;
    $O{'LOG_ROOT'} //= "$dir/logs";
    $O{'beta.url'} //= "http://"._beta_hostname();
    $O{'java.library.path'} //= $classpath;

    my $beta_settings = _beta_settings();
    my @java_args = (
        ($beta_settings->{java_agent_path} ? "-agentpath:$beta_settings->{java_agent_path}" : ()),
        ($classpath ? ('-cp' => "$classpath/*:") : ()),
        (map { "-D$_" . '=' . $O{$_} } keys %O),
        "-Xmx2g",
        "-Xms512m",
        "-XX:ParallelGCThreads=2",
        "-XX:CICompilerCount=2",
        );

    if (my $props = $beta_settings->{java_properties}) {
        croak "java_properties in $BETA_SETTINGS_FILE must be hash" unless ref $props eq 'HASH';
        push @java_args, map {yash_quote("-D$_=$props->{$_}")} keys %$props;
    }

    my @app_args = (
        "--log-configs-directory" => _beta_dir() . ($beta_settings->{java_resources_path} || "/arcadia/direct/libs-internal/config/src/main/resources/"),
        );

    my $java = $JAVA_PATH->{$REAL_JDK_VERSION}
            // die "Unsupported java version: $REAL_JDK_VERSION";

    my $java_cmd = join " ", map {yash_quote($_)} $java, @java_args, $app_cfg->{main_class}, @app_args;
    my $cmd;
    if ($detach) {
        my $start_daemon_cmd = join ' ', 'start-stop-daemon', '--start', '--chdir', $dir, '--pidfile', "$dir/run/$name.pid", '--background', '--verbose', '--make-pidfile',
        '--exec', '/bin/bash', '--', '-c', "'exec $java_cmd >>$dir/logs/stdout 2>>$dir/logs/stderr'";
        $cmd = $start_daemon_cmd;
    } else {
        $cmd = "cd $dir && exec $java_cmd";
    }

    # подргужаем серкереты для канваса
    local $ENV{'MONGODB_CONNECTION_STRING'} = _get_canvas_mongo_conn_string();
    local $ENV{'DIRECT_AUTH_TOKEN'} = _get_direct_token('direct_auth_token_canvas-test');
    local $ENV{'INTERNAL_API_TOKEN'} = _get_direct_token('internal_api_token_canvas-test');
    local $ENV{'FILE_API_INTERNAL_TOKEN'} = _get_direct_token('file_api_internal_token_canvas-test');
    local $ENV{'MDS_AUTH_TOKEN'} = _get_direct_token('mds_canvas-test');
    local $ENV{'MONITORING_TOKEN'} = _get_direct_token('monitoring_canvas-test');
    local $ENV{'SANDBOX_HOOK_SECRET'} = _get_direct_token('sandbox_hook_secret_canvas-test');
    local $ENV{'SANDBOX_OAUTH_TOKEN'} = _get_direct_token('sandbox_oauth_token_canvas-test');
    # конец секретов

    yash_system('ulimit -v 100000000 && ' . $cmd);
}

sub _stop_java_service {
    my ($name) = @_;
    my $user = [getpwuid($<)]->[0];
    my $port = _beta_port();
    my $pidfile = _beta_dir() . "/java/$JAVA_APP_CFG{$name}{path}/run/$name.pid";
    yash_system('start-stop-daemon', '--stop', '--pidfile', $pidfile, '--retry', 'TERM/30', '--oknodo', '--verbose');
    unlink $pidfile;
    my $cmd = "pgrep -u $user -f 'java.*Dbeta.url=http://$port.beta'";
    my @pids = map { chomp $_; $_ } `$cmd`;
    if (@pids) {
        warn "failed to stop java service $name property, initialting kill command on pid, please check if it works, otherwise fix the application";
        warn "gracefully terminating pids: ", join(', ', @pids) . " might take a while for app to actually stop";
        kill 'TERM', @pids;
    }
}

sub canvas_stop {
    my $canvas_service_dir = _beta_dir() . "/beta-services/canvas";
    if (-d $canvas_service_dir) {
        yash_system('beta-ctl', 'stop', 'canvas');
    }
}

sub messages_log {
    my ($args) = @_;

    my ($FOLLOW, $PRETTY, $HOST, $DATE);
    my ($INVERT, $CASE_INSENSITIVE);
    Getopt::Long::GetOptionsFromArray($args,
        'h|help' => sub {
            print "Usage: mk messages-log -- OPTIONS PATTERN\n"
                ."    mk messages-log -- bsClientData --pretty --follow | format-direct-log\n"
                ."Options:\n"
                ." -h, --help - show help message\n"
                ." -f, --follow - start tail -f instead of cat\n"
                ." -p, --pretty - strip unuseless fields of messages log\n"
                ." --host= - host substring for filter, f.e. ppcscripts\n"
                ." --date= - date, format YYYY-MM-DD or YYYYMMDD, default - today\n"
                ."Options passed directly to grep: -i -v\n"
                ;
            exit;
        },
        'f|follow' => \$FOLLOW,
        'p|pretty' => \$PRETTY,
        'd|date=s' => \$DATE,
        'host=s' => \$HOST,

        # grep options
        'v' => \$INVERT,
        'i' => \$CASE_INSENSITIVE,
        ) || die "can't parse options, stop";

    if (@$args > 1) {
        die "Supported only one argument - filter pattern";
    }

    if (defined $HOST && $HOST !~ /^[\w\.\-_]+$/) {
        die "Incorrect host pattern: '$HOST'";
    }

    if ($DATE && $FOLLOW) {
        die "Incomatible options: --date & --follow";
    }

    my $date;
    if (!defined $DATE) {
        $date = strftime("%Y%m%d", localtime);
    } elsif ($DATE =~ /^(\d\d\d\d)-?(\d\d)-?(\d\d)$/) {
        $date = "$1$2$3";
    } else {
        die "Incorrect date: '$DATE'";
    }
    my $date_ym = substr($date, 0, 6);

    my @files;
    if ($HOST) {
        @files = sort(map {glob} map {"/mnt/remote-log-rfs/*$HOST*/yandex/direct-production/$_"} "$date_ym/messages.$date.gz", "messages.$date");
    } else {
        @files = grep {-f} map {"/var/log/yandex/direct-production/$_"} "$date_ym/messages.$date.gz", "messages.$date";
    }
    if (!@files) {
        die "Log files don't exists";
    }

    my $cmd = ($FOLLOW ? "tail -f" : "zcat --force")." ".join(" ", map {yash_quote($_)} @files);

    if (!$HOST) {
        my $host_pattern = _beta_port().".".Net::Domain::hostfqdn();
        $cmd .= " | grep --line-buffered -- ".yash_quote($host_pattern);
    }

    if (@$args > 0) {
        my $opts = join " ",
                ($INVERT ? "-v" : ()),
                ($CASE_INSENSITIVE ? "-i" : ()),
        ;
        $cmd .= " | grep --line-buffered $opts -- ".yash_quote($args->[0]);
    }
    if ($PRETTY) {
        $cmd .= " | perl -lne ".yash_quote(q!BEGIN {$|++;} my ($dt, $meta, $data) = split /\s+/, $_, 3; my @meta = split /,/, $meta; print "$dt $meta[1] $data"!);
    } else {
        # чтобы от грепа не остался exit code > 0 - ничего не проматчилось
        $cmd .= " | cat";
    }
    system($cmd)
        && die "Can't run '$cmd': $!"
}

sub _java_apps {
    return @{_beta_settings()->{java_apps} // []};
}

sub _dna_enabled {
    return (_beta_settings()->{dna_enabled} // 'false') eq 'true';
}

sub _configuration {
    return yash_qx(perl => -I => 'perl/settings/', -e => 'use Settings; print $Settings::CONFIGURATION');
}

sub _beta_settings {
    if (-f $BETA_SETTINGS_FILE) {
        return from_json(path($BETA_SETTINGS_FILE)->slurp_utf8);
    }
    return {};
}

sub _get_canvas_mongo_conn_string {
    _beta_settings()->{canvas_mongo_string} // _get_direct_token("mongo_canvas-devtest")
}

sub _get_direct_token {
    my $token_name = shift;
    my $token_file_name = "/etc/direct-tokens/$token_name";
    if (!-f $token_file_name) {
        die "no file name $token_file_name"
    } else {
        my $token = path($token_file_name)->slurp_utf8;
        chomp $token;
        return $token;
    }
}

sub _sb_cli() {
    return _beta_dir() . '/sandbox_cli.py';
}

sub _update_sandbox_cli {
    open my $sb_cli_fh, '-|', 'svn', 'cat', 'svn+ssh://arcadia.yandex.ru/arc/trunk/arcadia/direct/bin/sandbox_cli.py' or die "can't svn cat ya: $!";
    open my $fh, '>', _sb_cli();
    print $fh $_ while (<$sb_cli_fh>);
    close $fh;
    close $sb_cli_fh;

    yash_system('chmod', '+x', _sb_cli());
}

1;
