use warnings;
use strict;

=head1 NAME

RBACElementary -- элементарный интерфейс к rbac

=head1 DESCRIPTION

Не знает ничего про бизнес-логику Директа, 
только про структура данных в rbac'е.

Не должен зависеть ни от чего.

=cut

package RBACElementary;

use Yandex::DBTools;
use Yandex::DBShards;

use base qw/Exporter/;

use Settings;
use Rbac qw/:const/;

our @EXPORT = qw(
    rbac_who_is
    rbac_multi_who_is
    rbac_who_is_detailed
    rbac_multi_who_is_detailed
    rbac_is_internal_user
    rbac_is_internal_ad_role

    rbac_is_client_chief_rep
    rbac_multi_is_client_chief_rep
    rbac_get_agency_rep_type
    rbac_get_agencies_rep_types
    rbac_is_agency_chief_rep

    rbac_get_chief_rep_of_client_rep
    rbac_get_chief_rep_of_client
    rbac_get_chief_reps_of_clients
    rbac_get_chief_reps_of_client_reps
    rbac_get_main_reps_of_client
    rbac_get_main_reps_of_clients

    rbac_get_main_reps_of_agency
    rbac_get_chief_rep_of_agency
    rbac_get_chief_rep_of_agency_rep
    rbac_get_chief_reps_of_agencies
    rbac_get_limited_reps_of_agency


    rbac_get_client_clientid_by_uid
    rbac_get_client_clientids_by_uids
    rbac_get_client_uids_by_clientid

    rbac_get_agency_clientid_by_uid
    rbac_get_agencies_clientids_by_uids
    
);


# внутрение роли
our %IS_INTERNAL_ROLE = map {$_ => 1} (
    $ROLE_SUPER,
    $ROLE_SUPERREADER,
    $ROLE_PLACER,
    $ROLE_MEDIA,
    $ROLE_SUPPORT,
    $ROLE_MANAGER,
);

# роли операторов внутренней рекламы (aka Банана)
our %IS_INTERNAL_AD_ROLE = map {$_ => 1} (
    $ROLE_INTERNAL_AD_ADMIN,
    $ROLE_INTERNAL_AD_MANAGER,
    $ROLE_INTERNAL_AD_SUPERREADER,
);

=head2 rbac_who_is

  $role_str = rbac_who_is($rbac, $uid);
  $role_str is one of - 'super', 'placer', 'manager', 'agency', 'media', 'superreader', 'support', 'client', 'empty'

=cut

sub rbac_who_is
{
    my (undef, $uid) = @_;
    return rbac_multi_who_is(undef, [$uid])->{$uid};
}

=head2 rbac_multi_who_is

    $role_hash = rbac_multi_who_is($rbac, [$uid1, $uid2]);
    # $role_hash = {uid1 => role1, uid2 => role2, ...}

=cut

sub rbac_multi_who_is
{
    my (undef, $uids) = @_;

    my %ret;

    my @uids = @$uids;
    return \%ret if !@uids;    
    
    my $ret_detailed = rbac_multi_who_is_detailed(undef, $uids);
    for my $uid (keys %$ret_detailed) {
        $ret{$uid} = $ret_detailed->{$uid}->{role};
    }
    return \%ret;
}

=head2 rbac_who_is_detailed

    my $res = rbac_who_is_detailed($rbac, $cl_uid);
    на вход получает rbac и uid
    возвращает хеш вида
    {
        is_teamleader => 1,
        is_superteamleader => 1,
        is_any_teamleader => 1,
        ...
        role => 'manager'
    }
    
=cut

sub rbac_who_is_detailed($$) {
    my (undef, $uid) = @_;
    return rbac_multi_who_is_detailed(undef, [$uid])->{$uid};
}

=head2 rbac_multi_who_is_detailed
    
    my $res = rbac_multi_who_is_detailed($rbac, [$uid1, $uid2, .... $uid5]);
    на вход получает rbac и uidы
    возвращает хеш вида
    {
        uid1 => {
            is_teamleader => 1,
            is_superteamleader => 1,
            is_any_teamleader => 1,
            ...
            role => 'manager'
        },
        ...
        uid5 => {
            is_teamleader => 1,
            is_superteamleader => 1,
            is_any_teamleader => 1,
            ...
            role => 'manager'
        }
    }    

=cut

sub rbac_multi_who_is_detailed {
    my (undef, $uids) = @_;

    my $uid2perminfo = Rbac::get_key2perminfo(uid => $uids);

    my %ret;
    foreach my $uid (@$uids) {
         $ret{$uid} = rbac_get_who_is_hash_by_ppc($uid2perminfo->{$uid});
    }    
    return \%ret;
}

=head2 rbac_get_who_is_hash_by_ppc
    
    my $ret = rbac_get_who_is_hash_by_ppc($row);
    получает на вход хэш с данными из ppc: u.uid, u.rep_type, c.role, c.subrole, c.chief_uid
    возвращает хеш вида
    {
        is_teamleader => 1,
        is_superteamleader => 1,
        is_any_teamleader => 1,
        ...
        role => 'manager'
    }

=cut

sub rbac_get_who_is_hash_by_ppc {

    my $data = shift;

    my $role = $data->{role} // $ROLE_EMPTY;
    my $subrole = $data->{subrole} // '';
    my $rep_type = $data->{rep_type} // '';

    my $res = {
        role => $role,
    };

    $res->{is_teamleader} = 1 if $subrole eq $SUBROLE_TEAMLEADER;
    $res->{is_superteamleader} = 1 if $subrole eq $SUBROLE_SUPERTEAMLEADER;

    $res->{is_any_teamleader} = 1 if $res->{is_teamleader} || $res->{is_superteamleader};

    $res->{is_super_placer} = 1 if $subrole eq $SUBROLE_SUPERPLACER;
    $res->{is_super_media_planner} = 1 if $subrole eq $SUBROLE_SUPERMEDIA;
    
    $res->{is_any_client} = 1 if $role eq $ROLE_CLIENT;
       
    if ($role eq $ROLE_AGENCY) {
        $res->{is_agency_chief} = 1 if  $rep_type eq $REP_CHIEF;
        $res->{is_agency_main} = 1 if $rep_type eq $REP_MAIN;
        $res->{is_agency_limited} = 1 if $rep_type eq $REP_LIMITED;
    }

    $res->{is_client_chief} = 1 if $role eq $ROLE_EMPTY || $role eq $ROLE_CLIENT && $rep_type eq $REP_CHIEF;
    
    $res->{is_internal_user} = rbac_is_internal_user(undef, role => $res->{role});

    return $res;
}

=head2 rbac_is_internal_user 

  Проверка, является ли пользователь с данным uid внутренним

  rbac_is_internal_user($rbac, role => 'manager') # определение роли
  rbac_is_internal_user($rbac, uid => 12345) # определение логина

  0 - внешний пользователь
  1 - внутренний пользователь

=cut

sub rbac_is_internal_user 
{
    my (undef, %OPT) = @_;
    
    my $role = $OPT{role} || ($OPT{uid} ? rbac_who_is(undef, $OPT{uid}) : die "UID doesn't specified");

    return $IS_INTERNAL_ROLE{$role} || 0;
}

=head2 rbac_is_internal_ad_role

  Проверка, является ли роль внутренней рекламы

  0 - нет
  1 - да, роль внутренней рекламы

=cut

sub rbac_is_internal_ad_role
{
    my ($role) = @_;

    die "role doesn't specified" if !$role;

    return $IS_INTERNAL_AD_ROLE{$role} || 0;
}

#======================================================================
# работа с представителями
#======================================================================

=head2 rbac_is_client_chief_rep

  $is_chief = rbac_is_client_chief_rep($client_uid);

=cut

sub rbac_is_client_chief_rep($)
{
    my ($client_uid) = @_;

    return rbac_multi_is_client_chief_rep([$client_uid])->{$client_uid};
}

=head2 rbac_multi_is_client_chief_rep

    my $uid2is_chief = rbac_multi_is_client_chief_rep([$uid1, $uid2, ...]);
    $uid2is_chief => {
        $uid1 => 1,
        $uid2 => 0,
        ...
    }

=cut

sub rbac_multi_is_client_chief_rep($)
{
    my ($client_uids) = @_;

    my $uid2info = Rbac::get_key2perminfo(uid => $client_uids);
    my %ret;
    for my $uid (@$client_uids) {
        my $info = $uid2info->{$uid};
        my $role = $info->{role} // '';
        my $rep_type = $info->{rep_type} // '';
        $ret{$uid} = $role eq $ROLE_CLIENT && $rep_type eq $REP_CHIEF;
    }

    return \%ret;
}

=head2 rbac_get_agency_rep_type

  $agency_rep_type = rbac_get_agency_rep_type($agency_uid);

  'chief'
  'main'
  'limited'

=cut

sub rbac_get_agency_rep_type($)
{
    my ($agency_uid) = @_;

    my $res = rbac_get_agencies_rep_types([$agency_uid]);
    
    return $res->{$agency_uid};
}

=head2 rbac_get_agencies_rep_types

    my $hash = rbac_get_agencies_rep_types($agencies_uids);
    
    Check rep types for agencies. Possible rep type values - chief, main, limited
    
    Result:
        $hash = { uid1 => 'main', uid2 => 'chief', uid3 => 'limited' } or undef

=cut

sub rbac_get_agencies_rep_types($)
{
    my ($agencies_uids) = @_;

    my $uid2info = Rbac::get_key2perminfo(uid => $agencies_uids);

    my %ret;
    foreach my $uid (@$agencies_uids) {
        my $info = $uid2info->{$uid};
        $ret{$uid} = $info->{role} && $info->{role} eq $ROLE_AGENCY ? $info->{rep_type} : '';
    }

    return \%ret;
}

=head2 rbac_is_agency_chief_rep

  $is_chief = rbac_is_agency_chief_rep($agency_uid);

=cut

sub rbac_is_agency_chief_rep($)
{
    my ($agency_uid) = @_;
    my $info = Rbac::get_perminfo(uid => $agency_uid);
    return $info->{role} eq $ROLE_AGENCY && $info->{rep_type} eq $REP_CHIEF ? 1 : 0;
}

=head2 rbac_get_chief_rep_of_client_rep

  $client_chief_uid = rbac_get_chief_rep_of_client_rep($uid);

=cut

sub rbac_get_chief_rep_of_client_rep($)
{
    my ($uid) = @_;

    my $perminfo = Rbac::get_perminfo(uid => $uid);
    if (!$perminfo || $perminfo->{role} ne $ROLE_CLIENT) {
        return $uid;
    } else {        
        return $perminfo->{chief_uid} || $uid;
    }
}

=head2 rbac_get_chief_rep_of_client

  $client_chief_uid = rbac_get_chief_rep_of_client($ClientID);

=cut

sub rbac_get_chief_rep_of_client($)
{
    my ($client_id) = @_;

    my $perminfo = Rbac::get_perminfo(ClientID => $client_id);
    if (!$perminfo) {
        return undef;
    } elsif ($perminfo->{role} ne $ROLE_CLIENT) {
        return undef;
    } else {        
        return $perminfo->{chief_uid};
    }
}

=head2 rbac_get_chief_reps_of_clients

  $client_chief_uid = rbac_get_chief_reps_of_clients($ClientID_list);

  returns hashref : {
        ClientID_01 => uid_01,
        ClientID_02 => uid_02,
        ...
  }

=cut

sub rbac_get_chief_reps_of_clients($){
    my ($ClientID_list) = @_;

    return Rbac::get_chiefs_multi(ClientID => $ClientID_list, role => $ROLE_CLIENT);
}

=head2 rbac_get_chief_reps_of_client_reps(clients_uids_array_ref)

    my $hash_ref_uids_to_chiefs = rbac_get_chief_reps_of_client_reps($clients_uids)

=cut

sub rbac_get_chief_reps_of_client_reps($)
{
    my ($client_uids) = @_;

    return Rbac::get_chiefs_multi(uid => $client_uids, role => $ROLE_CLIENT);
}

=head2 rbac_get_main_reps_of_client

  $main_reps_of_client_arr_ref = rbac_get_main_reps_of_client($ClientID);

=cut

sub rbac_get_main_reps_of_client($)
{
    my ($client_id) = @_;

    return Rbac::get_reps(ClientID => $client_id, role => $ROLE_CLIENT, rep_type => [$REP_CHIEF, $REP_MAIN, $REP_READONLY]);
}

=head2 rbac_get_main_reps_of_clients

    $main_reps_of_clients = rbac_get_main_reps_of_clients($client_ids)

=cut

sub rbac_get_main_reps_of_clients($)
{
    my ($client_ids) = @_;

    return Rbac::get_reps_multi(ClientID => $client_ids, role => $ROLE_CLIENT, rep_type => [$REP_CHIEF, $REP_MAIN, $REP_READONLY]);
}

=head2 rbac_get_main_reps_of_agency

  $main_reps_of_agency_arr_ref = rbac_get_main_reps_of_agency($ClientID);

=cut

sub rbac_get_main_reps_of_agency($)
{
    my ($client_id) = @_;

    return Rbac::get_reps(ClientID => $client_id, role => $ROLE_AGENCY, rep_type => [$REP_CHIEF, $REP_MAIN]);
}

=head2 rbac_get_chief_rep_of_agency

  $chief_reps_of_agency_arr_ref = rbac_get_chief_rep_of_agency($ClientID);

=cut

sub rbac_get_chief_rep_of_agency($)
{
    my ($client_id) = @_;

    return Rbac::get_chiefs_multi(ClientID => $client_id, role => $ROLE_AGENCY)->{$client_id};
}

=head2 rbac_get_chief_reps_of_agencies

  $chief_reps_of_agencies = rbac_get_chief_reps_of_agencies([$AgClientID1, $AgClientID2]);

  $chief_reps_of_agencies: {
      $AgClientID1 => ag_chief_uid1,
      $AgClientID2 => ag_chief_uid2
  }

=cut

sub rbac_get_chief_reps_of_agencies($)
{
    my ($client_ids) = @_;

    return Rbac::get_chiefs_multi(ClientID => $client_ids, role => $ROLE_AGENCY);
}

=head2 rbac_get_chief_rep_of_agency_rep

  $chief_rep_of_agency = rbac_get_chief_rep_of_agency_rep($agency_uid);

=cut

sub rbac_get_chief_rep_of_agency_rep($)
{
    my ($agency_uid) = @_;

    return Rbac::get_chiefs_multi(uid => $agency_uid, role => $ROLE_AGENCY)->{$agency_uid};
}

=head2 rbac_get_limited_reps_of_agency

  $limited_reps_of_agency_arr_ref = rbac_get_limited_reps_of_agency($ClientID);

=cut

sub rbac_get_limited_reps_of_agency($)
{
    my ($client_id) = @_;

    return Rbac::get_reps(ClientID => $client_id, role => $ROLE_AGENCY, rep_type => $REP_LIMITED);
}

#======================================================================
# uid <--> clientid
#======================================================================

=head2 rbac_get_client_clientid_by_uid

  my $ClientID = rbac_get_client_clientid_by_uid($uid);

=cut

sub rbac_get_client_clientid_by_uid($)
{
    my ($uid) = @_;

    return Rbac::get_rep2client([$uid], role => $ROLE_CLIENT)->{$uid};
}

=head2 rbac_get_client_clientids_by_uids

    my $hash = rbac_get_client_clientids_by_uids($uids);
    
    Result:
        $hash = { uid1 => client_id1, uid2 => client_id2 }

=cut

sub rbac_get_client_clientids_by_uids($)
{
    my ($uids) = @_;

    return Rbac::get_rep2client($uids, role => $ROLE_CLIENT);
}

=head2 rbac_get_client_uids_by_clientid

  my $uids_arrref = rbac_get_client_uids_by_clientid($ClientID);

=cut

sub rbac_get_client_uids_by_clientid($)
{
    my ($client_id) = @_;

    return Rbac::get_reps(ClientID => $client_id, role => $ROLE_CLIENT);
}

=head2 rbac_get_agency_clientid_by_uid

  my $ClientID = rbac_get_agency_clientid_by_uid($uid);

=cut

sub rbac_get_agency_clientid_by_uid($)
{
    my ($uid) = @_;

    return Rbac::get_rep2client([$uid], role => $ROLE_AGENCY)->{$uid}
}

=head2 rbac_get_agencies_clientids_by_uids

  my $hash = rbac_get_agencies_clientids_by_uids($agencies_uids);
  
  Result:
      $hash = { uid1 => client_id1, uid2 => client_id2, .. } or undef

=cut

sub rbac_get_agencies_clientids_by_uids($)
{
    my ($agencies_uids) = @_;
    
    return Rbac::get_rep2client($agencies_uids, role => $ROLE_AGENCY);
}

1;
