package Rbac;

=head1 NAME

    RBAC3 - работа с данными о ролях и правах, хранящихся в PPC

=cut

use Direct::Modern;

use List::Util qw/any/;
use List::MoreUtils qw/uniq/;
use JSON qw/from_json to_json/;
use Readonly;
use RbacSimpleCache;
use Storable qw/freeze thaw/;

use Yandex::DBShards;
use Yandex::DBTools;
use Yandex::HashUtils qw/hash_merge hash_map hash_grep/;
use Yandex::ListUtils qw/nsort xflatten xisect xminus/;

use Settings;

use base qw/Exporter/;
our %EXPORT_TAGS = (
    const => [ qw/
        $ROLE_EMPTY
        $ROLE_CLIENT
        $ROLE_AGENCY
        $ROLE_MANAGER
        $ROLE_SUPER
        $ROLE_SUPERREADER
        $ROLE_SUPPORT
        $ROLE_PLACER
        $ROLE_MEDIA
        $ROLE_INTERNAL_AD_ADMIN
        $ROLE_INTERNAL_AD_MANAGER
        $ROLE_INTERNAL_AD_SUPERREADER
        $ROLE_LIMITED_SUPPORT

        $REP_CHIEF
        $REP_MAIN
        $REP_LIMITED
        $REP_READONLY

        $LIM_REP_CHIEF
        $LIM_REP_MAIN
        $LIM_REP_LEGACY

        $SUBROLE_SUPERTEAMLEADER
        $SUBROLE_TEAMLEADER
        $SUBROLE_SUPERPLACER
        $SUBROLE_SUPERMEDIA

        $PERM_SUPER_SUBCLIENT
        $PERM_XLS_IMPORT
        $PERM_MASS_ADVQ
        $PERM_MONEY_TRANSFER
    / ],
);

our @EXPORT_OK = map {@$_} values %EXPORT_TAGS;
push @EXPORT_OK, qw/
    set_manager_supervisor
    get_manager_data
    mass_get_manager_data
    remove_manager_data
    remove_managers_data
    add_manager_in_hierarchy

    get_perminfo
    has_perm
    has_role
/;

Readonly our $ROLE_EMPTY => 'empty';
Readonly our $ROLE_CLIENT => 'client';
Readonly our $ROLE_AGENCY => 'agency';
Readonly our $ROLE_MANAGER => 'manager';
Readonly our $ROLE_SUPER => 'super';
Readonly our $ROLE_SUPERREADER => 'superreader';
Readonly our $ROLE_SUPPORT => 'support';
Readonly our $ROLE_PLACER => 'placer';
Readonly our $ROLE_MEDIA => 'media';
Readonly our $ROLE_INTERNAL_AD_ADMIN => 'internal_ad_admin';
Readonly our $ROLE_INTERNAL_AD_MANAGER => 'internal_ad_manager';
Readonly our $ROLE_INTERNAL_AD_SUPERREADER => 'internal_ad_superreader';
Readonly our $ROLE_LIMITED_SUPPORT => 'limited_support';

Readonly our $REP_CHIEF => 'chief';
Readonly our $REP_MAIN => 'main';
Readonly our $REP_LIMITED => 'limited';
Readonly our $REP_READONLY => 'readonly';

Readonly our $LIM_REP_CHIEF => 'chief';
Readonly our $LIM_REP_MAIN => 'main';
Readonly our $LIM_REP_LEGACY => 'legacy';

Readonly our $SUBROLE_SUPERTEAMLEADER => 'superteamleader';
Readonly our $SUBROLE_TEAMLEADER => 'teamleader';
Readonly our $SUBROLE_SUPERPLACER => 'superplacer';
Readonly our $SUBROLE_SUPERMEDIA => 'supermedia';

Readonly our $PERM_SUPER_SUBCLIENT => 'super_subclient';
Readonly our $PERM_XLS_IMPORT => 'xls_import';
Readonly our $PERM_MASS_ADVQ => 'mass_advq';
Readonly our $PERM_MONEY_TRANSFER => 'money_transfer';
Readonly my %VALID_PERMS = (
    $PERM_SUPER_SUBCLIENT => 1,
    $PERM_XLS_IMPORT => 1,
    $PERM_MASS_ADVQ => 1,
    $PERM_MONEY_TRANSFER => 1,
);

my $NO_SUPERVISOR = 0;
my $MAX_HIERARCHY_DEPTH = 2;

=head2 $CACHE_OPTIONS

    Ссылка на хэш с опциями для кеша
    Пока кешируются только роли, кеш сбрасывается в InitReq и при любом Handle

=cut
our $CACHE_OPTIONS ||= {
        size       => 10_000,
        ttl        => 10,
};

=head2 local $USE_CACHE = 1;

    Флаг, которым можно временно отключить использование кеша
    (может использоваться для улучшения производительности)

=cut
our $USE_CACHE = 1;


=head2 _cache()

    Формат данных "(ClientID|uid).123" => freeze(perminfo)

=cut
sub _cache() {
    state $cache;
    $cache //= RbacSimpleCache->new(%$CACHE_OPTIONS);
    return $cache;
}

=head2 clear_cache()

    Очистка кеша, безусловно выполняется в начале каждого запроса

=cut
sub clear_cache() {
    _cache->clear();
}

sub _role_checker($) {
    my ($role) = @_;
    my %roles_hash = $role
            ? map {$_ => 1} ref $role ? @$role : ($role)
            : ();

    return sub {
        my $_role = shift;
        return scalar !%roles_hash || $roles_hash{$_role}
    };
}

=head2 get_reps(uid|ClientID => $id, %opts)

    возвращает ссылку на массив с представителями клиента, подробности в get_reps_multi

=cut
sub get_reps($$%) {
    my ($key, $id, %opts) = @_;
    croak "Incorrect id: $id" if ref $id;
    return get_reps_multi($key, $id, %opts)->{$id} // [ ];
}

=head2 get_reps_multi(uid|ClientID => $ids, %opts) -> {id1 => [uid1, ...], ... }

    Возвращает ссылку на хэш с массивами представителей клиентов
    В opts могут дополнительно быть:
    rep_type - возвращать представителей только указанного типа
    role - возвращать представителей, только если роль клиента совдапает с заданной ролью

=cut
sub get_reps_multi($$%) {
    my ($key, $ids, %opts) = @_;
    my %where;
    if (my $rep_type = delete $opts{rep_type}) {
        $where{'r.rep_type'} = $rep_type;
    }
    if (my $role = delete $opts{role}) {
        $where{'cl.role'} = $role;
    }
    if (%opts) {
        croak "Unsupported options: ".to_json(\%opts);
    }

    my %ret;
    if ($key eq 'ClientID') {
        my $data = get_all_sql(PPC(ClientID => $ids), [
                        "SELECT cl.ClientID, r.uid as rep_uid
                           FROM clients cl
                                JOIN users as r ON r.ClientID = cl.ClientID
                        ", WHERE => {
                                %where,
                                'cl.ClientID' => SHARD_IDS,
                        } ]);

        for my $row (@$data) {
            push @{$ret{$row->{ClientID}}}, $row->{rep_uid};
        }
    } elsif ($key eq 'uid') {
        my $data = get_all_sql(PPC(uid => $ids), [
                        "SELECT u.uid, r.uid as rep_uid
                           FROM users u
                                JOIN clients cl on cl.ClientID = u.ClientID
                                JOIN users r ON r.ClientID = cl.ClientID
                        ", WHERE => {
                                %where,
                                'u.uid' => SHARD_IDS,
                        } ]);

        for my $row (@$data) {
            push @{$ret{$row->{uid}}}, $row->{rep_uid};
        }
    } else {
        croak "Incorrect key: $key";
    }
    return \%ret;
}

=head2 get_agency_reps_with_same_rights_multi($ids) -> {id1 => [uid1, ...], ... }

        Возвращает ссылку на хэш, где ключ - uid представителя агентства, а значение -
    массив uid-ов представителей этого агентства, которые обладают как минимум тем же
    уровнем прав на доступ к кампаниям. Массив включают так же uid того представителя,
    чей uid используется в качестве соответствующего ключа.

=cut
sub get_agency_reps_with_same_rights_multi {
    my ($uids) = @_;

    my $ret = get_reps_multi(uid => $uids, role => 'agency', rep_type => [qw/ chief main /]);

    for my $uid (keys %$ret) {
        $ret->{ $uid } = [ uniq $uid, @{ $ret->{ $uid } } ];
    }

    return $ret;
}

=head2 get_chiefs_multi(uid|ClientID => $ids, %opts) -> {id1 => chief_uid1, ... }

    Возвращает ссылку на хэш с uid-ами шефов
    В opts могут дополнительно быть:
    role - возвращать представителей, только если роль клиента совпадает с заданной ролью

=cut
sub get_chiefs_multi($$%) {
    my ($key, $ids, %opts) = @_;
    my $role_checker = _role_checker(delete $opts{role});
    if (%opts) {
        croak "Unsupported options: ".to_json(\%opts);
    }

    my $uid2info = get_key2perminfo($key => $ids);
    return hash_grep {$_}
            hash_map {$_->{chief_uid}}
                    hash_grep {$role_checker->($_->{role})}
                            $uid2info;
}

=head2 get_chief(uid|ClientID => $id, | role => $role) => $chief_uid | undef

    аналог get_chiefs_multi но для одного id

=cut
sub get_chief($$%) {
    my ($key, $id, %opts) = @_;
    return get_chiefs_multi($key, [$id], %opts)->{$id};
}

=head2 get_agency_uid_multi(ClientID|uid => [1,2,3])

    По ClientID или uid представителя получить uid ограниченного представителя агентства,
    работающего с клиентом

=cut
sub get_agency_uid_multi($$) {
    my ($key, $ids) = @_;
    my $rows;
    if ($key eq 'uid') {
        $rows = get_all_sql(PPC(uid => $ids), [
            "SELECT uid, agency_uid, 'new_schema' as lrs
             FROM users u
             JOIN agency_lim_rep_clients alrc ON alrc.ClientID = u.ClientID",
             WHERE => { 'uid' => SHARD_IDS },
             "UNION
             SELECT u.uid as uid, agency_uid, 'old_schema' as lrs
             FROM users u
             JOIN clients c ON c.ClientID = u.ClientID",
             WHERE => { 'u.uid' => SHARD_IDS }
        ]);
    } elsif ($key eq 'ClientID') {
        $rows = get_all_sql(PPC(ClientID => $ids), [
            "SELECT ClientID, agency_uid, 'new_schema' as lrs
             FROM agency_lim_rep_clients",
             WHERE => { ClientID => SHARD_IDS },
             "UNION
             SELECT ClientID, agency_uid, 'old_schema' as lrs
             FROM clients",
             WHERE => { ClientID => SHARD_IDS }
        ]);
    }


    my $result = {};
    foreach my $row (@$rows) {
        push (@{$result->{$row->{$key}}{$row->{lrs}}}, $row->{agency_uid}) if $row->{agency_uid};
    }

    return $result;
}

=head2 get_rep2client($uids, | role => $role)

    По uid-ам представителей получить ClientID (можно ограничиться определённой ролью)

=cut
sub get_rep2client($%) {
    my ($uids, %opts) = @_;
    my $role_checker = _role_checker(delete $opts{role});
    croak "Unsupported options: ".to_json(\%opts) if %opts;

    my $uid2info = get_key2perminfo(uid => $uids);
    return  hash_grep {$_}
            hash_map {$_->{ClientID}}
                    hash_grep {$role_checker->($_->{role})}
                            $uid2info;
}

=head2 get_key2perminfo(key => val)

    Получить информацию о правах клиентов.
    Внимание - используется кеш.

    Возможные значения key:
      ClientID, uid

    Результирующий хэш:
      - ClientID
      - chief_uid
      - role
      - subrole
      - agency_client_id
      - agency_uid
      - rep_type (только по uid)
      - perms - строка из $PERM_...

=cut
sub get_key2perminfo($$) {
    my ($key, $vals) = @_;
    $vals = [$vals] if !ref $vals;

    my $cached_res = { };
    my $todo = [ ];
    if ($USE_CACHE) {
        for my $id (@$vals) {
            if (my $v = _cache->get("$key.$id")) {
                $cached_res->{$id} = thaw($v);
            } else {
                push @$todo, $id;
            }
        }
    } else {
        $todo = $vals;
    }
    $todo = [nsort @$todo];

    my $db_res;
    if ($key eq 'ClientID') {
        $db_res = get_hashes_hash_sql(PPC(ClientID => $todo), [
                        "SELECT c.ClientID, c.role, c.subrole, c.chief_uid
                              , c.agency_client_id, c.agency_uid
                              , c.primary_manager_uid, c.primary_manager_set_by_idm
                              , c.perms
                           FROM clients c",
                        WHERE => { 'c.ClientID' => SHARD_IDS }
                ]);
        my $lim_reps_data = get_all_sql(PPC(ClientID => $todo), [
                'SELECT ClientID, agency_uid FROM agency_lim_rep_clients', WHERE => { ClientID => SHARD_IDS }
            ]);
        foreach my $lim_rep (@$lim_reps_data) {
            my $client_data = $db_res->{$lim_rep->{ClientID}};
            if ($client_data) {
                $client_data->{agency_uids} //= [];
                push @{$client_data->{agency_uids}}, $lim_rep->{agency_uid};
            }
        }
    } elsif ($key eq 'uid') {
        $db_res = get_hashes_hash_sql(PPC(uid => $todo), [
                        "SELECT u.uid, u.rep_type, c.ClientID, c.role, c.subrole, c.chief_uid
                              , c.agency_client_id, c.agency_uid
                              , c.primary_manager_uid, c.primary_manager_set_by_idm
                              , c.perms
                           FROM users u JOIN clients c on c.ClientID = u.ClientID",
                        WHERE => { 'u.uid' => SHARD_IDS }
                ]);
        my $lim_reps_data = get_all_sql(PPC(uid => $todo), [
                'SELECT uid, agency_uid FROM users u JOIN agency_lim_rep_clients alrc USING(ClientID)', WHERE => { uid => SHARD_IDS }
            ]);
        foreach my $lim_rep (@$lim_reps_data) {
            my $client_data = $db_res->{$lim_rep->{uid}};
            if ($client_data) {
                $client_data->{agency_uids} //= [];
                push @{$client_data->{agency_uids}}, $lim_rep->{agency_uid};
            }
        }

        my $mccs_data = get_all_sql(PPC(uid => $todo), [
                'SELECT uid, client_id_to client_id FROM users u JOIN reverse_clients_relations rcr ON rcr.client_id_from = u.ClientID', WHERE => { uid => SHARD_IDS, 'rcr.type' => 'mcc' }
            ]);
        foreach my $mcc_data (@$mccs_data) {
            my $client_data = $db_res->{$mcc_data->{uid}};
            if ($client_data) {
                $client_data->{mcc_client_ids} //= [];
                push @{$client_data->{mcc_client_ids}}, $mcc_data->{client_id};
            }
        }


        delete $_->{uid} for values %$db_res;
    } else {
        croak "Incorrect key in get_client2perminfo: $key";
    }

    if ($USE_CACHE) {
        _cache->set("$key.$_", freeze($db_res->{$_})) for keys %$db_res;
    }

    return hash_merge { }, $cached_res, $db_res;
}

=head2 get_perminfo(key => val)

    Получить информацию о правах клиента.
    Внимание - используется кеш.

    Возможные значения key:
      ClientID, uid

    Описание результата см. в get_key2perminfo

=cut
sub get_perminfo($$) {
    my ($key, $val) = @_;
    if ($key eq 'ClientID' || $key eq 'uid') {
        return get_key2perminfo($key, [ $val ])->{$val};
    } else {
        croak "Incorrect key in get_client2perminfo: $key";
    }
}

=head2 set_client_perm(ClientID, $PERM_... => 1|0, ...)

    Прописать на ClientID дополнительное право (или удалить право)

=cut
sub set_client_perm($%) {
    my ($ClientID, %perms) = @_;
    return unless %perms;

    for my $perm (keys %perms) {
        croak "Unknown permission: '$perm'" unless $VALID_PERMS{$perm};
    }
    do_update_table(PPC(ClientID => $ClientID), 'clients',
                    {perms__smod => \%perms},
                    where => {ClientID => $ClientID});
    clear_cache();
}

=head2 has_perm($perminfo, $perm)

    Проверить, есть ли cоответствующая роль в структуре perminfo

=cut
sub has_perm($$) {
    my ($perminfo, $perm) = @_;
    return $perminfo && $perminfo->{perms} && (any {$_ eq $perm} split /,/, $perminfo->{perms}) ? 1 : 0;
}

=head2 has_role($perminfo, $ROLE...)

    Проверить, есть ли соответствующая роль в структуре perminfo

=cut
sub has_role($$) {
    my ($perminfo, $role) = @_;
    return $perminfo && ($perminfo->{role}//'') eq $role ? 1 : 0;
}

=head2 get_client_agency_uid(ClientID, $agency_uid)

    если клиент привязан к указанному агентству - выбрать того ограниченного представителя, к которому привязан,
    иначе вернуть uid переданного агентства

    удобство на время переходного периода, кажется потом функция такого формата должна быть не нужна

=cut
sub get_client_agency_uid($$) {
    my ($ClientID, $agency_uid) = @_;
    my $perminfo = Rbac::get_perminfo(ClientID => $ClientID);
    my $ag_perminfo = Rbac::get_perminfo(uid => $agency_uid);
    if ($ag_perminfo->{role} ne $ROLE_AGENCY) {
        print STDERR "rbac2:get_client_agency_uid - not a agency, client=$ClientID, agency_uid=$agency_uid";
    } elsif ($perminfo->{agency_client_id} != $ag_perminfo->{ClientID}) {
        print STDERR "rbac2:get_client_agency_uid - bibded to another agency, client=$ClientID, agency_uid=$agency_uid";
    }
    return $perminfo->{agency_uid} && $perminfo->{agency_client_id} == $ag_perminfo->{ClientID} ? $perminfo->{agency_uid} : $agency_uid;
}

=head2 set_manager_supervisor($manager_client_id, $supervisor_client_id)

    Прописать менеджеру непосредственного начальника и список всех начальников вверх по иерархии,
    а также добавить менеджера в списки подчиненных менеджеров для всех его начальников в иерархии.

    Если $supervisor_client_id undef - удалить начальника, очистить список начальников и исключить
    менеджера из списков подчиненных менеджеров для всех начальников вверх по иерархии

=cut

sub set_manager_supervisor {
    my ( $manager_client_id, $new_supervisor_client_id ) = @_;

    croak "No manager ClientID given!" unless $manager_client_id;

    my $current_managers_info = mass_get_manager_data([ grep { $_ } $manager_client_id, $new_supervisor_client_id ]);

    my $manager_info = $current_managers_info->{ $manager_client_id };
    croak "No info for manager with ClientID $manager_client_id found!" unless $manager_info;

    # подчиненным может быть только простой менеджер или супертимлидер
    croak "Superteamleader can not has supervisor"
        if $manager_info->{subrole} and $manager_info->{subrole} eq 'superteamleader';

    my $new_chiefs_client_id = [];
    if ( $new_supervisor_client_id ) {
        my $new_supervisor_info = $current_managers_info->{ $new_supervisor_client_id };
        croak "No info for manager with ClientID $new_supervisor_client_id found!" unless $new_supervisor_info;

        # supervisor-ом могут быть только тимлидер или супертимлидер, после установки нового
        # начальника глубина иерархии не должна стать глубже чем $MAX_HIERARCHY_DEPTH
        croak "Only superteamleader or teamleader can be supervisor"
            if ! $new_supervisor_info->{subrole} or $new_supervisor_info->{chiefs_cnt} + 1 > $MAX_HIERARCHY_DEPTH;

        $new_chiefs_client_id = $new_supervisor_info->{chiefs_client_id};
        push @$new_chiefs_client_id, $new_supervisor_client_id;
    }

    my $old_chiefs_client_id   = $manager_info->{chiefs_client_id};
    my $subordinates_client_id = $manager_info->{subordinates_client_id};

    my $related_managers_info = {};
    if ( @$new_chiefs_client_id || @$old_chiefs_client_id || @$subordinates_client_id) {
        $related_managers_info = mass_get_manager_data([ uniq @$new_chiefs_client_id, @$old_chiefs_client_id, @$subordinates_client_id ]);
    }

    my $values = _calc_changes( $manager_client_id, $new_supervisor_client_id, { %$current_managers_info, %$related_managers_info } );

    _pack_json_fields( $values );

    my $updated;
    do_in_transaction {
        $updated = do_mass_update_sql( PPC(ClientID => [ keys %$values ]), 'manager_hierarchy', 'manager_client_id', $values );
    };

    # количество обновленных строк должно быть равно количеству менеджеров, чьи данные хотели обновить
    return 0 if $updated != scalar( keys %$values );

    return 1;
}

{
    my @fields = qw/
        chiefs_client_id
        chiefs_uid
        subordinates_client_id
        subordinates_uid
    /;

    sub _pack_json_fields {
        my ( $results ) = @_;

        for my $row ( values %$results ) {
            for my $field ( @fields ) {
                next unless exists $row->{ $field };
                my $val = $row->{ $field };
                $row->{ $field } = ($val && @$val) ? to_json([ map { int } @$val ]) : undef;
            }
        }

        return;
    }

    sub _unpack_json_fields {
        my ( $results ) = @_;

        for my $row ( values %$results ) {
            for my $field ( @fields ) {
                my $val = $row->{ $field };
                $row->{ $field } = $val ? from_json( $val ) : [];
            }
        }

        return;
    }
}

sub _calc_changes {
    my ( $manager_client_id, $new_supervisor_client_id, $all_managers_info ) = @_;

    my $manager_info = $all_managers_info->{ $manager_client_id };
    croak "No info for manager with ClientID $manager_client_id found!" unless $manager_info;

    my $manager_uid = $manager_info->{manager_uid};

    my $new_supervisor_uid;
    my $new_chiefs_client_id = [];
    my $new_chiefs_uid = [];
    if ( $new_supervisor_client_id ) {
        my $new_supervisor_info = $all_managers_info->{ $new_supervisor_client_id };
        croak "No info for manager with ClientID $new_supervisor_client_id found!" unless $new_supervisor_info;

        $new_supervisor_uid = $new_supervisor_info->{manager_uid};

        # список начальников менеджера формируем из списка начальников его нового
        # непосредственного начальника, добавив в него самого этого начальника
        $new_chiefs_client_id = $new_supervisor_info->{chiefs_client_id} // [];
        push @$new_chiefs_client_id, $new_supervisor_client_id;

        $new_chiefs_uid = $new_supervisor_info->{chiefs_uid} // [];
        push @$new_chiefs_uid, $new_supervisor_uid;
    }

    my $old_chiefs_client_id = $manager_info->{chiefs_client_id} // [];
    my $old_chiefs_uid       = $manager_info->{chiefs_uid} // [];

    my %values;

    # добавляем или меняем менеджеру начальника
    if ( $new_supervisor_client_id ) {
        $values{ $manager_client_id } = {
            supervisor_client_id => $new_supervisor_client_id,
            supervisor_uid       => $new_supervisor_uid,
            chiefs_client_id     => $new_chiefs_client_id,
            chiefs_uid           => $new_chiefs_uid,
        };
    }
    # удаляем непосредственного начальника у менеджера
    else {
        $values{ $manager_client_id } = {
            supervisor_client_id => $NO_SUPERVISOR,
            supervisor_uid       => $NO_SUPERVISOR,
            chiefs_client_id     => [],
            chiefs_uid           => [],
        };
    }

    # изменяем списки начальников у подчиненных менеджера
    my $subordinates_client_id = $manager_info->{subordinates_client_id} // [];
    my $subordinates_uid       = $manager_info->{subordinates_uid} // [];
    if ( @$subordinates_client_id ) {
        # включим самого менеджера в новые списки начальников для его подчиненных
        my @chiefs_client_id = ( @$new_chiefs_client_id, $manager_client_id );
        my @chiefs_uid       = ( @$new_chiefs_uid, $manager_uid );

        for my $client_id ( @$subordinates_client_id) {
            my $manager_info = $all_managers_info->{ $client_id };
            croak "No info for manager with ClientID $client_id found!" unless $manager_info;

            my $client_ids = $manager_info->{chiefs_client_id} // [];
            my $uids       = $manager_info->{chiefs_uid} // [];

            $values{ $client_id } = {
                chiefs_client_id => [ uniq @chiefs_client_id, xflatten( xminus( $client_ids, $old_chiefs_client_id ) ) ],
                chiefs_uid       => [ uniq @chiefs_uid, xflatten( xminus( $uids, $old_chiefs_uid ) ) ],
            };
        }
    }

    # запомним одинаковых руководителей в прежней и новой иерархиях, т.к. нет смысла изменять их данные
    my %common_chiefs_client_id = map { $_ => 1 } @{ xisect( $new_chiefs_client_id, $old_chiefs_client_id ) };

    # включим самого менеджера в списки подчиненных для исключения из списков прежних
    # и добавления в списки новых руководителей
    unshift @$subordinates_client_id, $manager_client_id;
    unshift @$subordinates_uid, $manager_uid;

    # исключаем менеджера и его подчинененных из списков подчиненных прежних руководителей
    for my $client_id ( @$old_chiefs_client_id ) {
        next if exists $common_chiefs_client_id{ $client_id };

        my $manager_info = $all_managers_info->{ $client_id };
        croak "No info for manager with ClientID $client_id found!" unless $manager_info;

        my $client_ids = $manager_info->{subordinates_client_id} // [];
        my $uids       = $manager_info->{subordinates_uid} // [];

        $values{ $client_id } = {
            subordinates_client_id => xminus( $client_ids, $subordinates_client_id ),
            subordinates_uid       => xminus( $uids, $subordinates_uid ),
        };
    }

    # добавляем менеджера и его подчиненных в списки подчиненных новых руководителей
    for my $client_id ( @$new_chiefs_client_id ) {
        next if exists $common_chiefs_client_id{ $client_id };

        my $manager_info = $all_managers_info->{ $client_id };
        croak "No info for manager with ClientID $client_id found!" unless $manager_info;

        my $client_ids = $manager_info->{subordinates_client_id} // [];
        my $uids       = $manager_info->{subordinates_uid} // [];

        $values{ $client_id } = {
            subordinates_client_id => [ uniq @$client_ids, @$subordinates_client_id ],
            subordinates_uid       => [ uniq @$uids, @$subordinates_uid ],
        };
    }

    return \%values;
}

=head2 get_manager_data

    Возвращает информацию о менеджере - саброль, шефов и подчиненных.

    $manager_info = get_manager_data($manager_client_id);

=cut

sub get_manager_data {
    my ( $client_id ) = @_;

    return unless $client_id;

    return mass_get_manager_data([ $client_id ])->{ $client_id };
}

=head2 mass_get_manager_data

    Возвращает информацию о менеджерах - саброль, шефов и подчиненных.

    $client_id2info = mass_get_manager_data([$manager_client_id1, $manager_client_id2, ...]);

=cut

sub mass_get_manager_data {
    my ($client_ids) = @_;

    return if ref $client_ids ne 'ARRAY' or ! scalar @$client_ids;

    my $results = get_hashes_hash_sql(PPC(ClientID => $client_ids), [
        'SELECT
            manager_client_id, manager_uid, supervisor_client_id, supervisor_uid,
            chiefs_client_id, chiefs_uid, subordinates_client_id, subordinates_uid,
            subrole, IF(chiefs_client_id IS NULL, 0, JSON_LENGTH(chiefs_client_id)) AS chiefs_cnt,
            IF(subordinates_client_id IS NULL, 0, JSON_LENGTH(subordinates_client_id)) AS subordinates_cnt
        FROM
            manager_hierarchy JOIN clients ON (manager_client_id = ClientID)',
        WHERE => { manager_client_id => SHARD_IDS }
    ]);

    _unpack_json_fields( $results );

    return $results;
}

=head2 remove_manager_data

    Удаляет данные о менеджерах из иерархии менеджеров.

    remove_manager_data([$manager_uid1, $manager_uid2, ...]);

=cut

sub remove_manager_data {
    my ($manager_uid) = @_;

    return remove_managers_data([ $manager_uid ]);
}

=head2 remove_managers_data

    Удаляет данные о менеджерах из иерархии менеджеров.

    remove_manager_data([$manager_uid1, $manager_uid2, ...]);

=cut

sub remove_managers_data {
    my ($manager_uids) = @_;

    return if ref $manager_uids ne 'ARRAY' or ! scalar @$manager_uids;

    my $uid2client_id = get_rep2client($manager_uids);

    # удаляем менеджеров из списка подчиненных для их руководителей
    # NB - код рассчитывает что из списков шефов менеджеры удалятся
    # при смене роли через Intapi::DostupUserManagement
    set_manager_supervisor($_, $NO_SUPERVISOR) for uniq values %$uid2client_id;

    return do_delete_from_table(PPC(uid => $manager_uids),
        'manager_hierarchy', where => { manager_uid => SHARD_IDS });
}

=head2 add_manager_in_hierarchy

    Добавляет данные о менеджере в иерархию менеджеров.

    add_manager_hierarchy( $manager_client_id, $manager_uid );

=cut

sub add_manager_in_hierarchy {
    my ($manager_client_id, $manager_uid) = @_;

    return unless $manager_client_id && $manager_uid;

    return do_insert_into_table(PPC(ClientID => $manager_client_id),
        'manager_hierarchy',
        {
            manager_client_id    => $manager_client_id,
            manager_uid          => $manager_uid,
            supervisor_client_id => $NO_SUPERVISOR,
            supervisor_uid       => $NO_SUPERVISOR,
        },
        ignore => 1,
        key => 'manager_client_id'
    );
}

=head2 get_client_managers_uids($ClientID)

    По ClientID клиента получить ClientID сервисирующих менеджеров

=cut
sub get_client_managers_uids($) {
    my ($ClientID) = @_;
    return get_one_column_sql(PPC(ClientID => $ClientID), "SELECT manager_uid FROM client_managers WHERE ClientID = ?", $ClientID);
}

=head2 get_agency_managers_uids($ClientID)

    По ClientID агентства получить ClientID сервисирующих менеджеров

=cut
sub get_agency_managers_uids($) {
    my ($ClientID) = @_;
    return get_one_column_sql(PPC(ClientID => $ClientID), "SELECT manager_uid FROM agency_managers WHERE agency_client_id = ?", $ClientID);
}

1;
