package RbacRolePerms;

use Direct::Modern;
use Rbac qw/:const has_perm/;

=head1 NAME

    RbacRolePerms - слепок отношения роль-тип_субъекта-право из RBAC2

=head1 DESCRIPTION

    Создавался хэш так:

    direct-sql -B -N pr:rbac:1 'select rolename, typename, pname from role_perm rp join roles r on r.roleid = rp.roleid join permissions p on p.pid = rp.pid join obj_types t on t.typeid = rp.otid order by 1,2,3' |
        perl -lane 'my ($role, $type, $perm) = @F; $role =~ s/^RBAC_//; $perm =~ s/^RBAC_//; push @{$S{$role}{$type}}, $perm; }{ use Data::Dumper; my $d = Data::Dumper->new([\%S], ["ROLE_PERMS"])->Sortkeys(1)->Indent(1); $d->{xpad} = " "x8; print $d->Dump()'

=cut

my ($ROLE_PERMS, $MAPPING);


sub _roles($) {
    my ($perminfo) = @_;
    return @{$MAPPING->{$perminfo->{role}//''}->{$perminfo->{subrole}//''}->{$perminfo->{rep_type}//''} || []};
}


=head2 system_perms(perminfo)

    получить системные права пользователя (те, которые в RBAC2 выдавались на System)

=cut
sub system_perms($) {
    my ($perminfo) = @_;

    my %ret;
    for my $rbac_role (_roles($perminfo)) {
        $ret{$_} = 1 for @{$ROLE_PERMS->{$rbac_role}->{System}};
    }

    return \%ret;
}


=head2 onclient_perms(operator_perminfo, client_perminfo)

    получить права оператора на клиента (те, которые в RBAC2 выдавались на ClientID)

=cut
sub onclient_perms($$$) {
    my ($UID, $operator, $client) = @_;

    my $O_CLIENT = $operator->{ClientID};
    my $O_CHIEF = $operator->{chief_uid};
    my $O_REPTYPE = $operator->{rep_type};

    my @operator_roles = _roles($operator);

    my @client_classes;
    if ($operator->{role} eq $ROLE_SUPER
            || $operator->{role} eq $ROLE_SUPERREADER
            || $operator->{role} eq $ROLE_SUPPORT
            || $operator->{role} eq $ROLE_PLACER 
        ) {
        if ($client->{role} eq $ROLE_CLIENT) {
            push @client_classes, "ThisClient", "ThisClientStructed", "ThisClientSimple";
            if ($client->{agency_client_id}) {
                push @client_classes, "SubClientsPriv";
                push @client_classes, 
                        has_perm($client, $PERM_SUPER_SUBCLIENT) 
                        ? "SupSubClients" : "SubClients";
                push @client_classes, "SSClientWithATM"
                    if has_perm($client, $PERM_MONEY_TRANSFER);
                push @client_classes, "SSClientWithAR"
                    if has_perm($client, $PERM_XLS_IMPORT);
            }
        }
    } elsif ($operator->{role} eq $ROLE_MEDIA) {
        # pass
    } elsif ($operator->{role} eq $ROLE_MANAGER) {
        if ($client->{role} eq $ROLE_CLIENT) {
            my %allowed_manager_uids = map {$_ => 1} $UID, @{Rbac::get_manager_data($O_CLIENT)->{subordinates_uid}};
            my $serv_managers = $client->{agency_client_id}
                ? Rbac::get_agency_managers_uids($client->{agency_client_id})
                : Rbac::get_client_managers_uids($client->{ClientID});
            if (grep {$allowed_manager_uids{$_}} @$serv_managers) {
                push @client_classes, "ThisClient", "ThisClientStructed", "ThisClientSimple";
                if ($client->{agency_client_id}) {
                    push @client_classes, "SubClientsPriv";
                    push @client_classes, 
                        has_perm($client, $PERM_SUPER_SUBCLIENT) 
                        ? "SupSubClients" : "SubClients";
                    push @client_classes, "SSClientWithATM"
                        if has_perm($client, $PERM_MONEY_TRANSFER);
                    push @client_classes, "SSClientWithAR"
                        if has_perm($client, $PERM_XLS_IMPORT);
                } else {
                    if (grep {$_==$UID} @$serv_managers) {
                        push @client_classes, "ServicedClients";                   
                    }
                }
            }
        } elsif ($client->{role} eq $ROLE_AGENCY) {
            my %allowed_manager_uids = map {$_ => 1} $UID, @{Rbac::get_manager_data($O_CLIENT)->{subordinates_uid}};
            my $serv_managers = Rbac::get_agency_managers_uids($client->{ClientID});
            if (grep {$allowed_manager_uids{$_}} @$serv_managers) {
                push @client_classes, "ThisAgency";
                if (grep {$_==$UID} @$serv_managers) {
                    push @client_classes, "ManagedAgencies";
                }
            }
        }
    } elsif ($operator->{role} eq $ROLE_AGENCY) {
        if ($client->{role} eq $ROLE_CLIENT 
            && $client->{agency_client_id} == $O_CLIENT
            && ($O_REPTYPE eq $REP_CHIEF || $O_REPTYPE eq $REP_MAIN || $client->{agency_uid} == $UID)
        ) {
            push @client_classes, "ThisClient";
            push @client_classes, "SubClientsPriv"
                if $client->{agency_uid} == $O_CHIEF || $O_REPTYPE eq $REP_LIMITED;
            push @client_classes, 
                    has_perm($client, $PERM_SUPER_SUBCLIENT) 
                    ? "SupSubClients" : "SubClients";
            push @client_classes, "SSClientWithATM"
                if has_perm($client, $PERM_MONEY_TRANSFER);
            push @client_classes, "SSClientWithAR"
                if has_perm($client, $PERM_XLS_IMPORT);
        } elsif ($client->{role} eq $ROLE_AGENCY) {
            push @client_classes, "ThisAgency";
            push @client_classes, "ThisAgencyMass" 
                if has_perm($client, $PERM_MASS_ADVQ); 
        }
    } elsif ($operator->{role} eq $ROLE_CLIENT) {
        if ($O_CLIENT == $client->{ClientID}) {
            push @client_classes, "ThisClient";
        }
    }

    my %ret;
    for my $operator_role (@operator_roles) {
        for my $client_class (@client_classes) {
            $ret{$_} = 1 for @{$ROLE_PERMS->{$operator_role}->{$client_class}};
        }
    }

    return \%ret;
}

# role -> subrole -> rep_type -> RbacRole
$MAPPING = {
    'super' => {
        '' => {
            'chief' => ['Ubermensch'],
        },
    },
    'support' => {
        '' => {
            'chief' => ['Support'],
        },
    },
    'limited_support' => {
        '' => {
            'chief' => ['LimitedSupport'],
        },
    },
    'placer' => {
        '' => {
            'chief' => ['Placer'],
        },
        'superplacer' => {
            'chief' => ['Placer', 'SuperPlacer'],
        },
    }, 
    'superreader' => {
        '' => {
            'chief' => ['SuperReader'],
        },
    },
    'media' => {
        '' => {
            'chief' => ['Mediaplanner'],
        },
        'supermedia' => {
            'chief' => ['Mediaplanner'],
        },
    },
    'client' => {
        '' => {
            'chief' => ['SimpleClient', 'Client', 'ClientMainRep', 'ClientChiefRep'],
            'main' => ['SimpleClient', 'Client', 'ClientMainRep'],
        },
    },
    'agency' => {
        '' => {
            'chief' => ['Agency', 'AgencyChiefRep', 'AgencyMainRep'],
            'main' => ['Agency', 'AgencyMainRep'],
            'limited' => ['Agency', 'AgencyLimitedRep'],
        }
    },
    'manager' => {
        '' => {
            'chief' => ['ManagerPrivate', 'Manager'],
        },
        'teamleader' => {
            'chief' => ['ManagerPrivate', 'Manager', 'TeamLeader'],
        },
        'superteamleader' => {
            'chief' => ['ManagerPrivate', 'Manager', 'TeamLeader', 'SuperTeamLeader'],
        },
    },
    'internal_ad_admin' => {
        '' => {
            'chief' => ['InternalAdAdmin'],
        },
    },
};

$ROLE_PERMS = {
        'Agency' => {
                'System' => [
                        'CreateClient',
                        'DoneWarn',
                        'ListWarn',
                        'Login',
                        'ModifySubClient',
                        'SendModerate',
                        'ViewPdfReports',
                        'ViewUserDescription'
                ]
        },
        'AgencyChiefRep' => {
                'AgencyLimitedReps' => [
                        'ChangeAgencyLrepToMrep',
                        'DropAgencyRep',
                        'MoveAlinkToLrep',
                        'SwitchAgencyChiefRep'
                ],
                'AgencyMainReps' => [
                        'ChangeAgencyMrepToLrep',
                        'DropAgencyRep',
                        'SwitchAgencyChiefRep'
                ],
                'ThisAgency' => [
                        'CreateAgencyLimitedRep',
                        'CreateAgencyMainRep',
                        'MoveAlinkToMain'
                ]
        },
        'AgencyLimitedRep' => {
                'ACampaigns' => [
                        'AgencyAccess',
                        'DropAgencyCampaign',
                        'EditCamp',
                        'Lookup',
                        'ShowStat'
                ],
                'ACampaignsXLS' => [
                        'ImportXlsBanners'
                ],
                'ClientChiefRep' => [
                        'AgencyControlRep',
                        'CreateAgencyCampaign'
                ],
                'ClientLimitedRep' => [
                        'AgencyControlRep',
                        'CreateAgencyCampaign'
                ],
                'ClientMainRep' => [
                        'AgencyControlRep',
                        'CreateAgencyCampaign'
                ],
                'SACampaigns' => [
                        'AgencyAccess',
                        'DropAgencyCampaign',
                        'EditCamp',
                        'Lookup',
                        'ShowStat'
                ],
                'SSClientWithAR' => [
                        'RevokeRightsFromSSC'
                ],
                'SSClientWithATM' => [
                        'ForbidTransferMoney'
                ],
                'SubClients' => [
                        'AllowTransferMoney',
                        'Control',
                        'MakeSuperSubClient'
                ],
                'SubClientsPriv' => [
                        'AgencyControlClient'
                ],
                'SupSubClients' => [
                        'AddRightsToSSC',
                        'AllowTransferMoney',
                        'Control',
                        'MakeSimpleSubClient'
                ],
                'ThisAgency' => [
                        'AgencyControlAgencyLim',
                        'BindAgency',
                        'BindAgencySuper',
                        'CreateAgencyCampaign',
                        'UnbindAgency'
                ],
                'ThisAgencyMass' => [
                        'AllowMassADVQ'
                ],
                'ThisAgencyPPS' => [
                        'PayPiecesOfShows'
                ],
                'ThisAgencyXLS' => [
                        'AlowAgencyRepXlsImport'
                ]
        },
        'AgencyMainRep' => {
                'ACampaigns' => [
                        'AgencyAccess',
                        'DropAgencyCampaign',
                        'EditCamp',
                        'Lookup',
                        'ShowStat'
                ],
                'ACampaignsXLS' => [
                        'ImportXlsBanners'
                ],
                'ClientChiefRep' => [
                        'AgencyControlRep',
                        'CreateAgencyCampaign'
                ],
                'ClientLimitedRep' => [
                        'AgencyControlRep',
                        'ControlAgencyLimRep',
                        'CreateAgencyCampaign',
                        'DropClientRep'
                ],
                'ClientMainRep' => [
                        'AgencyControlRep',
                        'CreateAgencyCampaign',
                        'DropClientRep'
                ],
                'SACampaigns' => [
                        'AgencyAccess',
                        'DropAgencyCampaign',
                        'EditCamp',
                        'Lookup',
                        'ShowStat'
                ],
                'SSClientWithAR' => [
                        'RevokeRightsFromSSC'
                ],
                'SSClientWithATM' => [
                        'ForbidTransferMoney'
                ],
                'SubClients' => [
                        'AllowTransferMoney',
                        'Control',
                        'MakeSuperSubClient'
                ],
                'SubClientsPriv' => [
                        'AgencyControlClient'
                ],
                'SupSubClients' => [
                        'AddRightsToSSC',
                        'AllowTransferMoney',
                        'Control',
                        'MakeSimpleSubClient'
                ],
                'ThisAgency' => [
                        'AgencyControlAgencyMain',
                        'BindAgency',
                        'BindAgencySuper',
                        'CreateAgencyCampaign',
                        'CreateAgencyLimitedRep',
                        'CreateAgencyMainRep',
                        'DropAgency',
                        'UnbindAgency'
                ],
                'ThisAgencyMass' => [
                        'AllowMassADVQ'
                ],
                'ThisAgencyPPS' => [
                        'PayPiecesOfShows'
                ],
                'ThisAgencyXLS' => [
                        'AlowAgencyRepXlsImport'
                ],
                'ThisClient' => [
                        'CreateClientLimitedRep',
                        'CreateClientMainRep',
                        'DropClient'
                ]
        },
        'Client' => {
                'System' => [
                        'Login',
                        'ViewMediaplan',
                        'ViewPdfReports'
                ]
        },
        'ClientChiefRep' => {
                'ClientChiefRep' => [
                        'EditRep'
                ],
                'ClientLimitedRep' => [
                        'DropClientRep',
                        'EditRep',
                        'SwitchClientChiefRep'
                ],
                'ClientMainRep' => [
                        'DropClientRep',
                        'EditRep',
                        'SwitchClientChiefRep'
                ],
                'ThisClient' => [
                        'CreateClientLimitedRep',
                        'CreateClientMainRep',
                        'DropClient',
                        'MergeClient'
                ]
        },
        'ClientLimitedRep' => {
                'ACampaigns' => [
                        'Lookup',
                        'ShowStat',
                        'SubClientAccess'
                ],
                'ACampaignsPriv' => [
                        'ClientControlAC',
                        'InitCampTransfer'
                ],
                'AgencyMadeSuper' => [
                        'CreateAgencyCampaign'
                ],
                'ClientLimitedRep' => [
                        'InitCampTransfer'
                ],
                'ClientTransCamps' => [
                        'CompleteCampTransfer',
                        'RejectCampTransfer'
                ],
                'NSCampaigns' => [
                        'DropNSCampaign',
                        'EditCamp',
                        'Lookup',
                        'ManageNSCampaign',
                        'OfferToServicing',
                        'ShowStat'
                ],
                'NSCampaignsPriv' => [
                        'ClientControlNSC',
                        'InitCampTransfer'
                ],
                'SACampaigns' => [
                        'DropAgencyCampaign',
                        'EditCamp',
                        'Lookup',
                        'ShowStat',
                        'SuperSubClientAccess',
                        'SuperSubClientPayCamp'
                ],
                'SACampaignsATM' => [
                        'AllowTransferMoneySubClient'
                ],
                'SACampaignsXLS' => [
                        'ImportXlsBanners'
                ],
                'SCampaigns' => [
                        'DropSCampaign',
                        'EditCamp',
                        'Lookup',
                        'ManageServicedCampaign',
                        'ShowStat'
                ],
                'SCampaignsPriv' => [
                        'ClientControlSC',
                        'InitCampTransfer'
                ],
                'ThisClient' => [
                        'ClientControlClient',
                        'ClientControlClientLim',
                        'MergeClient'
                ],
                'ThisClientI18N' => [
                        'EditInternationalization'
                ],
                'ThisRep' => [
                        'CreateAgencyCampaign',
                        'CreateNSCampaign'
                ]
        },
        'ClientMainRep' => {
                'ACampaigns' => [
                        'Lookup',
                        'ShowStat',
                        'SubClientAccess'
                ],
                'ACampaignsPriv' => [
                        'ClientControlAC'
                ],
                'AgencyMadeSuper' => [
                        'CreateAgencyCampaign'
                ],
                'ClientChiefRep' => [
                        'CreateAgencyCampaign',
                        'CreateNSCampaign'
                ],
                'ClientLimitedRep' => [
                        'ControlClientLimRep',
                        'CreateAgencyCampaign',
                        'CreateNSCampaign',
                        'DropClientRep'
                ],
                'ClientMainRep' => [
                        'CreateAgencyCampaign',
                        'CreateNSCampaign',
                        'DropClientRep'
                ],
                'NSCampaigns' => [
                        'DropNSCampaign',
                        'EditCamp',
                        'Lookup',
                        'ManageNSCampaign',
                        'OfferToServicing',
                        'ShowStat'
                ],
                'NSCampaignsPriv' => [
                        'ClientControlNSC'
                ],
                'SACampaigns' => [
                        'DropAgencyCampaign',
                        'EditCamp',
                        'Lookup',
                        'ShowStat',
                        'SuperSubClientAccess',
                        'SuperSubClientPayCamp'
                ],
                'SACampaignsATM' => [
                        'AllowTransferMoneySubClient'
                ],
                'SACampaignsXLS' => [
                        'ImportXlsBanners'
                ],
                'SCampaigns' => [
                        'DropSCampaign',
                        'EditCamp',
                        'Lookup',
                        'ManageServicedCampaign',
                        'ShowStat'
                ],
                'SCampaignsPriv' => [
                        'ClientControlSC'
                ],
                'ThisClient' => [
                        'ClientControlClient',
                        'ClientControlClientMain',
                        'CreateClientLimitedRep',
                        'CreateClientMainRep',
                        'DropClient',
                        'MergeClient'
                ],
                'ThisClientI18N' => [
                        'EditInternationalization'
                ]
        },
        'Manager' => {
                'ACampaigns' => [
                        'AgencyManagerAccess',
                        'DropAgencyCampaign',
                        'EditCamp',
                        'ImportXlsBanners',
                        'Lookup',
                        'ShowStat'
                ],
                'ClientChiefRep' => [
                        'CreateAgencyCampaign',
                        'ManagerControlRep'
                ],
                'ClientLimitedRep' => [
                        'CreateAgencyCampaign',
                        'DropClientRep',
                        'ManagerControlRep'
                ],
                'ClientMainRep' => [
                        'CreateAgencyCampaign',
                        'DropClientRep',
                        'ManagerControlRep'
                ],
                'ClientTransCamps' => [
                        'CompleteCampTransfer',
                        'RejectCampTransfer'
                ],
                'ManagedAgencies' => [
                        'AllowAgencyXlsImport'
                ],
                'NSCampaigns' => [
                        'OfferToServicing'
                ],
                'SACampaigns' => [
                        'AgencyManagerAccess',
                        'DropAgencyCampaign',
                        'EditCamp',
                        'ImportXlsBanners',
                        'Lookup',
                        'ShowStat'
                ],
                'SCampaigns' => [
                        'DropSCampaign',
                        'EditCamp',
                        'ImportXlsBanners',
                        'Lookup',
                        'ManagerManageServicedCampaign',
                        'ShowStat',
                        'UnserviceSCamp'
                ],
                'SSClientWithAR' => [
                        'RevokeRightsFromSSC'
                ],
                'SSClientWithATM' => [
                        'ForbidTransferMoney'
                ],
                'SubClients' => [
                        'AllowTransferMoney',
                        'Control',
                        'MakeSuperSubClient'
                ],
                'SupSubClients' => [
                        'AddRightsToSSC',
                        'AllowTransferMoney',
                        'Control',
                        'MakeSimpleSubClient'
                ],
                'System' => [
                        'AdmSendBalance',
                        'AllowImportXlsBanners',
                        'CheckMirrors',
                        'CopyCamp',
                        'CreateAgency',
                        'CreateClient',
                        'CreateLogin',
                        'DoneWarn',
                        'GetSuggestion',
                        'GetUrlPhrases',
                        'ListWarn',
                        'Login',
                        'MassDeleteBanners',
                        'ModifyMediaplan',
                        'ModifySubClient',
                        'MoveCampaigns',
                        'Search',
                        'SearchBanner',
                        'SendModerate',
                        'SetAgencyNotResident',
                        'SetContextLimit',
                        'ShowArchClient',
                        'ShowManagerMyClients',
                        'ShowOfferToServicing',
                        'ShowReportContext',
                        'SystemReport',
                        'UnArchClientAndServ',
                        'UnServicingCamp',
                        'UseAcceptServicing',
                        'UseDeclineServicing',
                        'UseOfferToServicing',
                        'ViewMediaplan',
                        'ViewMediaplanStat',
                        'ViewPdfReports',
                        'ViewUserDescription'
                ],
                'ThisAgency' => [
                        'AgencyControl',
                        'AllowAgencyAddRights',
                        'AllowAgencyMassADVQ',
                        'AllowAgencyXlsImport',
                        'BindAgency',
                        'CreateAgencyCampaign',
                        'CreateAgencyLimitedRep',
                        'CreateAgencyMainRep',
                        'DropAgency',
                        'UnbindAgency'
                ],
                'ThisAgencyAR' => [
                        'ForbidAgencyAddRights'
                ],
                'ThisAgencyMass' => [
                        'ForbidAgencyMassADVQ'
                ],
                'ThisAgencyXLS' => [
                        'ForbidAgencyXlsImport'
                ],
                'ThisClient' => [
                        'Control',
                        'DropClient'
                ],
                'ThisClientStructed' => [
                        'CreateClientLimitedRep',
                        'CreateClientMainRep'
                ],
                'ThisManager' => [
                        'BindManager',
                        'UnbindManager'
                ],
                'TransitCampaigns' => [
                        'AcceptServicing',
                        'DeclineServicing'
                ]
        },
        'ManagerPrivate' => {
                'ManagedAgencies' => [
                        'ManagerControlAgency'
                ],
                'SCampaignsPriv' => [
                        'ManagerControlSCamps'
                ],
                'ServicedClients' => [
                        'ManagerControlClient'
                ],
                'ThisManager' => [
                        'ManagerControlManager'
                ]
        },
        'Mediaplanner' => {
                'ACampaigns' => [
                        'EditCamp',
                        'Lookup',
                        'ShowStat'
                ],
                'AgencyChiefReps' => [
                        'UserLookup'
                ],
                'AgencyLimitedReps' => [
                        'UserLookup'
                ],
                'AgencyMainReps' => [
                        'UserLookup'
                ],
                'ClientChiefRep' => [
                        'EditRep'
                ],
                'ClientLimitedRep' => [
                        'EditRep'
                ],
                'ClientMainRep' => [
                        'EditRep'
                ],
                'NSCampaigns' => [
                        'EditCamp',
                        'Lookup',
                        'ShowStat'
                ],
                'SACampaigns' => [
                        'EditCamp',
                        'Lookup',
                        'ShowStat'
                ],
                'SCampaigns' => [
                        'EditCamp',
                        'Lookup',
                        'ShowStat'
                ],
                'System' => [
                        'AllowImportXlsBanners',
                        'FirstAid',
                        'GetSuggestion',
                        'GetUrlPhrases',
                        'ListWarn',
                        'Login',
                        'ModifyMcbThemes',
                        'ModifyMediaplan',
                        'Search',
                        'SearchBanner',
                        'ShowAccessLogs',
                        'ShowManagerMyClients',
                        'SystemReport',
                        'ViewMediaplan',
                        'ViewMediaplanStat'
                ],
                'ThisRep' => [
                        'EditRep'
                ]
        },
        'Placer' => {
                'ACampaigns' => [
                        'EditCamp',
                        'ImportXlsBanners',
                        'Lookup',
                        'ShowStat'
                ],
                'AgencyChiefReps' => [
                        'UserLookup'
                ],
                'AgencyLimitedReps' => [
                        'UserLookup'
                ],
                'AgencyMainReps' => [
                        'UserLookup'
                ],
                'ClientChiefRep' => [
                        'CreateAgencyCampaign',
                        'CreateNSCampaign',
                        'EditRep'
                ],
                'ClientLimitedRep' => [
                        'CreateAgencyCampaign',
                        'CreateNSCampaign',
                        'EditRep'
                ],
                'ClientMainRep' => [
                        'CreateAgencyCampaign',
                        'CreateNSCampaign',
                        'EditRep'
                ],
                'NSCampaigns' => [
                        'EditCamp',
                        'ImportXlsBanners',
                        'Lookup',
                        'ShowStat'
                ],
                'SACampaigns' => [
                        'EditCamp',
                        'ImportXlsBanners',
                        'Lookup',
                        'ShowStat'
                ],
                'SCampaigns' => [
                        'EditCamp',
                        'ImportXlsBanners',
                        'Lookup',
                        'ShowStat'
                ],
                'System' => [
                        'AdmSendBKMD',
                        'AllowImportXlsBanners',
                        'CheckMirrors',
                        'CopyCamp',
                        'FirstAid',
                        'Login',
                        'MassDeleteBanners',
                        'ModifyMediaplan',
                        'PlaceBanners',
                        'Search',
                        'SearchBanner',
                        'SendModerate',
                        'ShowManagerMyClients',
                        'ShowOfferToServicing',
                        'SystemReport',
                        'UseDeclineServicing',
                        'UseOfferToServicing',
                        'ViewMediaplan'
                ],
                'ThisAgency' => [
                        'CreateAgencyCampaign'
                ],
                'ThisClient' => [
                        'Control'
                ],
                'ThisRep' => [
                        'CreateNSCampaign',
                        'EditRep'
                ]
        },
        'SimpleClient' => {
                'NSCampaigns' => [
                        'ClientControlNSC',
                        'DropNSCampaign',
                        'EditCamp',
                        'InitCampTransfer',
                        'Lookup',
                        'ManageNSCampaign',
                        'OfferToServicing',
                        'ShowStat'
                ],
                'ThisClient' => [
                        'ClientControlClient',
                        'ClientControlClientSimple',
                        'DropClient',
                        'MergeClient'
                ],
                'ThisClientI18N' => [
                        'EditInternationalization'
                ],
                'ThisClientSimple' => [
                        'MakeClientStructed'
                ],
                'ThisRep' => [
                        'CreateNSCampaign',
                        'EditRep'
                ]
        },
        'SuperPlacer' => {
                'System' => [
                        'AutoModerate',
                        'ShowAccessLogs'
                ]
        },
        'SuperReader' => {
                'ACampaigns' => [
                        'Lookup',
                        'ShowStat'
                ],
                'AgencyChiefReps' => [
                        'UserLookup'
                ],
                'AgencyLimitedReps' => [
                        'UserLookup'
                ],
                'AgencyMainReps' => [
                        'UserLookup'
                ],
                'ClientChiefRep' => [
                        'UserLookup'
                ],
                'ClientLimitedRep' => [
                        'UserLookup'
                ],
                'ClientMainRep' => [
                        'UserLookup'
                ],
                'Mediaplanners' => [
                        'UserLookup'
                ],
                'NSCampaigns' => [
                        'Lookup',
                        'ShowStat'
                ],
                'Placers' => [
                        'UserLookup'
                ],
                'SACampaigns' => [
                        'Lookup',
                        'ShowStat'
                ],
                'SCampaigns' => [
                        'Lookup',
                        'ShowStat'
                ],
                'SuperMediaplanners' => [
                        'UserLookup'
                ],
                'SuperPlacers' => [
                        'UserLookup'
                ],
                'Supports' => [
                        'UserLookup'
                ],
                'System' => [
                        'AdmSendBalance',
                        'AdmSendBKMD',
                        'CheckMirrors',
                        'FirstAid',
                        'GetSuggestion',
                        'GetUrlPhrases',
                        'GetVendorReport',
                        'ListWarn',
                        'Login',
                        'Search',
                        'SearchBanner',
                        'ShowAccessLogs',
                        'ShowArchClient',
                        'ShowLogs',
                        'ShowManagerMyClients',
                        'ShowOfferToServicing',
                        'ShowRegionalTraffic',
                        'ShowReportContext',
                        'SystemReport',
                        'ViewMediaplan',
                        'ViewMediaplanStat',
                        'ViewPdfReports',
                        'ViewUserDescription'
                ],
                'ThisManager' => [
                        'UserLookup'
                ],
                'ThisRep' => [
                        'UserLookup'
                ],
                'TransitCampaigns' => [
                        'DeclineServicing'
                ],
                'Ubermensches' => [
                        'UserLookup'
                ]
        },
        'SuperTeamLeader' => {
                'AgencyLimitedReps' => [
                        'DropAgencyRep',
                        'SwitchAgencyChiefRep'
                ],
                'AgencyMainReps' => [
                        'DropAgencyRep',
                        'SwitchAgencyChiefRep'
                ],
                'ClientLimitedRep' => [
                        'SwitchClientChiefRep'
                ],
                'ClientMainRep' => [
                        'SwitchClientChiefRep'
                ],
                'SuperteamedTeams' => [
                        'LeadTheSuperTeam',
                        'MoveManagerFromToTeam'
                ],
                'System' => [
                        'AdmBanPay',
                        'ChangeManagerOfAgencySys',
                        'ChangeManagerSys',
                        'ChangeOwner',
                        'ChangeOwnerOfAgency',
                        'CreateManager',
                        'ManageTeamleaders',
                        'ShowReportContext',
                        'UnServicingCamp'
                ],
                'TeamedManagers' => [
                        'ChangeManager',
                        'ChangeManagerOfAgency',
                        'DeleteManagerFromTeam',
                        'MoveAgencyBetweenMgs',
                        'MoveManagerBetweenTeams'
                ],
                'TeamlessManagers' => [
                        'AddManagerToTeam'
                ],
                'ThisClient' => [
                        'MergeClient'
                ],
                'ThisManager' => [
                        'AssignDirectManager',
                        'BindManager',
                        'ChangeManager',
                        'DropManager',
                        'UnbindManager',
                        'WithdrawDirectManager'
                ]
        },
        'Support' => {
                'ACampaigns' => [
                        'EditCamp',
                        'ImportXlsBanners',
                        'Lookup',
                        'ShowStat'
                ],
                'AgencyChiefReps' => [
                        'EditRep'
                ],
                'AgencyLimitedReps' => [
                        'EditRep'
                ],
                'AgencyMainReps' => [
                        'EditRep'
                ],
                'ClientChiefRep' => [
                        'CreateAgencyCampaign',
                        'CreateNSCampaign',
                        'EditRep',
                        'UserLookup'
                ],
                'ClientLimitedRep' => [
                        'CreateAgencyCampaign',
                        'CreateNSCampaign',
                        'EditRep'
                ],
                'ClientMainRep' => [
                        'CreateAgencyCampaign',
                        'CreateNSCampaign',
                        'EditRep',
                        'UserLookup'
                ],
                'NSCampaigns' => [
                        'EditCamp',
                        'ImportXlsBanners',
                        'Lookup',
                        'OfferToServicing',
                        'ShowStat'
                ],
                'SACampaigns' => [
                        'EditCamp',
                        'ImportXlsBanners',
                        'Lookup',
                        'ShowStat'
                ],
                'SCampaigns' => [
                        'EditCamp',
                        'ImportXlsBanners',
                        'Lookup',
                        'ShowStat',
                        'UnserviceSCamp'
                ],
                'SubClients' => [
                        'MakeSuperSubClient'
                ],
                'SupSubClients' => [
                        'MakeSimpleSubClient'
                ],
                'System' => [
                        'AdmBanPay',
                        'AdmSendBKMD',
                        'AllowImportXlsBanners',
                        'AutoModerate',
                        'ChangeManagerOfAgencySys',
                        'CheckMirrors',
                        'EditMirrors',
                        'GetSuggestion',
                        'GetVendorReport',
                        'ListWarn',
                        'ManageTeamleaders',
                        'ManageSTeamleaders',
                        'ModifyMediaplan',
                        'ModifySubClient',
                        'ModifyUser',
                        'MoveCampaigns',
                        'Search',
                        'SearchBanner',
                        'SendModerate',
                        'SetContextLimit',
                        'ShowAccessLogs',
                        'ShowLogs',
                        'ShowManagerMyClients',
                        'ShowOfferToServicing',
                        'ShowRegionalTraffic',
                        'UnServicingCamp',
                        'UseDeclineServicing',
                        'UseOfferToServicing',
                        'ViewMediaplan',
                        'ViewPdfReports',
                        'ViewUserDescription'
                ],
                'ThisAgency' => [
                        'CreateAgencyCampaign'
                ],
                'ThisClient' => [
                        'MergeClient'
                ],
                'ThisManager' => [
                        'UserLookup'
                ],
                'ThisRep' => [
                        'CreateNSCampaign',
                        'EditRep',
                        'UserLookup'
                ],
                'TransitCampaigns' => [
                        'DeclineServicing'
                ]
        },
        LimitedSupport => {
                'ACampaigns' => [
                        'Lookup',
                        'ShowStat'
                ],
                'AgencyChiefReps' => [
                ],
                'AgencyLimitedReps' => [
                ],
                'AgencyMainReps' => [
                ],
                'ClientChiefRep' => [
                        'UserLookup'
                ],
                'ClientLimitedRep' => [
                ],
                'ClientMainRep' => [
                        'UserLookup'
                ],
                'NSCampaigns' => [
                        'Lookup',
                        'ShowStat'
                ],
                'SACampaigns' => [
                        'Lookup',
                        'ShowStat'
                ],
                'SCampaigns' => [
                        'Lookup',
                        'ShowStat',
                ],
                'SubClients' => [
                ],
                'SupSubClients' => [
                ],
                'System' => [
                        'AdmSendBalance',
                        'AdmSendBKMD',
                        'CheckMirrors',
                        'GetSuggestion',
                        'GetVendorReport',
                        'ListWarn',
                        'ModifySubClient',
                        'Search',
                        'SearchBanner',
                        'SetContextLimit',
                        'ShowAccessLogs',
                        'ShowLogs',
                        'ShowManagerMyClients',
                        'ShowOfferToServicing',
                        'ShowRegionalTraffic',
                        'ViewMediaplan',
                        'ViewPdfReports',
                        'ViewUserDescription'
                ],
                'ThisAgency' => [
                ],
                'ThisClient' => [
                ],
                'ThisManager' => [
                        'UserLookup'
                ],
                'ThisRep' => [
                        'UserLookup'
                ],
                'TransitCampaigns' => [
                ]
        },
        'TeamLeader' => {
                'AgencyLimitedReps' => [
                        'DropAgencyRep',
                        'SwitchAgencyChiefRep'
                ],
                'AgencyMainReps' => [
                        'DropAgencyRep',
                        'SwitchAgencyChiefRep'
                ],
                'ClientLimitedRep' => [
                        'SwitchClientChiefRep'
                ],
                'ClientMainRep' => [
                        'SwitchClientChiefRep'
                ],
                'System' => [
                        'AdmBanPay',
                        'ChangeManagerOfAgencySys',
                        'ChangeManagerSys',
                        'ChangeOwner',
                        'ChangeOwnerOfAgency',
                        'ShowReportContext',
                        'UnServicingCamp'
                ],
                'TeamedManagers' => [
                        'ChangeManager',
                        'ChangeManagerOfAgency',
                        'LeadTheTeam',
                        'MoveAgencyBetweenMgs'
                ],
                'ThisClient' => [
                        'MergeClient'
                ],
                'ThisManager' => [
                        'AssignDirectManager',
                        'BindManager',
                        'ChangeManager',
                        'DropManager',
                        'UnbindManager',
                        'WithdrawDirectManager'
                ]
        },
        'Ubermensch' => {
                'ACampaigns' => [
                        'DropAgencyCampaign',
                        'EditCamp',
                        'ImportXlsBanners',
                        'Lookup',
                        'ShowStat'
                ],
                'ACampaignsPriv' => [
                        'InitCampTransfer'
                ],
                'AgencyChiefReps' => [
                        'EditRep'
                ],
                'AgencyLimitedReps' => [
                        'ChangeAgencyLrepToMrep',
                        'DropAgencyRep',
                        'EditRep',
                        'MoveAlinkToLrep',
                        'SwitchAgencyChiefRep'
                ],
                'AgencyMainReps' => [
                        'ChangeAgencyMrepToLrep',
                        'DropAgencyRep',
                        'EditRep',
                        'SwitchAgencyChiefRep'
                ],
                'ClientChiefRep' => [
                        'CreateAgencyCampaign',
                        'CreateNSCampaign',
                        'EditRep',
                        'UserLookup'
                ],
                'ClientLimitedRep' => [
                        'ChangeClientLrepToMrep',
                        'CreateAgencyCampaign',
                        'CreateNSCampaign',
                        'DropClientRep',
                        'EditRep',
                        'InitCampTransfer',
                        'SwitchClientChiefRep'
                ],
                'ClientMainRep' => [
                        'ChangeClientMrepToLrep',
                        'CreateAgencyCampaign',
                        'CreateNSCampaign',
                        'DropClientRep',
                        'EditRep',
                        'SwitchClientChiefRep',
                        'UserLookup'
                ],
                'ClientTransCamps' => [
                        'CompleteCampTransfer',
                        'RejectCampTransfer'
                ],
                'Mediaplanners' => [
                        'DropMediaplanner'
                ],
                'NSCampaigns' => [
                        'DropNSCampaign',
                        'EditCamp',
                        'ImportXlsBanners',
                        'InitCampTransfer',
                        'Lookup',
                        'OfferToServicing',
                        'ShowStat'
                ],
                'NSCampaignsPriv' => [
                        'InitCampTransfer'
                ],
                'Placers' => [
                        'DropPlacer'
                ],
                'SACampaigns' => [
                        'DropAgencyCampaign',
                        'EditCamp',
                        'ImportXlsBanners',
                        'Lookup',
                        'ShowStat'
                ],
                'SCampaigns' => [
                        'DropSCampaign',
                        'EditCamp',
                        'ImportXlsBanners',
                        'Lookup',
                        'ShowStat',
                        'UnserviceSCamp'
                ],
                'SCampaignsPriv' => [
                        'InitCampTransfer'
                ],
                'SSClientWithAR' => [
                        'RevokeRightsFromSSC'
                ],
                'SSClientWithATM' => [
                        'ForbidTransferMoney'
                ],
                'SubClients' => [
                        'AllowTransferMoney',
                        'MakeSuperSubClient'
                ],
                'SupSubClients' => [
                        'AddRightsToSSC',
                        'AllowTransferMoney',
                        'MakeSimpleSubClient'
                ],
                'SuperMediaplanners' => [
                        'DropSuperMedia'
                ],
                'SuperPlacers' => [
                        'DropSuperPlacer'
                ],
                'SuperReaders' => [
                        'DropSuperReader'
                ],
                'SuperTeamLeaders' => [
                        'DropSuperTeamLeader'
                ],
                'SuperteamedTeams' => [
                        'DeleteTeamFromSuperTeam',
                        'MoveManagerFromToTeam',
                        'MoveTeamBetweenSuperTeams'
                ],
                'Supports' => [
                        'DropSupport'
                ],
                'System' => [
                        'AdmBanPay',
                        'AdmSendBalance',
                        'AdmSendBKMD',
                        'AllowImportXlsBanners',
                        'AutoModerate',
                        'BalanceAccess',
                        'ChangeManager',
                        'ChangeManagerOfAgency',
                        'ChangeManagerOfAgencySys',
                        'ChangeManagerSys',
                        'ChangeOwner',
                        'ChangeOwnerOfAgency',
                        'CheckMirrors',
                        'CopyCamp',
                        'CreateAgency',
                        'CreateClient',
                        'CreateLogin',
                        'CreateManager',
                        'CreateMediaplanner',
                        'CreatePlacer',
                        'CreateSuperMedia',
                        'CreateSuperPlacer',
                        'CreateSuperReader',
                        'CreateSupport',
                        'CreateUbermensch',
                        'EditInternationalization',
                        'EditMails',
                        'EditMirrors',
                        'FirstAid',
                        'GetSuggestion',
                        'GetUrlPhrases',
                        'GetVendorReport',
                        'ListWarn',
                        'Login',
                        'ManageTeamleaders',
                        'ManageSTeamleaders',
                        'MassDeleteBanners',
                        'ModifyMcbThemes',
                        'ModifyMediaplan',
                        'ModifySubClient',
                        'ModifyUser',
                        'MoveCampaigns',
                        'Search',
                        'SearchBanner',
                        'SendModerate',
                        'SetAgencyNotResident',
                        'SetContextLimit',
                        'ShowAccessLogs',
                        'ShowArchClient',
                        'ShowLogs',
                        'ShowManagerMyClients',
                        'ShowOfferToServicing',
                        'ShowRegionalTraffic',
                        'ShowReportContext',
                        'SystemReport',
                        'UnArchClientAndServ',
                        'UnServicingCamp',
                        'UseDeclineServicing',
                        'UseOfferToServicing',
                        'UsersAdministration',
                        'ViewMediaplan',
                        'ViewMediaplanStat',
                        'ViewPdfReports',
                        'ViewUserDescription'
                ],
                'TeamLeaders' => [
                        'AddTeamToSuperTeam',
                        'DropTeamLeader',
                        'MoveManagerFromToTeam'
                ],
                'TeamedManagers' => [
                        'DeleteManagerFromTeam',
                        'MoveManagerBetweenTeams'
                ],
                'TeamlessManagers' => [
                        'AddManagerToTeam',
                        'CreateSuperTeamLeader',
                        'CreateTeamLeader'
                ],
                'ThisAgency' => [
                        'AllowAgencyAddRights',
                        'AllowAgencyMassADVQ',
                        'AllowAgencyXlsImport',
                        'AllowPayPiecesOfShows',
                        'BindAgency',
                        'Control',
                        'CreateAgencyCampaign',
                        'CreateAgencyLimitedRep',
                        'CreateAgencyMainRep',
                        'DropAgency',
                        'MoveAlinkToMain',
                        'UnbindAgency'
                ],
                'ThisAgencyAR' => [
                        'ForbidAgencyAddRights'
                ],
                'ThisAgencyMass' => [
                        'ForbidAgencyMassADVQ'
                ],
                'ThisAgencyPPS' => [
                        'ForbidPayPiecesOfShows'
                ],
                'ThisAgencyXLS' => [
                        'ForbidAgencyXlsImport'
                ],
                'ThisClient' => [
                        'AllowClientEditI18n',
                        'Control',
                        'DropClient',
                        'ForbidClientEditI18n',
                        'GrantFBWToClient',
                        'MergeClient',
                        'RevokeFBWFromClient'
                ],
                'ThisClientSimple' => [
                        'MakeClientStructed'
                ],
                'ThisClientStructed' => [
                        'CreateClientLimitedRep',
                        'CreateClientMainRep'
                ],
                'ThisManager' => [
                        'AssignDirectManager',
                        'BindManager',
                        'DropManager',
                        'UnbindManager',
                        'WithdrawDirectManager'
                ],
                'ThisRep' => [
                        'CreateNSCampaign',
                        'EditRep',
                        'UserLookup'
                ],
                'TransitCampaigns' => [
                        'DeclineServicing'
                ],
                'Ubermensches' => [
                        'DropUbermensch'
                ]
        },
        'InternalAdAdmin' => {
                'System' => [],
        },
};

1;
