use warnings;
use strict;
use utf8;

package ADVQ6;

=head1 NAME
    
    ADVQ6

=head1 DESCRIPTION

    Interface for ADVQ6
    $Id$

    Описание интерфейса можно смотреть на http://wiki.yandex-team.ru/advq6/api 
    или на http://test1.advq.yandex.ru/advq/help (доступно с ppcdev'а).

=cut


use vars qw(@ISA @EXPORT @EXPORT_OK);
require Exporter;

use feature 'state';
use Encode;
use EnvTools;
use HTTP::Request::Common;
use HTTP::Response;
use Digest::MD5 qw(md5_hex);
use JSON;
use List::Util qw/sum max min/;
use List::MoreUtils qw/any uniq all/;
use POSIX qw/ceil/;
use LogTools qw/log_messages/;
use MinusWordsTools;
use MinusWords;
use EnvTools;
use Property;
use SolomonTools;

use Settings;

use YAML::Syck;
$YAML::Syck::ImplicitUnicode = 1;

use Yandex::MyGoodWords;
use Yandex::HashUtils qw/hash_cut/;
use Yandex::Trace qw/current_trace/;
use Yandex::HTTP qw/http_parallel_request http_fetch/;
use Yandex::Memcached::Lock;
use Yandex::StopWords qw/is_stopword/;

use Currency::Rate qw/convert_currency/;
use Property;

@ISA         = qw(Exporter);
@EXPORT_OK = @EXPORT = qw/ 
                            advq_get_stat
                            advq_get_detailed_stat
                            advq_get_phrases_shows
                            advq_get_phrases_shows_multi
                            advq_get_phrases_detailed_stat
                            advq_forecast_multibudget
                            advq_get_phrases_multibudget_forecast_multi
                            advq_get_phrases_budget_dist_forecast_multi
                            format_advq_query
                        /;

# ----------------------------------------------------------------------
our $ADVQ_URL ||= 'http://back-normal.advq.yandex.net/advq';
our $ADVQ_VIDEO_URL ||= 'http://advq-video.yandex.ru/advq';
our $ADVQ_FORECAST_URL ||= 'http://forecast.advq.yandex.ru/pkz';
our $ADVQ_CLOUD_FORECAST_URL ||= 'http://back-forecast.advq.yandex.net/pkz';
our $USE_CLOUD_ADVQ_FORECAST = Property->new('USE_CLOUD_ADVQ_FORECAST');
our $ADVQ_TIMEOUT ||= 120;
our $ADVQ_MAX_PHRASES_CNT ||= 5;
our $ADVQ_PARALLEL_LEVEL ||= 5;
# не использовать при запросах минус-слова на кампанию - их очень много, advq не справляется
our $SKIP_CAMP_MINUS_WORDS_PROP = Property->new('ADVQ_SKIP_CAMP_MINUS_WORDS');

our %ADVQ_CALLS_LOG_SETTINGS;
if (!%ADVQ_CALLS_LOG_SETTINGS) {
    %ADVQ_CALLS_LOG_SETTINGS = (
        log_file_name => "advq_calls.log",
        date_suf => "%Y%m%d",
    );
}

# Нужно ли вести лог фраз в запросах к ADVQ. Ведется везде, кроме прода.
our $REQUESTS_LOG = is_production() ? 0 : 1;
# Нужно ли вести логирование как таковое
our $WRITE_LOG = 1;

#Дефолтные настройки Yandex::Memcached::Lock.
#Используются для ограничения числа запросов, отправляемых в ADVQ с одного хоста.
#Переопределяются через advq_requests_lock_settings_json
our $LOCK_SETTINGS_PROPERTY_NAME = 'advq_requests_lock_settings_json';
#Максимальное количество одновременно существующих блокировок
our $LOCK_MAX_NUM = 20;
#Время, в течение которого пытаемся получить lock. Если 0 - делаем только одну попытку.
our $LOCK_TIMEOUT = 5;

state $lock_settings_json = Property->new($LOCK_SETTINGS_PROPERTY_NAME);

{
    my $forecast_log;

    sub _forecast_log {
        my ($reqid, $data) = @_;

        return if ! $WRITE_LOG;

        if (! defined $forecast_log) {
            $forecast_log = Yandex::Log->new(%ADVQ_CALLS_LOG_SETTINGS);
        }

        my $prefix_guard = $forecast_log->msg_prefix_guard(sprintf('[reqid=%d]', $reqid // 0));

        $forecast_log->out($data);
    }

    sub _request_log {
        my ($reqid, $data) = @_;

        return if !$REQUESTS_LOG;

        _forecast_log($reqid, $data);
    }
}

=head2 advq_get_stat

    return list of hashes for each processed phrase on success, or for part of phrases on ADVQ error, or [] on ADVQ error;

    Параметры позиционные:
        words - array ref, фразы
        geo - регион
    Опциональные:
        timeout
        function -- функция advq, которую надо запросить (search|time_hist). По умолчанию 'search'
        precision - точность вычислений от 0 до 1 (меньше точность - больше скорость, больше точность - меньше скорость.) По умолчанию 1.
        lang -- язык для леммера; по умолчанию используется Русский/Английский, указав "tr" можно переключиться на Турецкий
        devices - с какого устройства был выполнен запрос, возможные варианты [qw/phone tablet desktop all/]

=cut

sub advq_get_stat {
    my ($words, $geo, %O) = @_;

    my $timeout = $O{timeout} && $O{timeout} + 0.0 > 0 ? $O{timeout} : $ADVQ_TIMEOUT;
    return advq_request($words, regions  => $geo,
                                function => $O{function},
                                (defined $O{precision} ? (precision => $O{precision}) : ()),
                                devices => $O{devices},
                                timeout   => $timeout,
                                lang => $O{lang},
                                video_advq => $O{video_advq},
                                collections_advq => $O{collections_advq},
    );
}

=head2 advq_request

    exec parallel advq requests

    Параметры позиционные:
        $words - array ref
    Опциональные:
        timeout
        function -- функция advq, которую надо запросить (search|time_hist). По умолчанию 'search'
        precision - точность вычислений от 0 до 1 (меньше точность - больше скорость, больше точность - меньше скорость.) По умолчанию 1.
        timeout_on_phrase - timeout на фразу
        lang -- язык для леммера; по умолчанию используется Русский/Английский, указав "tr" можно переключиться на Турецкий
        devices -
        video_advq - флаг, запрашиваем ли статистику для видео (для видео используется другой URL)
        collections_advq - флаг, запрашиваем ли статистику для коллекций (для них уменьшаем количество показов в 10 раз, см. DIRECT-108374)

=cut

sub advq_request {
    
    my ($words, %O) = @_;
    my $function = $O{function} = $O{function} || 'search';

    my @common_params = map { $_ => $O{$_} } qw/function precision regions ph_page_size ph_page assocs timeout_on_phrase/;
    push @common_params, map {$_ => $O{$_}} grep {defined $O{$_}} qw/calc_total_hits fast_mode/;

    push @common_params, dbname => $O{lang} && $O{lang} eq 'tr' ? 'tur' : 'rus';  
    push @common_params, devices => join ',', @{$O{devices}} if $O{devices};
    my @words = ref $words eq 'ARRAY' ? @$words : ($words);
    my $effective_timeout = $O{timeout} || $ADVQ_TIMEOUT;

    my %tasks;
    my $id = 0;
    my $trace_id = Yandex::Trace::current_span_id();
    my $video_advq = $O{video_advq} || 0;
    my $url;
    if ($video_advq != 1) {
        $url = "$ADVQ_URL/$function";
    } else {
        $url = "$ADVQ_VIDEO_URL/$function";
    }
    foreach my $w (_cut_words(\@words)) {
        # Составляем одноуровневый массив параметров: 
        # 1. парамеры общие для всех блоков фраз
        my @params = @common_params;
        # 2. очередной блок фраз
        push @params, map {(words => $_)} @{ $w || [] };
        # 3. Таймаут
        $effective_timeout = $O{timeout_on_phrase} * @$w if exists $O{timeout_on_phrase};
        push @params, (timeout => $effective_timeout);
        $tasks{$id} = {
            url => $url,
            body => \@params,
            headers => {
                'X-Yandex-Trace' => join(',', map {$_//0} $trace_id, Yandex::Trace::generate_traceid(), $trace_id, 0),
                'X-Advq-Customer' => 'direct',
            },
        };
        $id++;
    }

    _request_log($trace_id, {url => $url, devices => $O{devices}, regions => $O{regions}, words => \@words,});
    my $lock = _set_lock($effective_timeout);

    # Параллельный запрос + таймер
    my $profile = Yandex::Trace::new_profile('advq:advq_request', tags => $function, obj_num => scalar(@words));

    my $advq_resp = http_parallel_request(
        POST => \%tasks, 
        max_req => $ADVQ_PARALLEL_LEVEL,
        timeout => 2 * $effective_timeout, # http таймаут должен быть дольше, чем advq-таймаут
        headers => {
            'User-Agent' => undef,
            'Referer' => undef,
            'Content-Type' => "application/x-www-form-urlencoded",
        },
    );

    undef $profile;
    undef $lock;

    # Разбираем ответ
    my @data;
    
    # сортируем чанки по id, чтобы отдать ответ в том же порядке, что был в запросе
    foreach my $id (sort {$a <=> $b} keys %$advq_resp) {
        my $resp = $advq_resp->{$id};
        unless ( $resp->{is_success}) {
            warn "ADVQ request failed $resp->{headers}->{URL}: $resp->{headers}->{Reason}"; 
            next;
        }
        push @data, _handle_response($function, Encode::decode_utf8($resp->{content}), $O{collections_advq});
    }
    
    return \@data;
}

sub _set_lock {
    my ($advq_timeout) = @_;
    my ($max_locks_num, $timeout);
    my $settings;

    my $log_prefix = 'advq_semaphore';
    my $host = $EnvTools::hostname;

    eval {
        $settings = JSON::from_json($lock_settings_json->get(180) // '{}');
    };
    if ($@) {
        log_messages($log_prefix => "Error occured while JSON parsing: $@");
    }

    $max_locks_num = $settings->{max_locks_num} // $LOCK_MAX_NUM;
    $timeout = $settings->{timeout} // $LOCK_TIMEOUT;

    my $lock = Yandex::Memcached::Lock->new(
        servers => $Settings::MEMCACHED_SERVERS,
        entry   => 'ADVQ_LOCK_'.$host,
        expire  => _get_lock_expire_by_timeout($advq_timeout || $ADVQ_TIMEOUT),
        timeout => $timeout,
        max_locks_num => $max_locks_num,
    );

    my $lock_status = $lock->get_lock();
    #Если $lock_status == 1 - лок успешно установлен
    if ( $lock_status ) {
        eval { SolomonTools::send_advq_semaphores_stat($host => 'acquired') };
        return $lock;
    }

    #Если $lock_status == 0 -лок установить не получилось, падаем
    if (defined $lock_status) {
        eval { SolomonTools::send_advq_semaphores_stat($host => 'failed') };
        log_messages($log_prefix => "Max lock count ($max_locks_num) reached. Died.");
        die("$log_prefix: max lock count reached");
    }

    #Eсли $lock_status - undef, получить его не удалось из-за проблем с серверами или другой ошибки, работаем без лока
    log_messages($log_prefix => "Unable to get lock");
    eval { SolomonTools::send_advq_semaphores_stat($host => 'unknown') };

    return $lock;
}

sub _get_lock_expire_by_timeout {
    my ($timeout) = @_;
    # 2*$timeout - ограничение на http-запрос. Добавляем к нему еще две секунды запаса.
    return 2 + 2*$timeout;
}

# cut all phrases by chunks for parallel requests
sub _cut_words {
    
    my $words = shift;

    my @chunks;
    my @phrases = @$words;
    while (my @ch = splice @phrases, 0, $ADVQ_MAX_PHRASES_CNT) {
        push @chunks, \@ch;
    }
    
    return @chunks;
}

# convert advq response to perl struct
# in: 
# - function name
# - body of response
# - is request for collections
sub _handle_response {
    
    my ($function, $content, $collections_advq) = @_;

    my $answer = YAML::Syck::Load($content);
    my @results;
    if ($function eq 'search') {
        foreach (@{$answer->{requests}}) {
            push @results, {
                count => _get_collections_count($_->{stat}{total_count}, $collections_advq),
                phrase => $_->{req}, 
                precision => $_->{stat}{precision}, 
                including_phrases => _change_collections_array_count($_->{stat}{including_phrases}, $collections_advq),
                has_next_page => $_->{stat}{has_next_page}, 
                associations => _change_collections_array_count($_->{stat}{associations}, $collections_advq),
                advq_date => $_->{stat}{db_date_str} 
            } if ($_->{stat}{tainted} || 1) =~ m/^(false|0)$/i;
        }
    } elsif (any {$function eq $_} qw/time_hist monthly_hist weekly_hist/) {
        @results = @{$answer->{requests}};
    }
    return @results;    
}

sub _change_collections_array_count {
    my ($array, $collections_advq) = @_;
    if (!$collections_advq) {
        return $array;
    }
    my @results;
    foreach (@{$array}) {
        $_->{cnt} = _get_collections_count($_->{cnt}, $collections_advq);
        push @results, $_;
    }
    return \@results;
}

=head2 _get_collections_count

    Для коллекций уменьшаем количество показов в 10 раз, см. DIRECT-108374

=cut
sub _get_collections_count {
    my ($count, $collections_advq) = @_;
    return $collections_advq ? ceil($count / 10) : $count;
}

=head2 advq_get_phrases_shows

    fill in showsForecast field in properly processed phrases, others are filled with undefined values
    return 'real' all phrases processed; else return ''

=cut
sub advq_get_phrases_shows {
    my ( $phrases_arr, $geo, %O ) = @_;
    return advq_get_phrases_shows_multi([{geo => $geo, phrases=>$phrases_arr}], %O);
}

=head2 advq_get_phrases_shows_multi

    Получает из ADVQ прогноз показов по массиву блоков [групп или блоков медиаплана]
    Прогноз показов добавляется в каждую фразу в виде ключа showsForecast

    $blocks = [
        {
            geo =>
            phrase|phr =>
            minus_words =>
            banner_minus_words =>
            campaign_minus_words =>
        },
        ...
    ];
    %O = (
        function =>
        minus_words =>
        timeout =>
        precision =>
        lang =>
        devices => [....]
        period =>
    );
    $result = advq_get_phrases_shows_multi($blocks, %O);
    $result => 'real'|''

=cut

sub advq_get_phrases_shows_multi {
    my ( $blocks, %O ) = @_;

    my $geo_phrases = {};
    for my $block (@$blocks) {
        for my $ph (@{$block->{phrases}}) {
            my $phrase = defined $ph->{phrase_wo_common_minus_words} ? $ph->{phrase_wo_common_minus_words} : (defined $ph->{phrase} ? $ph->{phrase} : $ph->{phr});
            if (defined $phrase && $phrase ne '') {
                $ph->{phrase_for_advq} = create_advq_request_phrase($phrase, $ph->{minus_words}, $O{minus_words}, $block);
                push @{$geo_phrases->{$block->{geo} // ''}}, $ph;
            }
        }
    }

    my %advq_params = (
        function => $O{function} || 'search',
        map { $_ => $O{$_} } qw/timeout precision lang devices/
    );
    foreach my $geo (keys %$geo_phrases) {
        my @phrases_to_process = @{$geo_phrases->{$geo} || []};

        if (@phrases_to_process) {
            my @phrases_for_advq = uniq map { $_->{phrase_for_advq} } @phrases_to_process;

            my $last_month_stat = advq_get_stat(\@phrases_for_advq, $geo, %advq_params);
            my $last_month_hash = { map {$_->{phrase} => $_->{count}} @$last_month_stat };
            
            my $time_coef;
            if( defined $O{period}){
                $time_coef = advq_get_time_coef(
                    geo => $geo, phrases => \@phrases_for_advq,
                    period => $O{period}, lang => $O{lang},
                    devices => $O{devices}
                );
            }
            
            for my $ph (@phrases_to_process){
                my $phrase = $ph->{phrase_for_advq};
                my $tc = defined $O{period} ? ($time_coef->{$phrase}->{coef} || 0 ) : 1; 
                my $old_abs_count = defined $O{period} ? ($time_coef->{$phrase}->{old_abs_count} || 0 ) : 0; 
                my $current_count = $last_month_hash->{ $phrase };
                if ( ! defined $current_count ){
                    $ph->{showsForecast} = undef;
                    $ph->{showsTimeCoef} = undef;
                } elsif ($old_abs_count > 0) {
                    $ph->{showsForecast} = $old_abs_count;
                    $ph->{showsTimeCoef} = $old_abs_count * 1.0 / ($current_count || $old_abs_count);
                } else {
                    $ph->{showsForecast} = $last_month_hash->{ $phrase } * $tc;
                    $ph->{showsTimeCoef} = $tc;
                }
            }

            return '' unless scalar(@$last_month_stat) == scalar(@phrases_for_advq);
        }
    }
    return 'real';
}


=head2 advq_get_time_coef

    Вычисляет сезонные поправочные коэффициенты для прогноза показов 
    Основывается на "хроносрезе" из ADVQ

    Модель сезонности: 
        + линейный рост год от года 
        + одинаковое распределение показов по месяцам внутри каждого года (ДЛЯ ФИКСИРОВАННОЙ ФРАЗЫ). 

    Алгоритм работы: 
      Берем статистику показов по фразе за последние 13 месяцев: chrono[-1], chrono[-2], ... chrono[-13] 
        (особенность: хроносрезы считаются без учета минус-слов)
      Пусть нас интересует коэффициент для марта, и пусть предыдущий март -- это chrono[k]. 
      Коэффициент coef = chrono[k] / chrono[-13]

      Берем точную статистику (с минус-словами) за последние 30 дней: stat[-1]
      Прогноз на следующий март = stat[-1] * coef

    Справка ADVQ:
    wget -O - "http://ust2-1.advq.yandex.ru/advq/help"
    Пример хроносреза:
    wget -O - "http://ust2-1.advq.yandex.ru/advq/time_hist?words=dell"

    Параметры именованные: 
        geo 
        phrases -- ссылка на массив фраз 
        period  -- требуемый период: месяц или квартал, нумерация с 1
                   { month => 7 } или { quarter => 3 }
                   или год { year => 1 }

    Возвращаемое значение: 
        ссылка на хеш 
        {
            "<текст фразы>" => {coef => <коэффициент>, old_abs_count => <число>}
            ...
        }

    Коэффициент надо умножать на колчество показов за последние 30 дней.
    Если old_abs_count > 0 -- для низкочастотных фраз можно использовать его, 
    это сумма показов за соответствующий период прошлого года без учета минус-слов

=cut
sub advq_get_time_coef
{
    my (%O) = @_;
    my $profile = Yandex::Trace::new_profile('advq:advq_get_time_coef');
    my $ret = {};

    if ( $O{period}->{week} ){
        # Для подсчёта недельного коэффициента нам не нужно ходить в хроносрезы
        for my $ph (@{$O{phrases}}) {
            $ret->{$ph}->{coef} = 7/30;
        }
        return $ret;
    }

    # time_hist не понимает никакие операторы, поэтому мы их вырезаем и сохраняем соотвествие исходным фразам
    my %req2phrases;
    for my $ph (@{$O{phrases}}) {
        my $plus_ph = _cut_opers($ph);
        # отбрасываем фразы, целиком состоящие из стоп-слов
        next if all {is_stopword($_)} split /\s+/, $plus_ph;
        push @{$req2phrases{$plus_ph}}, $ph;
    }

    my $time_hist = advq_get_stat([keys %req2phrases], $O{geo},
                        timeout => 180, function =>"time_hist", 
                        lang => $O{lang}, devices => $O{devices});
 
    for my $rec ( @$time_hist ){
        # TODO вместо die обрабатывать ошибки
        die "advq_get_time_coef error: history length < 13 in ".to_json($rec) if @{$rec->{hist}} < 13;
        my $prev_base = $rec->{hist}->[-13];
        my $base = $rec->{hist}->[-1];
        die 'bases' unless $prev_base->{year} == $base->{year}-1 && $prev_base->{month} == $base->{month};
        @{$rec->{hist}} = @{$rec->{hist}}[-12..-1];
        my %shows_by_month = map { $_->{month} => $_->{total_count} } @{$rec->{hist}};
        die 'count' unless keys %shows_by_month == 12;

        my $c = 0;
        my $old_abs_count = 0;
        if ( $prev_base->{total_count} <= 0 && $base->{total_count} <= 500 ){
            # нет статистики за год назад и маленькая статистика за последний месяц 
            # -- переносим абсолютные числа из аналогичного периода прошлого года
            # аварийный вариант для сильно сезонных малочастотных фраз, у которых в низкие сезоны нулевые показатели
            if ( $O{period}->{month} ){
                $c = 1;
                $old_abs_count = $shows_by_month{$O{period}->{month}};
            } elsif ( $O{period}->{quarter} ){
                $c = 3;
                my $q = $O{period}->{quarter};
                my $shows = sum map { $shows_by_month{$_} } ( (1+($q-1)*3) .. $q*3 );
                $old_abs_count = $shows;
            } elsif ( $O{period}->{year} ) {
                $c = 12;
                my $shows = sum map { $shows_by_month{$_} } ( 1 .. 12 );
                $old_abs_count = $shows;
            }
        } elsif ( $prev_base->{total_count} <= 0 ){
            # нет статистики за "год назад", но за последний месяц показов много -- простая пропроциональность
            # похоже на частотную, но новую фразу
            if ( $O{period}->{month} ){
                $c = 1;
            } elsif ( $O{period}->{quarter} ){
                $c = 3;
            } elsif ( $O{period}->{year} ) {
                $c = 12;
            }
        } else {
            # статистика есть -- считаем из статистики
            if ( $O{period}->{month} ){
                $c = $shows_by_month{$O{period}->{month}} / $prev_base->{total_count};
            } elsif ( $O{period}->{quarter} ){
                my $q = $O{period}->{quarter};
                my $shows = sum map { $shows_by_month{$_} } ( (1+($q-1)*3) .. $q*3 );
                $c = $shows / $prev_base->{total_count};
            } elsif ( $O{period}->{year} ) {
                my $shows = sum map { $shows_by_month{$_} } ( 1 .. 12 );
                $c = $shows / $prev_base->{total_count};
            }
        }

        for my $ph (@{$req2phrases{$rec->{req}}}) {
            $ret->{$ph} = {
                coef => $c,
                old_abs_count => $old_abs_count,
            };
        }
    }

    for my $ph (@{$O{phrases}}){
        $ret->{$ph}->{coef} ||= 0;
    }

    return $ret;
}


=head2 advq_get_detailed_stat

    Return list of hashes for each matched phrase, or undef on ADVQ error;
    lang: -- language, "ru"|"tr" only
    calc_total_hits - default 1 - считать ли количество хитов основной фразы
    fast_mode - default 0 - не искать похожие звпросы

=cut

sub advq_get_detailed_stat($$$$%) {
    my ($geo, $phrase, $page, $page_size, %O) = @_;

    my $stats = advq_request([$phrase], regions  => $geo,
                                        ph_page_size => $page_size || 50,
                                        ph_page => $page,
                                        assocs => 1,
                                        devices => $O{devices},
                                        function => 'search',
                                        timeout  => $ADVQ_TIMEOUT,
                                        lang => $O{lang} // 'ru',
                                        calc_total_hits => $O{calc_total_hits} // 1,
                                        fast_mode => $O{fast_mode} // 0,
                                        video_advq => $O{video_advq},
                                        collections_advq => $O{collections_advq},
                            );


    return scalar @$stats && $stats->[0]->{including_phrases}
        ? $stats->[0]
        : undef;
}

=head2 advq_get_phrases_detailed_stat

    Return assotiations and including phrases

=cut

sub advq_get_phrases_detailed_stat
{
    my ($geo, $wlist, $page, $ph_page_size, $phrase_timeout, $timeout) = @_;
    
    $timeout = $timeout && $timeout + 0.0 > 0 ? $timeout : $ADVQ_TIMEOUT;
    return advq_request($wlist, regions => $geo,
                                ph_page => $page,
                                ph_page_size => $ph_page_size,
                                assocs => 1, 
                                function => 'search',
                                timeout => $timeout,
                                timeout_on_phrase => $phrase_timeout);
}

=head2 _cut_opers(phrase)

    вырезать из фразы все операторы: ! + - [ ] " "

=cut
sub _cut_opers {
    my ($phrase) = @_;
    $phrase =~ s/[\+\]\[\"]/ /g;              # Спецсимволы
    $phrase =~ s/\!//g;
    $phrase =~ s/\s+-\((?:\s*\S+\s*)+\)//g;     # Минус-фразы
    $phrase =~ s/\s+-\S+//g;                    # Минус-слова
    $phrase =~ s/\s+/ /g;
    $phrase =~ s/^ | $//g;
    return $phrase;
}

=head2 advq_forecast_multibudget

    Получение бюджета по фразам

    Параметры позиционные:
        $queries - array ref, фразы
        $geo - регионы
    Опциональные:
        timeout
        lang -- язык для леммера; по умолчанию используется Русский/Английский, указав "tr" можно переключиться на Турецкий
        devices - 
        min_bid (минимальная ставка, целое число, микрофишки)
        max_bid (максимальная ставка, целое число, микрофишки)
        positions - позиции по которым нужно возвращать данные
=cut

sub advq_forecast_multibudget {
    my ($queries, $geo, %O) = @_;

    my $timeout = $O{timeout} && $O{timeout} + 0.0 > 0 ? $O{timeout} : $ADVQ_TIMEOUT;
    return advq_forecast_request(
        $queries,
        function => 'multibudget',
        regions => $geo,
        timeout => $timeout,
        %{hash_cut(\%O, qw/lang devices min_bid max_bid positions/)},
    );
}

=head2 advq_forecast_budget_dist

    Получение распределения бюджета по фразам

    Параметры позиционные:
        $queries - array ref, фразы
        $geo - регионы
    Опциональные:
        timeout
        lang -- язык для леммера; по умолчанию используется Русский/Английский, указав "tr" можно переключиться на Турецкий
        devices - 
        min_bid (минимальная ставка, целое число, микрофишки)
        max_bid (максимальная ставка, целое число, микрофишки)
        bid_fractions (array ref, ставка для каждой фразы в микрофишках)
=cut

sub advq_forecast_budget_dist {
    my ($queries, $geo, %O) = @_;

    my $timeout = $O{timeout} && $O{timeout} + 0.0 > 0 ? $O{timeout} : $ADVQ_TIMEOUT;
    return advq_forecast_request(
        $queries,
        function => 'budget_dist',
        regions => $geo,
        timeout => $timeout,
        %{hash_cut(\%O, qw/lang devices min_bid max_bid bid_fractions/)},
    );
}

=head2 advq_forecast_request

    exec advq forecast requests

    Параметры позиционные:
        $queries - array ref
    Обязательные параметры:
        function -- функция forecast, которую надо запросить (multibudget|budget)
    Опциональные:
        regions
        timeout
        lang -- язык для леммера; по умолчанию используется Русский/Английский, указав "tr" можно переключиться на Турецкий
        devices - 
        min_bid (минимальная ставка, целое число, микрофишки)
        max_bid (максимальная ставка, целое число, микрофишки)
        positions - позиции по которым нужно возвращать данные

=cut

# NB: значение константы взято с потолка и совершенно не принципиально.
#     Т.к. http_parallel_request умеет выполнять несколько http запросов
#     одновременно, то нужно как-то идентифицировать каждый запрос для того,
#     что бы понять какой ответ какому запросу соответствует. В нашем случае
#     запрос один, поэтому конкретное значение идентификатор запроса не важно.
use constant HTTP_REQUEST_ID => 456;
use constant HTTP_MAX_FAILED_RESPONSE_CONTENT_LENGTH => 10_000;

sub advq_forecast_request {
    my ($queries, %O) = @_;

    my $function = $O{function} || die 'advq_forecast_request: missing function argument';

    my @params = map { defined($O{$_}) ? ($_ => $O{$_}) : () } qw/regions min_bid max_bid/;
    push @params, dbname => $O{lang} && $O{lang} eq 'tr' ? 'tur' : 'rus';
    push @params, devices => join ',', @{$O{devices}} if $O{devices};
    my @queries = ref $queries eq 'ARRAY' ? @$queries : ($queries);
    push @params, map{(query => $_)} @queries;
    my @bid_fractions = ref $O{bid_fractions} eq 'ARRAY' ? @{$O{bid_fractions}} : ();
    push @params, map{(bid_fraction => $_)} @bid_fractions;
    my $effective_timeout = $O{timeout} || $ADVQ_TIMEOUT;
    push @params, (timeout => $effective_timeout);
    push @params, positions => join ',' => @{$O{positions}} if $O{positions};

    my $trace_id = Yandex::Trace::current_span_id();
    my $trace_header = defined $trace_id ? join(',', map {$_//0} $trace_id, Yandex::Trace::generate_traceid(), $trace_id, 0) : undef;

    # Параллельный запрос + таймер
    my $profile = Yandex::Trace::new_profile('advq:advq_forecast_request', tags => $function, obj_num => scalar(@queries));

    my $forecast_url;
    if($USE_CLOUD_ADVQ_FORECAST->get(60)) {
        $forecast_url = $ADVQ_CLOUD_FORECAST_URL;
    } else {
        $forecast_url = $ADVQ_FORECAST_URL;
    }

    my $url = join('/' => $forecast_url, $function);

    my %reqs = (HTTP_REQUEST_ID() => {
        url => $url,
        body => \@params,
    });

    my $lock = _set_lock($effective_timeout);
    my %common_params = (
        timeout => 2 * $effective_timeout, # http таймаут должен быть дольше, чем advq-таймаут
        headers => {
            'X-Yandex-Trace' => $trace_header,
            'User-Agent' => undef,
            'Referer' => undef,
            'Content-Type' => 'application/x-www-form-urlencoded',
        },
        content_ref => 1,
    );

    my $responses = http_parallel_request(POST => \%reqs, %common_params);

    undef $profile;
    undef $lock;

    my $response = $responses->{HTTP_REQUEST_ID()};
    
    my $status = $response->{headers}->{Status} // 'status unknown';
    my $reason = $response->{headers}->{Reason} // 'reason unknown';
    my $elapsed = $response->{elapsed} // 'unknown';
    my $content = $response->{content_ref};

    if (!$response || ($response->{headers}{Status} !~ /^2\d{2}$/ && $response->{headers}{Status} !~ /^400$/)) {
        my $content_cut = substr($$content // 'undef', 0, HTTP_MAX_FAILED_RESPONSE_CONTENT_LENGTH);

        _forecast_log($trace_id, {url => $url, params => \@params, status => $status, reason => $reason, elapsed => $elapsed, content => $content_cut});

        die "ADVQ request faied: $status $reason\n";
    }

    my $log_data = {url => $url, status => $status, reason => $reason, elapsed => $elapsed};
    if ($REQUESTS_LOG) {
        $log_data->{words} = \@queries;
    }
    _forecast_log($trace_id, $log_data);

    # Разбираем ответ
    my $result = JSON::from_json($$content);

    if ($response->{headers}{Status} =~ /^400$/) {
        return {error => {message => $result->{message}, error => $result->{error}}};
    }

    return $result;
}

=head2 advq_get_phrases_multibudget_forecast_multi

    Получает из ADVQ прогноз мультибюджета по массиву блоков [групп или блоков медиаплана]
    Прогноз бюджета добавляется в каждую фразу в виде ключа forecast

    $blocks = [
        {
            geo =>
            phrases =>
            minus_words =>
            banner_minus_words =>
            campaign_minus_words =>
            currency => валюта запроса
            advq_min_bid => минимально возможная ставка в валюте
            advq_max_bid => максимально возможная ставка в валюте
        },
        ...
    ];
    %O = (
        minus_words =>
        timeout =>
        lang =>
        devices =>
        period =>
        positions =>
        with_nds =>
    );
    $result = advq_get_phrases_multibudget_forecast_multi($blocks, %O);
    $result => 'real'|''

=cut

sub advq_get_phrases_multibudget_forecast_multi {
    my ( $blocks, %O ) = @_;

    my %advq_params = %{hash_cut(\%O, qw/timeout lang devices positions log/)};

    for my $block (@$blocks) {
        my @queries;
        for my $ph (@{$block->{phrases}}) {
            my $phrase = defined $ph->{phrase_wo_common_minus_words} ? $ph->{phrase_wo_common_minus_words} : (defined $ph->{phrase} ? $ph->{phrase} : $ph->{phr});
            if (defined $phrase && $phrase ne '') {
                my $query = create_advq_request_phrase($phrase, $ph->{minus_words}, $O{minus_words}, $block);
                $ph->{phrase_for_advq} = $query;
                $ph->{forecast_result_index} = scalar @queries;
                push @queries, $query;
            }
        }
        if (@queries) {
            # конвертируем валюту в целые микрофишки для advq
            my $with_nds = $O{with_nds};
            my $currency = $block->{currency};
            my $min_bid = defined $block->{advq_min_bid} 
                ? int(_currency_to_advq(1e6 * $block->{advq_min_bid}, $currency, $with_nds)) : undef;
            my $max_bid = defined $block->{advq_max_bid} 
                ? int(_currency_to_advq(1e6 * $block->{advq_max_bid}, $currency, $with_nds)) : undef;

            my $filtered_advq_params = _filter_advq_params(\%advq_params);
            
            my $response = advq_forecast_multibudget(
                \@queries,
                $block->{geo},
                %$filtered_advq_params,
                min_bid => $min_bid,
                max_bid => $max_bid
            );

            if (exists $response->{error}) {
                $block->{error} = $response->{error};
                return;
            }
            
            my $results = $response->{results};
            for my $ph (@{$block->{phrases}}) {
                my $index = delete $ph->{forecast_result_index};
                next unless defined $index;
                $ph->{positions} = { map { $PlacePrice::ADVQ_TO_FORECAST_POSITION{delete $_->{pos}} => $_ } @{$results->[$index]->{stats}} };
                # если вызывающий запрашивал прогноз на четвертое СР, но из ADVQ оно не пришло, копируем с третьего
                # нужно когда-нибудь убрать, т.к. этот костыль в какой-то момент потеряет актуальность (ADVQ начнет выдавать прогнозы на 4СР)
                my $need_p14 = !defined($advq_params{positions}) || any { $_ eq 'P14' } @{$advq_params{positions}};
                if ($need_p14 && $ph->{positions}{third_premium} && !$ph->{positions}{fourth_premium}) {
                    $ph->{positions}{fourth_premium} = $ph->{positions}{third_premium};
                }

                my $coef = 30.0 / 28.0; # в прогнозаторе 28 дней, а в интерфейсе 30
                if (defined $ph->{shows_time_coef}) {
                    $coef *= $ph->{shows_time_coef};
                }
                for my $poskey (keys $ph->{positions}) {
                    my $pos = $ph->{positions}->{$poskey};
                    # Возвращаем только поддерживаемые и правильно откорректированные значения
                    $ph->{positions}->{$poskey} = {
                        bid => ceil(_currency_from_advq($pos->{bid}, $currency, $with_nds)),
                        ctr => $pos->{ctr},
                        budget => ceil(_currency_from_advq($pos->{budget} * $coef, $currency, $with_nds)),
                        clicks => ceil($pos->{clicks} * $coef),
                        shows => ceil($pos->{shows} * $coef),
                        shows_by_position => { map { $_ => ($pos->{shows} > 0 ? $pos->{shows_by_position}->{$_} / $pos->{shows} : 0) } keys %{$pos->{shows_by_position}} },
                    };
                }
            }
        }
    }
    return 'real';
}

sub _filter_advq_params {
    my ($advq_params) = @_;

    my %new_advq_params = %$advq_params;

    # пока ADVQ не поддерживает 4-е спецразмещение, отфильтровываем его из клиентского запроса
    # мы запомним, что клиент запрашивал цену 4-го, и если ADVQ вернет цену 3-го, то отдадим ее клиенту под
    # видом 4-го
    if ($new_advq_params{positions} && any { $_ eq 'P14' } @{$new_advq_params{positions}}) {
        $new_advq_params{positions} = [grep { $_ ne 'P14' } @{$new_advq_params{positions}}];
    }

    return \%new_advq_params;
}

=head2 advq_get_phrases_budget_dist_forecast_multi

    Получает из ADVQ прогноз распределения бюджета по массиву блоков [групп или блоков медиаплана]
    Распределение прогнозов бюджета добавляется в каждую фразу в виде ключа distribution, где:

        bid - распределение ставок (в микровалюте)
        budget - бюджет фразы с соответствующей ставкой (в микровалюте)
        clicks - кол-во кликов с соответствующей ставкой
        shows - кол-во показов с соответствующей ставкой

    В блоке в каждой фразе можно добавить параметры:

        advq_bid - желаемая ставка в валюте

    Желаемые ставки используются для поддержания соотношения ставок между фразами.

    $blocks = [
        {
            geo =>
            phrases =>
            minus_words =>
            banner_minus_words =>
            campaign_minus_words =>
            currency => валюта запроса
            advq_min_bid => минимально возможная ставка в валюте
            advq_max_bid => максимально возможная ставка в валюте
        },
        ...
    ];
    %O = (
        minus_words =>
        timeout =>
        lang =>
        devices =>
        period =>
        with_nds =>
    );
    $result = advq_get_phrases_budget_dist_forecast_multi($blocks, %O);
    $result => 'real'|''

=cut

sub advq_get_phrases_budget_dist_forecast_multi {
    my ( $blocks, %O ) = @_;

    my %advq_params = (
        map { $_ => $O{$_} } qw/timeout lang devices log/
    );
    for my $block (@$blocks) {
        my @queries;
        my @query_bids;
        my %advq_to_ph;
        my $ph_indexes = 0;
        my $currency = $block->{currency};
        for my $ph (@{$block->{phrases}}) {
            my $ph_index = $ph_indexes++;
            my $phrase = defined $ph->{phrase_wo_common_minus_words} ? $ph->{phrase_wo_common_minus_words} : (defined $ph->{phrase} ? $ph->{phrase} : $ph->{phr});
            if (defined $phrase && $phrase ne '') {
                my $query = create_advq_request_phrase($phrase, $ph->{minus_words}, $O{minus_words}, $block);
                $ph->{phrase_for_advq} = $query;
                $ph->{forecast_result_index} = scalar @queries;
                $advq_to_ph{scalar @queries} = $ph_index;
                my $query_bid = $ph->{advq_bid} || 1;
                push @queries, $query;
                push @query_bids, $query_bid;
            }
        }
        if (@queries) {
            # конвертируем валюту в целые микрофишки для advq
            my $with_nds = $O{with_nds};
            my $min_bid = defined $block->{advq_min_bid} 
                ? int(_currency_to_advq(1e6 * $block->{advq_min_bid}, $currency, $with_nds)) : undef;
            my $max_bid = defined $block->{advq_max_bid} 
                ? int(_currency_to_advq(1e6 * $block->{advq_max_bid}, $currency, $with_nds)) : undef;
            my @bid_fractions = map { int(_currency_to_advq(1e6 * $_, $currency)) } @query_bids;
            my $result = advq_forecast_budget_dist(
                \@queries,
                $block->{geo},
                %advq_params,
                min_bid => $min_bid,
                max_bid => $max_bid,
                bid_fractions => \@bid_fractions,
            );
            if (exists $result->{error}) {
                $block->{error} = $result->{error};
                return;
            }
            my $results_by_phrase = $result->{results};
            for my $ph (@{$block->{phrases}}) {
                my $index = delete $ph->{forecast_result_index};
                next unless defined $index;
                my $coef = 30.0 / 28.0; # в прогнозаторе 28 дней, а в интерфейсе 30
                if (defined $ph->{shows_time_coef}) {
                    $coef *= $ph->{shows_time_coef};
                }
                my $res = $results_by_phrase->[$index];
                $ph->{distribution} = {
                    bid => [map { ceil(_currency_from_advq($_, $currency, $with_nds)) } @{$res->{bid}}],
                    budget => [map { ceil(_currency_from_advq($_ * $coef, $currency, $with_nds)) } @{$res->{budget}}],
                    clicks => [map { ceil($_ * $coef) } @{$res->{clicks}}],
                    shows => [map { ceil($_ * $coef) } @{$res->{shows}}],
                    shows_by_position => _relative_shows_by_position($res->{shows_by_position}, $res->{shows}),
                };
            }
        }
    }
    return 'real';
}

=head2 create_advq_request_phrase

    Сцепить фразу с минус-словами и минус-фразами для создания строки-запроса в ADVQ

        phrase - базовая фраза (может содержать минус-слова)
        phrase_minus_words - arrayref с минус словами и минус-фразами уровня фразы
        optional_minus_words - arrayref с минус словами и минус-фразами общего уровня
        block - hashref, значениями которого являются дополнительные минус-фразы и минус-слова
            в формате, аналогичном minus_words

    Формат хешрефа block:

    $block = {
        minus_words =>
        banner_minus_words =>
        campaign_minus_words =>
    }

=cut

sub create_advq_request_phrase {
    my ($phrase, $phrase_minus_words, $optional_minus_words, $block) = @_;

    # К фразам с кавычками ничего не добавляем
    return $phrase if $phrase =~ m/^\s*\"/;

    my $group_minus_words = MinusWords::merge_private_and_library_minus_words($block->{minus_words}, [map {$_->{words}} @{$block->{library_minus_words} || []}]) || [];
    my @minus_objects;
    for my $mwords (grep { defined $_ && @$_ } ($phrase_minus_words, $optional_minus_words, $group_minus_words, $block->{banner_minus_words})) {
        push @minus_objects, @$mwords;
    }
    if ($block->{campaign_minus_words} && !$SKIP_CAMP_MINUS_WORDS_PROP->get(60)) {
        push @minus_objects, @{$block->{campaign_minus_words}};
    }

    return format_advq_query($phrase, minus_words => \@minus_objects);
}

=head2 format_advq_query

    Создать строку запроса в ADVQ

        phrase - базовая фраза (может содержать минус-слова)

    Именованные аргументы:

        minus_words - список минус слов
        remove_intersection - нужно ли убирать пересечения минус-слов с фразой
        polish_minus_words - нужно ли "полировать" минус фразы удаляя лишние конструкции. Не нужно, если задан ADVQ remove_intersection => 1

=cut

sub format_advq_query {
    my ($phrase, %O) = @_;

    # К фразам с кавычками ничего не добавляем
    return $phrase if $phrase =~ m/^\s*\"/;

    my $minus_objects_str = '';
    if ($O{minus_words} && @{$O{minus_words}}) {
        my $minus_objects = [uniq @{$O{minus_words}}];

        for my $mo (@$minus_objects) {
            # Мы не пропускаем одно слово в квадратных скобках в ADVQ
            $mo =~ s/^[-\s]*\[([^\s]+)\]\s*$/$1/;

            # Еще нужно проверить, что нет незафиксированных стоп-слов
            my @mo_words = split /\s+/, $mo;
            for my $mw (@mo_words) {
                if (Yandex::MyGoodWords::is_stopword($mw)) {
                    $mw = "!$mw";
                }
            }
            $mo = join ' ', @mo_words;
        }
        # Могут опять появиться дубликаты
        $minus_objects = [uniq @$minus_objects];

        if ($O{remove_intersection}) {
            my $invalid_minus_words = MinusWords::key_words_with_minus_words_intersection(key_words => [$phrase], minus_words => $minus_objects || [], max_overall_length => -1)->{minus_words};
            if ($invalid_minus_words && @$invalid_minus_words) {
                $minus_objects = Yandex::ListUtils::xminus($minus_objects, $invalid_minus_words);
            }
        } elsif ($O{polish_minus_words}) {
            # В процессе MinusWords::key_words_with_minus_words_intersection уже делался polish_minus_words_array
            $minus_objects = MinusWords::polish_minus_words_array($minus_objects);
        }

        if ($minus_objects && @$minus_objects) {
            $minus_objects_str = MinusWordsTools::minus_words_array2str_with_brackets_and_minus($minus_objects);
        }
    }

    $phrase =~ s/\[([^\s]+)\]/$1/g;
    my $query = $minus_objects_str ? join(' ', ($phrase, $minus_objects_str)) : $phrase;

    return $query;
}

sub _relative_shows_by_position {
    my ($showsbp, $shows) = @_;
    my $res_by_position = {};
    for my $pos (keys %$showsbp) {
        my $posshows = $showsbp->{$pos};
        my $res = $res_by_position->{$pos} = [];
        for my $i (0 .. $#{ $posshows }) {
            push @$res, $shows->[$i] > 0 ? $posshows->[$i] / $shows->[$i] : 0;
        }
    }
    return $res_by_position;
}

=head2 _currency_to_advq

    Конвертирует значение из валюты $currency в фишки advq

    $sum = _currency_to_advq($sum, $currency);

=cut

sub _currency_to_advq {
    my ($sum, $currency, $with_nds) = @_;
    $with_nds = 0 if !defined $with_nds;
    # В advq все данные в микрофишках, однако напрямую конвертировать из валюты в YND_FIXED нельзя
    # Т.к. в этом случае конвертация будет по фиксированному курсу фишек, а нам нужен текущий курс
    if (defined $currency && $currency ne 'YND_FIXED') {
        # Сначала конвертируем сумму в рубли
        $sum = convert_currency($sum, $currency, 'RUB', with_nds => $with_nds);
        # Затем конвертируем сумму в фишки
        $sum = convert_currency($sum, 'RUB', 'YND_FIXED', with_nds => $with_nds);
    }
    return $sum;
}

=head2 _currency_from_advq

    Конвертирует значение из фишек advq в валюту $currency

    $sum = _currency_from_advq($sum, $currency);

=cut

sub _currency_from_advq {
    my ($sum, $currency, $with_nds) = @_;
    $with_nds = 0 if !defined $with_nds;
    # В advq все данные в микрофишках, однако напрямую конвертировать из YND_FIXED в валюту нельзя
    # Т.к. в этом случае конвертация будет по фиксированному курсу фишек, а нам нужен текущий курс
    if (defined $currency && $currency ne 'YND_FIXED') {
        # Сначала конвертируем сумму в рубли
        $sum = convert_currency($sum, 'YND_FIXED', 'RUB', with_nds => $with_nds);
        # Затем конвертируем сумму в валюту
        $sum = convert_currency($sum, 'RUB', $currency, with_nds => $with_nds);
    }
    return $sum;
}

1;
