package API;

# $Id$
# API

use Direct::Modern;

use Carp;
use YAML::Syck;
use Data::Dumper;

use POSIX qw/strftime/;

use SOAP::Lite;

# По-другому SOAP::Lite не умеет получать soap-заголовки пользователя
use base 'SOAP::Server::Parameters';

use Storable qw/dclone/;
use JSON;

use Fcntl ':flock';
use vars qw($VERSION @ISA @EXPORT);

use Settings;

use APIMethods;
use API::Methods::Sandbox;
use API::Methods::Finance;
use API::Methods::Events;
use API::Methods::Staff;
use API::Methods::Tags;
use API::Methods::Mediaplan;
#use API::Methods::Prices;
use API::Methods::Clients;
use API::Methods::Retargeting;
use API::Methods::AdImage;
use API::Methods::Keyword;
use API::Methods::AccountManagement;
use API::Limits qw/has_spec_limits get_spec_limit/;
use API::Authorize qw/authorize_user/;
use API::Settings;

use APICommon qw(:subs);
use API::Errors;
use API::Validate;

use EnvTools;
use IpTools;
use Tools;
use MailService;
use DirectCache;
use Campaign::Types;
use Yandex::DBShards;
use Yandex::Trace qw/current_trace/;
use Yandex::Validate qw/is_valid_int/;
use Yandex::DateTime qw/now/;
use Primitives;
use PrimitivesIds;
use Rbac;
use Property;
use LogTools qw//;

use Yandex::HashUtils;

use Time::HiRes qw/time/;

use Yandex::Memcached::Lock;
use Yandex::I18n;
use Yandex::Log;
use Yandex::DBTools;

use List::MoreUtils qw/minmax/;

use base qw/DoCmd::Base/;

our $VERSION = 5;

#$SIG{__WARN__} = \&Carp::cluck;

our $api_version ||= 4; # минимальная поддерживаемая версия
our $plack_request; # глобальная переменная для передачи объекта Plack::Request

our $latest;
# дробный номер версии, например 4.5 для 4live
our $api_version_full ||=4;

# подверсии для обновлений wsdl
our $api_wsdl_subversion;


=head2 Отключение API4

    Ремарка  version => { more => 4 }, просто краткая запись для version => { versions => [4, 41, 42, 104, 5 ] }
    Этап 0. Метод включен в 4 и 4Live
    Method => {
        sub => {..},
        version => { more => 4 }
    }

    Этап 1. Начинаем иногда выдавать ошибку в 4ой версии
    Method => {
        sub => {..},
        version => { more => 4, deprecated => [4] }
    }

    Этап 2. Закрываем в 4ой версии совсем, но оставляем доступ специальным приложениям

    Method => {
        sub => {..},
        version => { versions => [ 104 ], also_exists_in_versions => [ 4 ] }
    }

    Этап 3. Закрываем в 4ой версии совсем, но оставляем доступ специальным приложениям, начинаем иногда сыпать ошибкой в 4Live

    Method => {
        sub => {..},
        version => { versions => [ 104 ], also_exists_in_versions => [ 4 ], deprecated => [ 104 ] }
    }

    Этап 4. Закрываем совсем в 4 и 4Live версии, но оставляем доступ приложениям с галочкой

    Method => {
        sub => {..},
        version => { versions => [ 6 ], also_exists_in_versions => [ 4, 104 ] }
    }

    Этап 5. Закрываем совсем, но возвращаем ошибку о том, что метод устарел

    Method => {
        sub => {..},
        version => { versions => [ 6 ] }
    }
    sub _app_has_access_to_restricted_methods { return 0 }

    Этап 5. Закрываем совсем

    Выпиливаем метод целиком начинаем отвечать ошибкой "метод не найден"

    Этап 6. Когда закрыты все методы отключаем URL

=cut

our %METHODS = (
    'CreateNewReport'   => [{
        sub => sub {},
        version => { versions => [6] },
    }],
    'GetReportList'     => [{
        sub => \&APIMethods::GetReportList,
        version => { versions => [6], also_exists_in_versions => [4,104] }
    }],
    'DeleteReport'      => [{
        sub => \&APIMethods::DeleteReport,
        version => { versions => [6], also_exists_in_versions => [4,104] }
    }],
    'GetSummaryStat'    => [{
        sub => \&APIMethods::GetSummaryStat,
        version => { versions => [6], also_exists_in_versions => [104] }
    }],
    'GetBannersStat'    => [
            {
                sub => sub {},
                version => {versions => [6]},
            }
        ],

    'GetRegions'        => \&APIMethods::GetRegions,
    'GetRubrics'        => \&APIMethods::GetRubrics,

    'GetTimeZones'      => \&APIMethods::GetTimeZones,

    'GetVersion'        => \&APIMethods::GetVersion,
    'PingAPI'           => \&APIMethods::PingAPI,
    'PingAPI_X'         => \&APIMethods::PingAPI_X,

    'CreateNewForecast' => [{
        sub => \&APIMethods::CreateNewForecast,
        version => { versions => [104,5] }
    }],
    'GetForecastList'   => \&APIMethods::GetForecastList,
    'GetForecast'       => [{
        sub => \&APIMethods::GetForecast,
        version => { versions => [104,5] }
    }],
    'DeleteForecastReport' => \&APIMethods::DeleteForecastReport,

    'GetBalance'        => [{
        sub => \&API::Methods::Finance::GetBalance,
        version => { versions => [6] }
    }],

    # API::Methods::Clients
    'GetClientInfo'     => [{
        sub => \&API::Methods::Clients::GetClientInfo,
        version => { versions => [6], also_exists_in_versions => [104] }
    }],
    'UpdateClientInfo'     => [{
        sub => \&API::Methods::Clients::UpdateClientInfo,
         version => { versions => [6], also_exists_in_versions => [4,104] }
    }],
    'GetClientsList'    => [{
        sub => \&API::Methods::Clients::GetClientsList,
        version => { versions => [6], also_exists_in_versions => [104] }
    }],
    'GetSubClients'     => [{
        sub => \&API::Methods::Clients::GetSubClients,
        version => { versions => [6], also_exists_in_versions => [4,104] }
    }],
    'GetClientsUnits'   => \&API::Methods::Clients::GetClientsUnits,
    'CreateNewSubclient' => [{
        sub => \&API::Methods::Clients::CreateNewSubclient,
        version => { versions => [6], also_exists_in_versions => [4,104,5] }
    }],

    # set banners status
    'ModerateBanners'   => [{
        sub => \&APIMethods::ModerateBanners,
        version => { versions => [6], also_exists_in_versions => [4, 104] }
    }],

    'GetAvailableVersions' => \&APIMethods::GetAvailableVersions,

    'GetKeywordsSuggestion' =>
                [
                    {
                        sub => \&APIMethods::GetKeywordsSuggestion,
                        sandbox_sub => \&API::Methods::Sandbox::GetKeywordsSuggestion,
                        version => {more => 4}
                    }
                ],
    'GetNormalizedKeywords' => [
            {
                sub => \&APIMethods::GetNormalizedKeywords,
                version => {versions => [104,5]},
            }
        ],
    'CreateNewWordstatReport' => \&APIMethods::CreateNewWordstatReport,
    'GetNormalizedKeywordsData' => [
            {
                sub => \&APIMethods::GetNormalizedKeywordsData,
                version => {versions => [104,5]},
            }
        ],
    'GetWordstatSync' => [
                    {
                        sub => \&APIMethods::GetWordstatSync,
                        version => {versions => [104,5]},
                    }
                ],
    'GetForecastSync' => [
                    {
                        sub => \&APIMethods::GetForecastSync,
                        version => {versions => [104,5]},
                    }
                ],
    'GetWordstatReportList' => \&APIMethods::GetWordstatReportList,
    'GetWordstatReport' => \&APIMethods::GetWordstatReport,
    'DeleteWordstatReport' => \&APIMethods::DeleteWordstatReport,

    'GetStatGoals' => \&APIMethods::GetStatGoals,

    'GetEventsLog' =>
                [
                    {
                        sub => \&API::Methods::Events::GetEventsLog,
                        version => {versions => [104,5]},
                    }
                ],

    'GetCampaignsTags' =>
                [
                    {
                        sub => \&API::Methods::Tags::GetCampaignsTags,
                        version => {versions => [104,5]},
                    }
                ],

    'UpdateCampaignsTags' =>
                [
                    {
                        sub => \&API::Methods::Tags::UpdateCampaignsTags,
                        version => {versions => [104,5]},
                    }
                ],
    'GetBannersTags' =>
                [
                    {
                        sub => \&API::Methods::Tags::GetBannersTags,
                        version => {versions => [104,5]},
                    }
                ],

    'UpdateBannersTags' =>
                [
                    {
                        sub => \&API::Methods::Tags::UpdateBannersTags,
                        version => {versions => [104,5]},
                    }
                ],

    'GetMetroStations' =>
                [
                    {
                        sub => \&APIMethods::GetMetroStations,
                        version => {more => 5},
                    }
                ],

    'SaveSubscription' => [
                {
                    sub => \&API::Methods::Events::SaveSubscription,
                    version => {versions => [104,5]},
                }
            ],
    'DeleteSubscription' => [
                {
                    sub => \&API::Methods::Events::DeleteSubscription,
                    version => {versions => [104,5]},
                }
            ],
    'GetSubscription' => [
                {
                    sub => \&API::Methods::Events::GetSubscription,
                    version => {versions => [104,5]},
                }
            ],
    'SearchClients' => [
                {
                    sub => \&API::Methods::Staff::SearchClients,
                    version => {versions => [104,5]},
                }
            ],
    'GetKeywordsIntersection' => [
                {
                    sub => \&API::Methods::Keyword::GetKeywordsIntersection,
                    version => {versions => [104,5]},
                }
    ],
    'RearrangeKeywords' => [
                {
                    sub => \&API::Methods::Keyword::RearrangeKeywords,
                    version => {versions => [104,5]},
                }
    ],

    'GetRetargetingGoals' => [
                {
                    sub => \&API::Methods::Retargeting::GetRetargetingGoals,
                    version => {versions => [104,5]},
                }
            ],
    'RetargetingCondition' => [
                {
                    sub => \&API::Methods::Retargeting::RetargetingCondition,
                    version => {versions => [6], also_exists_in_versions => [104]},
                }
            ],
    'Retargeting' => [
                {
                    sub => \&API::Methods::Retargeting::Retargeting,
                    version => {versions => [6], also_exists_in_versions => [104]},
                }
            ],
    'AdImage' => [
                {
                    sub => \&API::Methods::AdImage::AdImage,
                    version => {versions => [104,5]},
                }
            ],
    'AdImageAssociation' => [
                {
                    sub => \&API::Methods::AdImage::AdImageAssociation,
                    version => {versions => [104,5]},
                }
            ],
# Новый API:
    'Mediaplan' => [{
        sub => \&API::Methods::Mediaplan::MediaplanWrapper,
        version => {versions => [6]},
    }],
    'MediaplanAdGroup' => [{
        sub => \&API::Methods::Mediaplan::MediaplanAdGroupWrapper,
        version => {versions => [6]},
    }],
    'MediaplanAd' => [{
        sub => \&API::Methods::Mediaplan::MediaplanAdWrapper,
        version => {versions => [6]},
    }],
    'MediaplanKeyword' => [{
        sub => \&API::Methods::Mediaplan::MediaplanKeywordWrapper,
        version => {versions => [6]},
    }],
    'MediaplanCategory' => [{
        sub => \&API::Methods::Mediaplan::MediaplanCategoryWrapper,
        version => {versions => [6]},
    }],
# финансовые операции:

    'TransferMoney' => [{
        sub => \&API::Methods::Finance::TransferMoney,
        version => { versions => [104,5] }
    }],
    'GetCreditLimits' => \&API::Methods::Finance::GetCreditLimits,
    'CreateInvoice' => [{
        sub => \&API::Methods::Finance::CreateInvoice,
        version => { versions => [104,5] }
    }],
    'PayCampaigns' => [{
        sub => \&API::Methods::Finance::PayCampaigns,
        version => { versions => [104,5] }
    }],
    'PayCampaignsByCard' => [
                {
                    sub => \&API::Methods::Finance::PayCampaignsByCard,
                    version => {versions => [104,5]},
                }
    ],
    'CheckPayment' =>  [
                {
                    sub => \&API::Methods::Finance::CheckPayment,
                    version => {versions => [104,5]},
                }
    ],

    'AccountManagement' => [
                {
                    sub => \&API::Methods::AccountManagement::AccountManagement,
                    version => {versions => [104,5]},
                }
    ],
    
    'EnableSharedAccount' => [
                {
                    sub => \&API::Methods::Finance::EnableSharedAccount,
                    version => {versions => [104,5]},
                }
    ],

#    '' => \&APIMethods::,
);

my %DEPRECATED_METHODS;
foreach my $method (keys %METHODS) {
    next unless ref $METHODS{$method} eq 'ARRAY';

    foreach my $item (@{$METHODS{$method}}) {
        if (exists $item->{version} && exists $item->{version}{deprecated}) {
            $DEPRECATED_METHODS{$method}{$_} = 1 foreach @{$item->{version}{deprecated}};
        }
    }
}

# Финансовые методы, которые требуют дополнительной валидации по мастер токену

our %FIN_METHODS = (
    'TransferMoney' => 1,
    'GetCreditLimits' => 1,
    'CreateInvoice' => 1,
    'PayCampaigns' => 1,
    # 'PayCampaignsByCard' => 1, # не требует авторизацию по токену
    # 'CheckPayments' => 1, # не требует авторизацию по токену
    'AccountManagement Invoice' => 1,
    'AccountManagement Deposit' => 1,
    # 'AccountManagement DepositByCard' => 1, # не требует авторизации по токену
    # 'AccountManagement CheckPayments' => 1, # не требует авторизацию по токену
    'AccountManagement TransferMoney' => 1,
);

our %TEST_METHODS = (
# все методы перенесли в intapi но хеш еще может пригодиться
);

# методы, для которых не требуется формировать ответ ( ответ формируется внутри самого метода )
our %STREAM_METHODS = (
    'GetCampaignsSpecialStat' => 1,
    'GetSearchQueryReport' => 1,
);

#
# функция запускается, если не найден вызванный метод в данном модуле
#

sub AUTOLOAD {
    our $AUTOLOAD;

    # лога запросов к API
    my $log = new Yandex::Log(
        log_file_name => "APIRequests.log",
        date_suf => "%Y%m%d",
        $Settings::BETA_SYSLOG_PREFIX ? (no_log => 1, use_syslog => 1, syslog_prefix => $Settings::BETA_SYSLOG_PREFIX) : (),
    );
    Yandex::DBShards::clear_cache();
    Campaign::Types::clear_cache();
    Rbac::clear_cache();

    # в переменной AUTOLOAD будет записано API::MethodName
    # достаем название необходимого метода
    my $sub = $AUTOLOAD;
    (my $method = $sub) =~ s/.*:://;

    # очень хочется эту интерфейсно-зависимую часть перенести в соотв контролеры
    my $headers = pop;

    my $api_interface = 'json';
    if (ref($headers) eq 'SOAP::SOM') {
        $api_interface = 'soap';
        $headers = $headers->header;
    }

    my ($token, $persistent_token, $login, $finance_token, $operation_num, $payment_token) = ('', '', '', '', '', '');
    # TODO: после релиза коммандера удалить все, что связано с этим полем
    # при его передаче пускаем клиентов агентств с доступом
    my $agcl;
    my $self = {};

    $self->{locale} = 'non';

    if (ref $headers eq 'HASH') {
        # если заголовок в запросе есть, но он пустой - <SOAP:Headers/>
        #   то SOAP::Lite делает $headers скаляром...

        $token = $headers->{token};
        $persistent_token = $headers->{persistent_token};
        $finance_token = $headers->{finance_token};
        $operation_num = $headers->{operation_num};
        $payment_token = $headers->{payment_token};
        $login = $headers->{login};
        $self->{locale} = lc($headers->{locale} || 'non');

        $agcl = $headers->{agcl};
    }

    #print STDERR 'AUTOLOAD: '.$AUTOLOAD."\n";
    #print STDERR Dumper ['@_',\@_];

    # если существует стоп-файл - ничего не делаем, отдыхаем
    if (-f $Settings::WEB_STOP_FLAG) {
        # не логируется - и не нужно писать это в ppclog
        dieSOAP('503');
    }
    
    my $reqid = Yandex::Trace::trace_id();

    # хэш с параметрами для логирования
    my $log_data = {
        start_time => Time::HiRes::time(),
        reqid => $reqid,
        # чтобы в логе отличать версию от latest - прибавляем 100
        api_version => $api_version < 100 && $latest ? (100 + $api_version) : $api_version,
        api_interface => $api_interface,
        method => $method,
    };

    # для отладки нагрузки на бд
    # $Yandex::DBTools::QUERIES_LOG = "dbquery-API-$method-".int(time()).".log";

    my $auth_res = authorize_user(token => $token, persistent_token => $persistent_token,
                                  login => $login, remote_addr => $plack_request->address, reqid => $reqid);

    if (! $auth_res) {
        $log->out("UserInvalid: UID not found: token=".substr($token||'', 0,1).'..'.substr($token||'', -3));
        log_and_dieSOAP($log_data, 'UserInvalid');
    }
    if ($auth_res->{error}) {
        log_and_dieSOAP($log_data, $auth_res->{error}, $auth_res->{error_detail});
    } else {
        $self->{authorization_token} = $token;
        $self->{app_min_api_version_override} = $persistent_token ? 4
            : $auth_res->{access} ? $auth_res->{access}->min_api_version_override//0 : 0;
    }

    $self->{uid} = $log_data->{uid} = $auth_res->{uid};
    local $LogTools::context{UID} = $self->{uid};
    $self->{application_id} = $log_data->{application_id} = $auth_res->{application_id};

    $log_data->{login} = $login || $auth_res->{login};

    # проверяем доступность указанной версии АПИ
    my $ver_avail = check_avaliable_version($api_version, $latest);
    if ($ver_avail eq 'not') {
        log_and_dieSOAP($log_data, 'NotSupportedVersion');
    }

    if ($self->{locale} ne 'non' && !Yandex::I18n::is_valid_lang($self->{locale})) {
        log_and_dieSOAP($log_data, 'UnknownLocale', iget('Неизвестный язык %s', $self->{locale}));
    }

    # TODO: нужно один раз на запрос выбирать ВСЕ атрибуты пользователя и использовать их далее по коду
    # сейчас на каждый запрос делается неск-ко запросов в таблицу users для получения информации про оператора запроса
    # сейчас часть параметров сохраняется в $self->{user_info} в функции api_initialize

    $self->{plack_request} = $plack_request;
    $self->{api_version} = $api_version;
    $self->{latest} = $latest;
    $self->{api_version_full} = $api_version_full;

    $self = api_initialize($self);

    # ошибки отсюда не логируем - это только для супер пользователей, так что не нужно
    # фэйковая авторизация для супер-пользователей
    my $is_fake_auth = 0;

    if (! is_production() && $headers->{fake_login}) {
        $log->out("Fake login start", { operator_login => $self->{operator_login}, fake_login => $headers->{fake_login}, method => $method });
        $is_fake_auth = 1;
        $self->{operator_login} = $headers->{fake_login};
        $self->{uid} = $log_data->{uid} = get_uid_by_login2($self->{operator_login});

        if ( ! is_ip_in_list(($ENV{'X-Real-IP'} || $ENV{REMOTE_ADDR}), $Settings::INTERNAL_NETWORKS) ) {
            $log->out("Fake login error: wrong ip: ".($ENV{'X-Real-IP'} || $ENV{REMOTE_ADDR}));
            dieSOAP('AccessDenied', iget('Доступ запрещен не из внутренней сети'));
        }
        if (! $self->{uid}) {
            $log->out("Fake login error: wrong user: $self->{uid}");
            dieSOAP('UnknownUser');
        }

        $self = api_initialize($self);
    }

    # добавляем UID к комментариям sql-запросов
    my $old_vars = $Yandex::DBTools::TRACE_COMMENT_VARS;
    local $Yandex::DBTools::TRACE_COMMENT_VARS = sub {return {%{$old_vars->()}, operator => $self->{uid}}; };

    if ($ver_avail eq 'blocked') {
        # проверяем наличия доступа к версии для конкретного пользователя
        unless (check_access_user_version($self->{uid})) {
            log_and_dieSOAP($log_data, 'AccessDenied', iget('Доступ к указанной версии API запрещен'));
        }
    }

    if (! $self->{user_info}{login}) {
        log_and_dieSOAP($log_data, 'UnknownUser');
    }

    # проверяем имеет ли пользователь доступ к АПИ
    my ($res_access, $error) = api_check_user_access($self, $agcl);
    if ($res_access) {
        # вопрос отдавать такую ошибку, либо прятать за NoRights или UserInvalid
        # кажется, что UserInvalid - безопаснее, но слишком много будет писем в саппорт от обычных пользователей
        log_and_dieSOAP($log_data, $error || 'AccessDenied', $res_access);
    }

    if ($API::STREAM_METHODS{$method} && $api_interface ne 'json') {
        # иначе SOAP::Lite портит ответ
        log_and_dieSOAP($log_data, 'AccessDenied', iget('Доступ к методу возможен только по json-протоколу'));
    }
    # проверяем стоп-файл для части контроллеров
    if (Tools::check_available_cmd_by_stop_file('api', $method)) {
       log_and_dieSOAP($log_data, 'TmpCmdUnavailable');
    }

    $self->{method} = $method;
    my $src_params = $_[1];
    if (ref $src_params && ref $src_params eq 'HASH' && exists $src_params->{Action}) {
        $self->{action} = $src_params->{Action};
    }

    # если метод существует и доступен - то выполняем, иначе ругаемся
    my $check_method = check_available_api_method($self);

    if ($check_method->{res} eq 'available') {
        # если метод доступен в указанной версии API

        # пытаемся выставить memcached lock, если доступны сервера, в противном случае пропускаем
        my $max_locks_num;
        if (defined $self->{user_info}->{ClientID}) {
            if (defined has_spec_limits($self->{user_info}{ClientID})
                  && defined get_spec_limit($self->{user_info}{ClientID}, 'api_special_simultaneously_conn_limit')
            ) {
                $max_locks_num = get_spec_limit($self->{user_info}{ClientID}, 'api_special_simultaneously_conn_limit');
            } elsif (defined (my $limit = $self->{special_options}->{concurrent_calls})) {
                $max_locks_num = $limit;
            }  else {
                $max_locks_num = $Settings::API_PARALLEL_QUERIES;
            }
        }

        my $mcl = Yandex::Memcached::Lock->new(
            servers => $Settings::MEMCACHED_SERVERS, entry => $self->{uid}, max_locks_num => $max_locks_num, no_auto_lock => 0);

        # если сервера для локов недоступны, то не используем локи вообще.
        my $locked = defined $mcl->get_lock() ? $mcl->get_lock() : 1;

        my ($soap_result, $soap_status) = ('', 0);
        my $clone_params = ref $src_params ? Storable::dclone($src_params) : $src_params;

        $self->{uhost}->reset_reserved_units();

        my $soap_error_detail = '';
        my $last_die_stacktrace;
        local $Carp::MaxArgLen = 4096;

        eval {
            local $SIG{__DIE__} = sub { $last_die_stacktrace = Carp::longmess(); };
            local $SIG{__WARN__} = sub {
                my $req_id = Yandex::Trace::current_span_id();
                warn strftime("%Y-%m-%d %H:%M:%S", localtime( time() ) )
                    . ($req_id ? " [$req_id]: " : ': ')
                    . join(' ', @_);
                return 1;
            };

            # если метод финансовый - проводим дополнительную проверку
            if ( $check_method->{fin} && !$is_fake_auth ) {
                # запрещаем внутренним ролям
                if ($self->{rbac_login_rights}->{role} =~ /^(super|support|media|placer|superreader)$/) {
                    dieSOAP('NoRights', iget('Вызов финансовых методов запрещен для внутренних ролей'));
                }
                if (!$payment_token && !$finance_token) {
                    # пока не палим наружу платежные токены
                    #dieSOAP('PayTokenNotProvided', iget('Не передан финансовый токен или платежный токен'));
                    dieSOAP('FinanceTokenInvalid', iget('Не передан финансовый токен'));
                } elsif ( !$payment_token && $finance_token && !$operation_num ) {
                    dieSOAP('FinanceTokenInvalid', iget('Не передан номер операции для финансового токена'));
                } elsif (!is_valid_int($operation_num, 1, $API::Settings::API_MAX_FIN_OPERATION_NUM)) {
                    dieSOAP('InvalidFinOperationNum', iget('Номер финансовой операции должен быть целым числом от 1 до %s', $API::Settings::API_MAX_FIN_OPERATION_NUM));
                } elsif ($payment_token) {
                    # Здесь валидация payment_token через API ЯДа
                    my $pay_token_status = check_payment_token($self->{uid}, $payment_token);
                    if (!$pay_token_status) {
                        $self->{payment_token} = $payment_token;
                    } elsif ($pay_token_status == 1) {
                        dieSOAP('InvalidPaymentToken');
                    } elsif ($pay_token_status == 3) {
                        dieSOAP('AnothersPaymentToken', iget('Если вы хотите оплачивать рекламные кампании с другого логина, зарегистрируйте его в качестве представителя'));
                    } else {
                        dieSOAP('FinOpsTmpUnavail');
                    }
                } else {
                    if ( my @check_avail_errors = check_fin_operations_available($self->{uid}) ) {
                        dieSOAP(@check_avail_errors);
                    }
                    my @finance_errors = check_finance_token($self->{uid}, $finance_token, $method, $operation_num, ($login || $auth_res->{login}), ($self->{action} || ''));
                    if (@finance_errors) {
                        update_finance_operation_stat($self->{uid}, $self->{rbac}, fault => 1);
                        dieSOAP(@finance_errors);
                    }
                }
            }

            if (! $locked) {
                dieSOAP('506');
            }

            # чтобы записывать в лог ошибки, связанные с превышением ограничений
            api_check_limit_autoload($self, $self->{uid}, $method, $self->{rbac_login_rights});

            my $direct_cache = new DirectCache(groups => ['lemmer']);

            # запрещаем вызывать метод, если его разрешено вызвать не более, чем 0 раз
            if (defined (my $limit = $self->{special_options}->{method_limits}->{$method})) {
                return ('AccessDenied') unless $limit;
            }

            my $client_shard = get_shard(uid => $self->{uid}) || 1;

            # валидируем параметры до вызова метода.
            # Если функции в API.Validate нет, то оставляем валидацию на откуп методу
            $self->{ret} = [];
            my @validation_result;
            {
                @validation_result = API::Validate::validate_params( $self, $method, $src_params, not_die_on_no_validate => 1 );
            }

            if (@validation_result) {
                # если найдена ошибка
                dieSOAP(@validation_result);
            } else {
                ($soap_result) = $check_method->{sub}->($self, $src_params);

                if (is_beta()) {
                    if (ref $soap_result eq 'HASH'
                            && defined $soap_result->{ActionsResult}
                            && scalar grep {exists $_->{Errors} || exists $_->{Warnings}} @{ $soap_result->{ActionsResult} }
                    ) {
                        # на бетах логируем результат в БД при положительном ответе и наличии ошибок, ии предупреждений в ответе
                        my $json = JSON->new->allow_nonref;
                        $soap_error_detail = $json->encode($soap_result);
                    }
                }
            }

            $direct_cache->on_request_end;
        };

        my $soap_fault;
        if( $@ ) {
            $soap_fault = $@;
            if( ref($@) eq 'SOAP::Fault' ) {
                $soap_status = $@->{_faultcode};
                $soap_error_detail = $@->{_faultdetail};
            } else {
                $soap_status = 500;
                warn strftime("%Y-%m-%d %H:%M:%S\: ", localtime( time() ))
                    . Dumper {"Strange error" => $soap_fault};
                send_die_letters($soap_fault, "$method\n$last_die_stacktrace");
                my $die_details;
                $die_details = "DEVTEST/$method: ".Dumper($soap_fault) if !is_production();
                $soap_fault = dieSOAP('500', $die_details, dont_die => 1);
            }
        }
        my $units_call_cost = 0;
        my $call_units = $self->{uhost}->get_units();
        for(@$call_units){
            $units_call_cost += $_->{UnitsCallCost};
        }

        hash_merge $log_data, {
            units_stats => $self->{uhost}->get_units_stats(),
            units => $units_call_cost,
            cluid => $self->{cluid} || [],
            soap_status => $soap_status,
            error_detail => $soap_error_detail,
        };

        APICommon::soap_log_request($log_data, $clone_params, $self->{syslog_data});

        # удалять лок нужно до того, как умираем
        unless ($mcl->no_auto_unlock()) {
            $mcl->unlock();
        }

        if( $soap_fault && $soap_status) {
            die $soap_fault;
        }

        api_update_limit_uid($self, $self->{uid}, $method, 1);

        my $full_method_name = $method;
        $full_method_name .= ' ' . $self->{action} if defined $self->{action};
        if ($FIN_METHODS{$full_method_name}) {
            update_finance_operation_stat($self->{uid}, $self->{rbac}, success => 1, operation_num => $operation_num);
        }

        # $api_version нужен соапизатору
        my @api_answer = ($soap_result, $api_version);

        # если тратились баллы - обновляем и добавляем в заголовок ответа
        # ВОПРОС: что будет в JSON ? как там сообщать баллы ?
        if ($units_call_cost) {
            unless ($Settings::UIDS_NO_LIMITS_API{$self->{uid}} || $self->{rbac_login_rights}->{role} eq 'manager') {
                # обновляем юниты для всех, кроме определенного списка
                $self->{uhost}->update_units();
            }

            # порядок в массиве ответа сервера не имеет значения т.к. сортировка внутри SOAP::Lite происходит по типу(SOAP::Header)
            push @api_answer, SOAP::Header->name(Units => $call_units);
        }

        if( $self->{auction_units} ) {
            push @{$plack_request->env->{extra_headers} //= []}, GetPhrasesLimit => auction_units_to_headers($self->{auction_units});
        }

        return @api_answer;

    } elsif ($check_method->{res} eq 'not_available') {
        # если метод существует, но в данной версии API не поддерживается
        log_and_dieSOAP($log_data, 'MethodNotAvailable');

    } elsif ($check_method->{res} eq 'deprecated') {
        my $msg = $self->{latest}
            ? iget('Метод %s API версии Live 4 устарел. Доступ к нему скоро будет отключен. Используйте API версии 5 (см. https://tech.yandex.ru/direct/doc/migration/)', $method)
            : iget('Метод %s API версии 4 устарел. Доступ к нему скоро будет отключен. Используйте API версии 5 (https://tech.yandex.ru/direct/doc/migration/)', $method);

        log_and_dieSOAP($log_data, 'MethodDeprecated', $msg);

    } else {
        # если метод не существует
        log_and_dieSOAP($log_data, 'MethodNotExist');
    }
}

=head2 auction_units_to_headers($auction_units_struct)

    Конвертируем данные о торгобаллов в http заголовок
    $auction_units_struct - хэш с параметрами
        spent - потрачено за текущий запрос (может быть не задано)
        balance - потрачено в текущем периоде (читай за последние 24 часа)
        limit - лимит на период (на 24 часа)
        seconds_till_next_interval - секунд до следующего интервала, когда будут начисленны балы за час
        start_time - epoch-время начала запроса в секундах, если задано, то будет вычтено из seconds_till_next_interval

    Если метод попал на смену часа то возвращаем нуль в качестве
    seconds_till_next_interval, тем самым показывая что прибавка в баллах уже
    произошла, но еще не отразилась в заголовке

    Возвращает строку для передачи как значение заголовка

=cut

sub auction_units_to_headers {
    my $units = shift;

    $units->{spent} //= 0;
    my $start_time = $units->{start_time};
    my $runtime = Time::HiRes::time() - $start_time;
    my $timer = int( $units->{seconds_till_next_interval} - $runtime );
    $units->{seconds_till_next_interval} = $timer >= 0 ? $timer : 0;

    return join('/', @{$units}{qw/spent balance limit seconds_till_next_interval/}) . ' secs';
}

=head2 check_available_api_method

    Функция проверяет доступность метода для данной версии API

=cut
{
    my $METHODS_BY_VERSION;

sub check_available_api_method
{
    my ($self) = @_;
    my $result = {};
    my $version = $self->{api_version};
    $version += 100 if $self->{latest};

    $METHODS_BY_VERSION ||= _get_methods_hash();

    if (defined $METHODS_BY_VERSION->{$self->{method}} && (!$TEST_METHODS{$self->{method}} || !is_production())) {
        $self->{app_has_access_override} = _app_has_access_to_restricted_methods($version, $self->{app_min_api_version_override});
        if (defined $METHODS_BY_VERSION->{$self->{method}}{$version}) {
            if (exists $DEPRECATED_METHODS{$self->{method}}
                && $DEPRECATED_METHODS{$self->{method}}{$version}
                && !$self->{app_has_access_override}
                && need_error_for_deprecated_method($self->{application_id})
            ) {
                $result = { res => 'deprecated' };
            } elsif ($METHODS_BY_VERSION->{$self->{method}}{$version}{only_available_to_selected_apps} && !$self->{app_has_access_override}) {
                $result = {
                    res => 'not_available'
                    , detail => iget("Метод %s не доступен в %s версии Direct API", $self->{method}, $version)
                };
            } else {
                my $fin_method_action = (exists $self->{action})? $self->{method}.' '.$self->{action} : $self->{method};
                $result = {
                    res => 'available'
                    , fin => $FIN_METHODS{$fin_method_action} ? 1 : 0
                    , sub => $METHODS_BY_VERSION->{$self->{method}}{$version}->{handler}
                };
            }
        } else {

            $result = {
                res => 'not_available'
                , detail => iget("Метод %s не доступен в %s версии Direct API", $self->{method}, $version)
            };
        }
    } else {
        $result = {res => 'not_exist'};
    }

    return $result;
}
}

=head2 _get_methods_hash

    По основному описанию доступности методов строит хэш

=cut

sub _get_methods_hash
{
    my $METHODS_HASH = {};

    # минимальная и максимальная доступная версия api
    my ($min_version, $max_version) = minmax grep {/^\d+$/} keys %Settings::AVAILABLE_API_VERSIONS;

    foreach my $method (keys %METHODS) {

        my $desc = $METHODS{$method};
        if (ref $desc eq 'CODE') {
            # если версия метода одна

            foreach my $v ($min_version .. $max_version) {
                $METHODS_HASH->{$method}{$v} = { handler => $METHODS{$method} };
                if ($v <= $max_version) {
                    $METHODS_HASH->{$method}{$v + 100} = { handler => $METHODS{$method} };
                }
            }

        } elsif (ref $desc eq 'ARRAY') {
            # перебираем все записи для метода
            #   и заполняем массив доступности версий методов

            my $mversions = {};

            foreach my $s (@$desc) {
                if (defined $s->{version}{versions}) {
                    # параметр versions содержит список версий, для которых доступна данная версия метода
                    foreach my $v (@{$s->{version}{versions}}) {
                        $METHODS_HASH->{$method}{$v} = { handler => is_sandbox() && $s->{sandbox_sub} ? $s->{sandbox_sub} : $s->{sub} };
                    }
                }

                if (defined $s->{version}{also_exists_in_versions}) {
                    foreach my $v (@{$s->{version}{also_exists_in_versions}}) {
                        my $handler_sub = is_sandbox() && $s->{sandbox_sub} ? $s->{sandbox_sub} : $s->{sub};
                        $METHODS_HASH->{$method}{$v} = { handler => $handler_sub, only_available_to_selected_apps => 1 };
                    }
                }
            }

            foreach my $s (@$desc) {
                if (defined $s->{version}{more}) {
                    # параметр more означает, что метод доступен начиная с указанной версии и до максимально доступной
                    my ($start_version) = $s->{version}{more};
                    foreach my $v ($start_version .. $max_version) {
                        $METHODS_HASH->{$method}{$v} = { handler => is_sandbox() && $s->{sandbox_sub} ? $s->{sandbox_sub} : $s->{sub} };
                        if ($v <= $max_version) {
                            $METHODS_HASH->{$method}{$v + 100} = { handler => is_sandbox() && $s->{sandbox_sub} ? $s->{sandbox_sub} : $s->{sub} };
                        }
                    }
                }
            }
        }
    }

    return $METHODS_HASH;
}


=head2 _app_has_access_to_restricted_methods

Возвращает булевское значение: если у приложения такой флаг min_api_version_override,
есть ли у него доступ к методам в этой версии API, которые доступны только специальным приложениям?

Правила зависят от min_api_version_override:
0 -- доступа нет
4 -- доступ есть к методам в API4 и API4-live
104 -- доступ есть к методам в API4-live, доступа нет к методам в API4

=cut

sub _app_has_access_to_restricted_methods {
    my ($version, $min_api_version_override) = @_;

    return 0 unless $min_api_version_override;

    if ($min_api_version_override == 4) {
        return 0 unless $version == 4 || $version == 104 || $version == 5;
    }

    if ($min_api_version_override == 104) {
        return 0 unless $version == 104 || $version == 5;
    }

    return 1;
}


=head2 need_error_for_deprecated_method($application_id)

    В функции определяется, нужно ли выдавать клиенту временную ошибку по поводу отключения устаревших методов:
    по последним двум знакам из $application_id определяем группу, в которую попадает приложение,
    для группы расчитываем интервал, в течении которого выдаем ошибку

=cut


sub need_error_for_deprecated_method {
    my ($application_id) = @_;

    my $rounds_count = _get_property_value($API::Settings::DEPRECATED_ERROR_SHOW_ROUNDS_COUNT_PROPERTY, $API::Settings::DEPRECATED_ERROR_SHOW_ROUNDS_COUNT_CACHE_TTL);
    $rounds_count = $API::Settings::DEPRECATED_ERROR_SHOW_ROUNDS_COUNT_DEFAULT unless is_valid_int($rounds_count, 0, 24);
    if ($rounds_count) {
        my $round_duration = 24 * 60 / $rounds_count;
        my $interval = _get_property_value($API::Settings::DEPRECATED_ERROR_SHOW_INTERVAL_PROPERTY, $API::Settings::DEPRECATED_ERROR_SHOW_INTERVAL_CACHE_TTL);
        if (is_valid_int($interval, 1, $round_duration)) {
            my $application_group_num = hex(substr($application_id, -2)) % $API::Settings::APPLICATION_GROUPS_COUNT;
            my $now = now();
            my $minute = ($now->hour * 60 + $now->minute) % $round_duration;
            my $start_minute = $application_group_num * ($round_duration / $API::Settings::APPLICATION_GROUPS_COUNT);
            my $finish_minute = ($start_minute + $interval) % $round_duration;
            if (($finish_minute > $start_minute && $minute >= $start_minute && $minute < $finish_minute)
                || ($finish_minute <= $start_minute && ($minute < $finish_minute || $minute >= $start_minute))) {
                return 1;
            }
        }
    }

    return 0;
}

=head2 _get_property_value

    Возвращает значение свойства $name, кэширует значение, прочитанное из БД на $ttl секунд

=cut

my %properties_cache;
sub _get_property_value {
    my ($name, $ttl) = @_;

    my $property_cache = $properties_cache{$name} //= { last_fetch_time => 0 };
    if (time() - $property_cache->{last_fetch_time} > $ttl) {
        $property_cache->{value} = Property->new($name)->get();
        $property_cache->{last_fetch_time} = time();
    }

    return $property_cache->{value};
}

1;
