=head1 NAME

API::App::Access

=head1 DESCRIPTION

    Модуль для работы с доступами приложений к АПИ

=cut
package API::App::Access;

use strict;
use warnings;
use utf8;

use DateTime;
use Yandex::DBTools;
use Yandex::Log::Messages;
use Yandex::DateTime;
use Yandex::SendMail;

use Settings;

use SandboxClient qw/sandbox_send_app_access/;

use Mouse;
use Mouse::Util::TypeConstraints;
use API::App::Types;

has application_id => (is => 'rw', isa => 'ApplicationId', required => 1);

has access_type => (is => 'rw', isa => 'Maybe[AccessTypeEnum]');

has manager_login => (is => 'rw', isa => 'Maybe[Str]');

has comment  => (is => 'rw', isa => 'Maybe[Str]');

has check_time => (is => 'rw', isa => 'Maybe[DateTime]', default => undef);

has found => (is => 'rw', isa => 'Bool', default => 0);

has min_api_version_override => (is => 'rw', isa => 'Int');

my $log = Yandex::Log::Messages->new();
$log->msg_prefix('api_app_access');

=head2 add

    Дает приложению доступ к АПИ,
    доступ может быть test normal advanced

=cut

sub add {
    my ($self, $access_type) = (shift, shift);
    my $data = {application_id => $self->application_id,
                access_type => $access_type // $self->access_type,
                manager_login => $self->manager_login,
                comment => $self->comment,
                check_time => $self->check_time // now(),
            };
    $log->out($self->application_id.' add access');

    $data->{check_time} = DateTime::Format::MySQL->format_datetime($data->{check_time});
    if (sandbox_send_app_access($data)) {
        $data->{sync_time__dont_quote} = 'NOW()';
    } else {
        $data->{sync_time} = undef;
    }

    _set_access($data);
}

=head2 delete

    Запрещает приложению доступ к АПИ

=cut

sub delete {
    my $self = shift;
    my $data = {application_id => $self->application_id,
                access_type => undef,
                manager_login => $self->manager_login,
                comment => $self->comment,
                check_time => $self->check_time // now(),
            };
    $log->out($self->application_id.' delete access');

    $data->{check_time} = DateTime::Format::MySQL->format_datetime($data->{check_time});
    if (sandbox_send_app_access($data)) {
        $data->{sync_time__dont_quote} = 'NOW()';
    } else {
        $data->{sync_time} = undef;
        send_alert('Could not send Access to Sandbox', 'Sandbox if not available');
    }

    _set_access($data);
}

=head2 to_hash

    Возвращает хеш с полями обьекта 

=cut

sub to_hash {
    my $self = shift;
    my %hash;
    my $meta = $self->meta;
    my $attributes = [$meta->get_all_attributes];
    my @attributes_names = map { $_->name } @$attributes;
    for my $name (@attributes_names) {
        if ($name eq 'check_time' and $self->check_time) {
            $hash{check_time} = DateTime::Format::MySQL->format_datetime($self->check_time);
        } else {
            $hash{$name} = $self->$name;
        }
    }
    return \%hash;
}

=head2 resend_app_access_to_sandbox

    Переотправляет доступ в песочницу,
    в случе если он до неё по каким-то причинам не дошел
    
=cut

sub resend_app_access_to_sandbox {
    my $self = shift;
    sandbox_send_app_access($self->to_hash());
    return;
}

=head2 save_app_access_in_sandbox

    Сохраняет в Песочницу доступ приложения

=cut

sub save_app_access_in_sandbox {
    my ($data) = @_;
    $data->{sync_time__dont_quote} = 'NOW()';
    if ($data->{access_type}) {
        $log->out($data->{application_id}.' add access to Sandbox');
    } else {
        $log->out($data->{application_id}.' delete access from Sandbox');
    }
    return _set_access($data);
}


sub _set_access {
    my $data = shift;
    do_insert_into_table(PPCDICT, 'api_app_access', $data,
                         on_duplicate_key_update => 1,
    );

    return 1;
}

=head2 set_min_api_version_override

=cut

sub set_min_api_version_override {
    my ($self, $min_api_version_override) = @_;

    my $affected_rows = do_update_table(PPCDICT, 'api_app_access',
        { min_api_version_override => $min_api_version_override },
        where => { application_id => $self->application_id});

    die 'No rows affected, access disappeared?' unless $affected_rows;

    $self->min_api_version_override($min_api_version_override);

    return;
}

__PACKAGE__->meta->make_immutable;

1;
