=head1 NAME

API::App::Request

=head1 DESCRIPTION

=cut
package API::App::Request;

use strict;
use warnings;
use utf8;

use JSON;
use DateTime;
use Scalar::Util qw/looks_like_number/;

use Settings;
use Yandex::I18n;
use Yandex::HashUtils;
use Yandex::DBTools;
use Yandex::DBShards;
use Yandex::OAuth;
use Yandex::DateTime;

use API::App::Request::History qw/app_request_history_save/;
use API::App::Types;

use Mouse;
use Mouse::Util::TypeConstraints;
   
has cr_id => (is => 'rw', isa => 'Int', default => 0);

has uid => (is => 'rw', isa => 'Int', required => 1);

has application_id => (is => 'rw', isa => 'ApplicationId', required => 1);

has name => (is => 'rw', isa => 'Str', required => 1);

has description => (is => 'rw', isa => 'Str', default => '');

has access_type => (is => 'rw', isa => 'AccessTypeEnum', required => 1);

has developer_email => (is => 'rw', isa => 'Str', default => '', required => 1);

has company_site => (is => 'rw', isa => 'Maybe[Str]', default => '');

has company_name => (is => 'rw', isa => 'Maybe[Str]', default => '');

has business_type => (
        is => 'rw',
        isa => enum([qw/client agency aggregator developer other/]),
        required => 1
    );

has other_business_type => (is => 'rw', isa => 'Maybe[Str]', default => '');

has app_goals => ( is => 'rw', isa => 'ArrayRef[Goal]', required => 1);

has other_app_goal => ( is => 'rw', isa => 'Maybe[Str]', default => '');

has app_functions => ( is => 'rw', isa => 'ArrayRef[Function]', required => 1);

has other_app_function => ( is => 'rw', isa => 'Maybe[Str]', default => '');

has app_profit => (is => 'rw', isa => 'Str', default => '', required => 1);

has comment => (is => 'rw', isa => 'Maybe[Str]', default => '');

has create_time => (
        is => 'rw',
        isa => 'DateTime',
        default => sub { now() },
        required => 1
    );

has check_time => (
        is => 'rw',
        isa => 'Maybe[DateTime]',
        default => undef
    );

has end_time => (is => 'rw', isa => 'DateTime', default => \&get_end_time);

has status => (is => 'rw', isa => enum([qw/New Approve Reject/]));

has manager_login => (is => 'rw', isa => 'Maybe[Str]');


my $log = Yandex::Log->new(
    log_file_name => 'CertificationRequest.log',
    date_suf      => "%Y%m%d",
    msg_prefix    => "[$$]",
    auto_rotate => 1
);

=head2 

   Если вдруг передали не описанные поля, их не сохраняем 

=cut

around BUILDARGS => sub {
    my $orig  = shift;
    my $self = shift;
    my $request = {@_};
    my $meta = $self->meta;
    my @attributes_names = map {$_->name} $meta->get_all_attributes;
    map {$_->name} $meta->get_all_attributes;
    my $res = hash_cut($request, \@attributes_names);
    return $self->$orig(%$res);

};

=head2 add(self)

   Добавляет заявку на сертификацию

=cut

sub add {
    my ($self, $id) = @_;
    my $app_info;
    eval {$app_info = Yandex::OAuth::oa_get_app_info($self->application_id);};
    $log->out($@) if $@;
    return if $@;
    
    $self->name($app_info->{name});
    $self->cr_id($id || get_new_id('cr_id'));

    my $data = prepare_for_db($self);
    do_insert_into_table(PPCDICT, 'api_app_certification_request', $data);
    return $self->cr_id;
}


=head2 update(self)

   Обновляет заявку на сертификацию

=cut

sub update($;$) {
    my ($self, $new_id) = @_;
    my $old_request = get_one_line_sql(PPCDICT, ["select request_info
                    from api_app_certification_request",
                    where => {application_id => $self->application_id, uid => $self->uid}]);
    my $request = $self->prepare_for_db();
    if ($request->{request_info} ne $old_request->{request_info}) {
        $self->cr_id($new_id || get_new_id('cr_id'));
        $self->status('New');
        $self->comment(undef);
        $request = $self->prepare_for_db();
        
        app_request_history_save($self->uid, $self->application_id);
        do_update_table(PPCDICT, 'api_app_certification_request',
          $request, where => { application_id => $self->application_id,
                            uid => $self->uid});
    }
    return $self->cr_id;
}

=head2 update_for_super(self)

   Обновляет поля доступные суперпользователям и саппортам

=cut

sub update_for_super($){
    my ($self) = @_;
    my $data = hash_cut($self, qw/end_time status check_time comment manager_login/);
    do_update_table(PPCDICT, 'api_app_certification_request',
          $data, where => { application_id => $self->application_id,
                            uid => $self->uid});
    return $self->cr_id;
}


sub _is_equal($$) {
    my ($val1, $val2) = @_;
    if (looks_like_number($val1) && looks_like_number($val2)) {
        return 1 if $val1 == $val2;
    } else {
        return 1 if $val1 eq $val2;
    }
    return;
}


=head2 delete(self)

    Удаляет заявку на сертификацию

=cut

sub delete {
    my ($self) = @_;
    do_delete_from_table(PPCDICT, 'api_app_certification_request',
        where => { application_id => $self->application_id,
                   uid => $self->uid}
    );
    return 1;
}

=head2 get_end_time($self)

    Временно считаем что дата окончания сертефиката
    равна дате подачи заявки + 3 года

=cut

sub get_end_time {
    my ($self) = @_;
    if ($self->create_time) {
        my $dt = $self->create_time->clone();
        $dt->add( years => 3 );
        return $dt;
    } else {
        return;
    }
}

=head2 to_hash

    Преобразует поля обьекта в хеш

=cut

sub to_hash {
    my $self = shift;
    my $meta = $self->meta;
    my @attributes_names = map {$_->name} $meta->get_all_attributes;
    my $hash = hash_cut($self, \@attributes_names);
    return $hash;
};

=head2 prepare_for_db(request)

    Преобразует заявку к виду который можно записать в базу
    
    Отделяет поля которые станут json'ом и отправятся в request_info
    от полей со служебной информацией
    Преобразует даты

=cut

sub prepare_for_db($) {
    my $self = shift;
    my $request = $self->to_hash();
    
    my %service_fields_names = (
        cr_id => 1,
        application_id => 1,
        uid => 1,
        name => 1,
        access_type => 1,
        create_time => 1,
        end_time => 1,
        status => 1,
        comment => 1,
        other_info => 1,
        check_time => 1,
    );
    my $result = hash_kgrep {$service_fields_names{$_}} $request;

    my $request_info = hash_kgrep {!$service_fields_names{$_}} $request;
    $request_info->{specifications_and_screenshots} =
        $request_info->{specifications_and_screenshots}->to_list()
            if defined $request_info->{specifications_and_screenshots};
    
    my $json = JSON->new();
    $json->canonical(1);
    my $request_info_json = $json->encode($request_info);

    $result->{create_time} = DateTime::Format::MySQL->format_datetime($request->{create_time})
        if defined $result->{create_time};
    $result->{end_time} = DateTime::Format::MySQL->format_datetime($request->{end_time})
        if defined $result->{end_time};
    $result->{check_time} = DateTime::Format::MySQL->format_datetime($request->{check_time})
        if defined $result->{check_time};
    delete $result->{check_time} if !defined $result->{check_time};

    $result->{request_info} = $request_info_json;
    return $result;
}


__PACKAGE__->meta->make_immutable;

1;
