=head1 NAME

API::App::Request::MassMethods

=head1 DESCRIPTION

=cut
package API::App::Request::MassMethods;

use strict;
use warnings;
use utf8;

use Carp;

use Settings;
use Yandex::I18n;
use Yandex::DBTools;
use Yandex::Validate qw/is_valid_int/;
use PrimitivesIds;
use Primitives;

use API::App::Request::Tools qw/prepare_from_db/;

use base qw/Exporter/;
our @EXPORT_OK = qw/
    app_requests_count
    app_requests_get_by_user
    app_requests_get_all
    app_requests_get_by_date
    app_requests_search
/;

my $log = Yandex::Log->new(
    log_file_name => 'CertificationRequest.log',
    date_suf      => "%Y%m%d",
    msg_prefix    => "[$$]",
    auto_rotate => 1
);


my @fields = qw/cr_id application_id name uid access_type request_info other_info
             create_time end_time status check_time comment manager_login/;
my $fields_string = join(',' => @fields);

=head2 app_requests_count($uid, %params)

    Получает количество приложений зарегистрированных пользователем
    есть не обязательный именованный параметр status,
    если он не указан возвращаются все заявки составленные пользователем
    
    например :
    app_requests_count(uid, status => 'Approve');
    
    есть 2 статуса Approve и Reject

=cut

sub app_requests_count($;@){
    my ($uid) = shift;
    my %params = @_;
    my $request = get_one_field_sql(PPCDICT, ["select COUNT(*)
                    from api_app_certification_request",
                    where => { uid => $uid, %params }]);
    return $request || 0;
}


=head2 app_requests_get_by_user($uid)

   Получает список заявок клиента

=cut

sub app_requests_get_by_user($){
    my ($uid) = @_;
    my @result;
    my $requests = get_all_sql(PPCDICT, ["select $fields_string
                    from api_app_certification_request",
                    where => {uid => $uid}, 'order by create_time DESC']);
    for my $request (@$requests){
        $request = prepare_from_db($request);

        push @result, $request;
    }
    return \@result; 
}

=head2 app_requests_get_all()

   Получает список из всех заявок

=cut

sub app_requests_get_all($$){
    my ($limit, $offset) = @_;
    my $limit_string = defined $limit ? "LIMIT $limit" : '';
    my $offset_string = defined $offset ? "OFFSET $offset" : '';
    my $requests = get_all_sql(PPCDICT, ["select  $fields_string
                from api_app_certification_request",
                "order by create_time $limit_string $offset_string"]);
    my @result;
    for my $request (@$requests){
        $request = prepare_from_db($request);
        push @result, $request;
    }
    return \@result; 
}

=head2 app_requests_get_by_date(start_date, end_date, page, objects_on_page)

   Получает список заявок по дате
   с постраничной навигацией

=cut
    
sub app_requests_get_by_date($$$;$){
    my ($start_date, $end_date, $page, $objects_on_page) = @_;
    croak 'page not int' if !is_valid_int($page);
    croak 'objects_on_page not int' if $objects_on_page && !is_valid_int($objects_on_page);
    
    my ($requests, $pagination, $limits);
    if ( $objects_on_page ) {
        my $limit = $objects_on_page;
        my $offset = ($page-1) * $objects_on_page;
        $requests = get_all_sql(PPCDICT, ["select SQL_CALC_FOUND_ROWS $fields_string
                from api_app_certification_request",
                where => {create_time__between => [$start_date, $end_date]},
                "order by create_time DESC limit $limit offset $offset"]);
        my $total = select_found_rows(PPCDICT);
        $pagination = _pagination($page, $total, $objects_on_page);
        
    } else {
        $requests = get_all_sql(PPCDICT, ["select $fields_string
                from api_app_certification_request",
                where => {create_time__between => [$start_date, $end_date]},
                "order by create_time DESC"
        ]);
        $pagination = {};
    }
    my @result = map {prepare_from_db($_)} @$requests;

    return {result =>\@result, pagination => $pagination}; 
}

=head2 app_requests_search(self)

   Принимает строку поиска,
   пытается вытащить из неё дату, application_id и название приложения
   ищет записи у которых соответствующие поля равны переданным

=cut

sub app_requests_search($$;$){
    my ($query_str, $page, $objects_on_page) = @_;
    croak 'page not int' if !is_valid_int($page);
    croak 'objects_on_page not int' if $objects_on_page && !is_valid_int($objects_on_page);

    my ($requests, $pagination);
    my $where = _prepare_search_where($query_str);
    if ( $objects_on_page ) {
        my $limit = $objects_on_page;
        my $offset = ($page-1) * $objects_on_page;
        $requests = get_all_sql(PPCDICT, ["select SQL_CALC_FOUND_ROWS *
                from api_app_certification_request",
                'where', $where, "order by create_time DESC limit $limit offset $offset"]);
        my $total = select_found_rows(PPCDICT);
        $pagination = _pagination($page, $total, $objects_on_page);
    } else {
        $requests = get_all_sql(PPCDICT, ["select *
                from api_app_certification_request",
                'where', $where, "order by create_time DESC"]);
        $pagination = {};
    }
    my @result = map {prepare_from_db($_)} @$requests;
    return {result =>\@result, pagination => $pagination};
}

=head2 _pagination(page, total)

   Получает данные для постраничной навигации

=cut

sub _pagination($$) {
    my ($page, $total, $objects_on_page) = @_;
    
    my $page_count = 7;

    my $max_page = POSIX::ceil($total/$objects_on_page);
    my $page_first = $page-($page_count-1)/2 > 1 ? $page-($page_count-1)/2 : 1;
    my $page_last = $page+($page_count-1)/2 < $max_page ? $page+($page_count-1)/2 : $max_page;
    my $pagination = {  max_page => $max_page,
                        pages => [$page_first..$page_last],
                        page => $page,
                        has_next => $page == $max_page ? 0 : 1,
                        has_previous => $page == 1 ? 0 : 1
                    };
    return $pagination;
}

=head2 _prepare_search_where($query_str)

   Собираем из строки поиска условие where

=cut

sub _prepare_search_where($) {
    my $query_str = shift;
    $query_str =~ s/^\s+|\s+$//g;
    my @fields = split(/\s+/, $query_str);
    my $where = '';
    for my $field (@fields){
        $where .= ' AND ' unless $where eq '';
        if ($field =~/[a-z0-9]{32}/) { 
            $where .= "application_id=".sql_quote($field);
        } elsif ($field =~/\d{4}-\d{2}(-\d{2})?/) {
            $where .= "create_time like ".sql_quote(sql_quote_like_pattern($field).'%');
        } else {
            my $uid = get_uid_by_login2($field);
            my $condition;
            if ($uid) {
                $condition .= "uid=$uid OR ";
            }
            $condition .= "name LIKE ".sql_quote('%'.sql_quote_like_pattern($field).'%');
            $where .= '('.$condition.')';
        }
    }
    return $where;
}



1;
