=head1 NAME

API::App::Request::Tools

=head1 DESCRIPTION

=cut
package API::App::Request::Tools;

use strict;
use warnings;
use utf8;

use JSON;

use Settings;
use PrimitivesIds;
use MailService;
use User;
use Yandex::HashUtils;
use Yandex::DBTools;
use Yandex::DateTime;
use Yandex::SendMail qw/sendmail/;
use Yandex::OAuth;

use API::App::Specification::List;

use base qw/Exporter/;
our @EXPORT_OK = qw/
    get_request_info
    get_request_info_by_cr_id
    send_mail_to_support
    prepare_from_db
    prepare_request_object
    prepare_response_object
    mix_app_info
    mix_ulogin
/;

my $log = Yandex::Log->new(
    log_file_name => 'CertificationRequest.log',
    date_suf      => "%Y%m%d",
    msg_prefix    => "[$$]",
    auto_rotate => 1
);


=head2 prepare_from_db(request)

   После того как получили данные в БД,
   десериализуем настройки заявки,
   добавляем информацию о приложении из паспорта

=cut

sub prepare_from_db($){
    my ($request) = @_;
    # десериализуем json
    my ($request_info, $app_info);
    eval{$request_info = from_json($request->{request_info});};
    hash_merge($request, $request_info) unless $@;
    $log->out($@) if $@;
    delete $request->{request_info};
    $request->{check_time} = undef if $request->{check_time} eq '0000-00-00 00:00:00';
    $request->{end_time} = undef if $request->{end_time} eq '0000-00-00 00:00:00';
    # добавляем информацию о приложении
    mix_app_info([$request]);
    return $request;
}

=head2 get_request_info_by_cr_id($cr_id)

    Вернуть данные заявки по ее id, не учитывает uid, поэтому вызывать только
    из админских контроллеров

=cut

sub get_request_info_by_cr_id {
    my ($cr_id) = @_;
    return _get_request_info_by_where({cr_id => $cr_id});
}

=head2 get_request_info(uid, application_id)

    Получить пользовательские поля

=cut

sub get_request_info {
    my ($uid, $application_id) = @_;
    return _get_request_info_by_where({application_id => $application_id, uid => $uid});
}

sub _get_request_info_by_where {
    my ($where) = @_;
    my $request = get_one_line_sql(PPCDICT, ["select
                        cr_id, application_id, name, uid, access_type, request_info, manager_login,
                        other_info, create_time, end_time, status, check_time, comment
                    from api_app_certification_request",
                    where => $where]);

    if ($request) {
        $request = prepare_from_db($request);
        delete $request->{check_time} unless $request->{check_time};
        delete $request->{end_time} unless $request->{end_time};

        $request->{check_time} = datetime($request->{check_time})
            if defined $request->{check_time};
        $request->{create_time} = datetime($request->{create_time})
            if defined $request->{create_time};
        $request->{end_time} = datetime($request->{end_time})
            if defined $request->{end_time};

        $request->{specifications_and_screenshots} =
            prepare_specifications_and_screenshots($request->{specifications_and_screenshots}, 1, $request->{uid})
                if defined $request->{specifications_and_screenshots};
        return $request;
    }
    return undef;
}

=head2 prepare_specifications_and_screenshots(uid, application_id)

    Преобразует поля с серификатами в объекты

=cut

sub prepare_specifications_and_screenshots($$$) {
    my ($specifications, $is_uploaded, $uid) = @_;
    my @specifications_objects;
    for my $specification (@$specifications) {
        $specification->{is_uploaded} = $is_uploaded;
        my $specification_object = API::App::Specification::Item->new(%$specification, uid => $uid); 
        push @specifications_objects, $specification_object;
    }
    return API::App::Specification::List->new(specifications =>\@specifications_objects);
}


=head2 send_mail_to_support

    После того как пользовательдобавил или отредактировал заяву
    Отправляет в саппорт письмо, о том что нужно её посмотреть

=cut

sub send_mail_to_support($$$){
    my ($vars, $uid, $SCRIPT) = @_;
    my $user_data = get_user_data($uid, [qw/login email/]);
    hash_merge($vars, $user_data);
    my $email_to = $Settings::OTRS_EMAIL;
    my $email_from = $Yandex::SendMail::MAIL_FOR_ALERTS;
    $vars->{SCRIPT} = $SCRIPT;
    my ($subj, $content) = MailService::_compile_mail_from_template('new_certification_request', 'ru', $vars);
    return sendmail($email_to, $email_from, $subj, \$content);
}

=head2 prepare_request_object

    Преобразует заявку в формат с которым умеет работать серверная часть 

=cut

sub prepare_request_object($){
    my $request = shift;
    # собираем specifications_and_screenshots из структуры которая пришла из формы
    my $pattern = 'specifications_and_screenshots_(\w+?)s\[(\d+)\]';
    my ($result, @specifications_and_screenshots);
    for my $field (keys %$request){
        if ($field =~ /$pattern/){
            my $key;
            if ($1 eq 'name') {
                $key = 'content';
            } else {
                $key = $1;
            }
            $specifications_and_screenshots[$2]{$key} = $request->{$field};
        } elsif ($field =~ /^(?:protocol|app_goals|app_functions)$/) {
            my @array = split(/,\s*/, $request->{$field});
            $result->{$field} = \@array;
        } else {
            $result->{$field} = $request->{$field};
        }
    }
    $result->{specifications_and_screenshots} =
        prepare_specifications_and_screenshots(\@specifications_and_screenshots, 0, $request->{uid});
    my $request_in_db = get_request_info($request->{uid}, $request->{application_id});
    $result->{specifications_and_screenshots}->merge($request_in_db->{specifications_and_screenshots})
            if exists $request_in_db->{specifications_and_screenshots};
    
    mix_app_info([$result]);
    return $result;
}

=head2 prepare_response_object

    Преобразует заявку в формат с необходимый для клиентской части 

=cut

sub prepare_response_object($){
    my $request = shift;

    $request->{create_time} = DateTime::Format::MySQL->format_datetime($request->{create_time})
        if defined $request->{create_time};
    $request->{end_time} = DateTime::Format::MySQL->format_datetime($request->{end_time})
        if defined $request->{end_time};
    $request->{check_time} = DateTime::Format::MySQL->format_datetime($request->{check_time})
        if defined $request->{check_time};

    $request->{specifications_and_screenshots} = $request->{specifications_and_screenshots}->to_list()
        if $request->{specifications_and_screenshots};

    return $request;
}

=head2 mix_app_info

    Добавляет к данным по заявкам, информацию о приложении из паспорта

=cut

sub mix_app_info($) {
    my $requests = shift;
    for my $request (@$requests){
        my $app_info;
        eval{$app_info = Yandex::OAuth::oa_get_app_info($request->{application_id});};
        $request->{description} = $app_info->{description} unless $@;
        $request->{name} = $app_info->{name} unless $@;
        warn $@ if $@;
    }
    return;
}

=head2 mix_ulogin

    Добавляет к данным по заявкам, ulogin

=cut

sub mix_ulogin($) {
    my $requests = shift;
    my @uids = map {$_->{uid}} @$requests;
    my $uid2login = get_uid2login(uid => \@uids);
    $_->{ulogin} = $uid2login->{ $_->{uid} } for (@$requests);
}


1;
