package API::App::Specification::Item;

=head1 NAME

    API::App::Specification::Item

=head1 DESCRIPTION

=cut
use strict;
use warnings;
use utf8;

use Carp;
use MIME::Base64;

use Settings;
use PrimitivesIds qw/get_clientid/;
use Direct::Storage;

use Mouse;
use Mouse::Util::TypeConstraints;

has storage => (
    is => 'rw',
    isa => 'Object',
    default => sub { Direct::Storage->new() },
);

has collection => ( is => 'ro', isa => 'Str', default => 'certification_request');

has is_uploaded => (is => 'rw', isa => 'Bool', default => 0);

has description => ( is => 'rw', isa => 'Str', default => '', required => 1 );

has name => (is => 'rw', isa => 'Str', default => '', required => 1 );

has content => (is => 'rw', isa => 'Any', default => \&download, lazy => 1);

has uid => ( is => 'rw', isa => 'Int', default => 0);

around BUILDARGS => sub {
    my $orig  = shift;
    my $self = shift;
    my %params = @_;
    my $content = $params{content};
    if ( ref $content eq 'Fh' ) {
        my $name = $content->asString();
        binmode ($content);
        $name =~ /(^.*?)\.(.*)$/;
        my $ext = $2;
        $params{name} = $name;
        croak "wrong file extension" unless $ext;

        local($/) = undef;
        $content = MIME::Base64::encode_base64(<$content>);
        $params{content} = $content;
    }
    return $self->$orig(%params);
};

=head2 download

    Получает файлы из Стораджа

=cut

sub download {
    my ($self) = @_;
    my $content;
    my $file = $self->storage->get_file($self->collection, ClientID => get_clientid(uid => $self->uid), filename => $self->name);
    if ($file) {
        $content = decode_base64($file->content);
        return $content;
    } else {
        die $self->name.": file not found";
    }
}

=head2 upload

    Загружает файлы с сертификатами и скриншотами в Сторадж

=cut

sub upload {
    my ($self) = @_;
    
    $self->storage->save($self->collection, $self->content, ClientID => get_clientid(uid => $self->uid), filename => $self->name);
    $self->is_uploaded(1);
    return 1;
}

=head2 delete

    Удаляет файл из Стораджа

=cut

sub delete {
    my ($self) = @_;

    $self->storage->delete_file($self->collection, ClientID => get_clientid(uid => $self->uid), filename => $self->name);
    return 1;
}

=head2 to_hash

    Преобразует поля объекта в хеш

=cut

sub to_hash {
    my ($self) = @_;
    my $hash = {
        description => $self->description,
        name => $self->name
    };
    return $hash;
}



1;
