package API::Authorize;

use Direct::Modern;

# $Id$

=pod

    Модуль для методов авторизации

=cut

use Data::Dumper;

use Yandex::Blackbox;
use Yandex::I18n;
use Yandex::DateTime;

use Settings;

use TextTools qw/normalize_login/;
use API::App::CheckAccess;
use API::Authorize::PersistentToken qw/check_token/;

require Exporter;
use base qw/Exporter/;

our @EXPORT_OK = qw/authorize_user/;

my $APPS_GROUP_COUNT = 4;
my $SHOW_ERROR_PERIOD =  10;

=head2 authorize_user(%OPT)

    Аутентификация пользователя (проверка сертификата или токена)
    НЕ ПРОВЕРЯЕТ: наличие флага clients_api_options.api_enabled и роль клиента

    Именованные аргументы:
    remote_addr - ip запроса
    token - из заголовков запроса
    persistent_token - из заголовков запроса
    login - из заголовков запроса

=cut

sub authorize_user
{
    my %auth = @_;
    if ($auth{persistent_token}) {
        return authorize_user_by_persistent_token(%auth);
    }

    my ($valid, $uid, $result);

     eval {
          if ($auth{token}) {
                # проверяем токен в oauth
               my $auth_res = Yandex::Blackbox::bb_oauth_token($auth{token}, $auth{remote_addr}, $Settings::API_SERVER_PATH, [$BB_LOGIN], undef, $auth{reqid});

               if ($auth_res->{hosted}) { # PDD логины не поддерживаются в Директе
                     $result = {
                         error => 'UserInvalid',
                         error_detail => iget('Логины вида @yourdomain.ru не поддерживаются сервисом Яндекс.Директ')
                    };
                    return;
               } elsif ($auth_res->{status} eq 'VALID') {
                    my $real_login = $auth_res->{'dbfield'}{$BB_LOGIN};

                    # проверяем список допустимых прав
                    unless ($auth_res->{scope}
                        && scalar grep { $auth_res->{scope}{$_} } ('direct:api', 'direct:api-1month', 'direct:api-3month')
                    ) {
                         return 0;
                    }
                    my $access = API::App::CheckAccess::check_registration($auth_res->{client_id});
                    unless ( $access->has_access ) {
                         if (is_new_application($auth_res->{OAuth}{client_ctime}{content})
                             || is_need_show_error($auth_res->{client_id})
                         ) {
                              $result = {
                                   error => 'AppNotRegistered',
                                   error_detail => iget('Необходимо заполнить для приложения заявку на доступ в интерфейсе Директа и дождаться её подтверждения')
                              }
                         }
                    }
                    unless ($result && exists $result->{error}) {
                         if (is_valid_login($auth{login}, $real_login)){
                              $valid = 1;
                              $result = {
                                   uid => $auth_res->{uid},
                                   login => $real_login,
                                   application_id => $auth_res->{client_id},
                                   access => $access,
                              };
                         } else {
                              $valid = 0;
                         }
                    }
               }
          } else {
               if ($ENV{'SERVER_ADDR'} ne '127.0.0.1') {
                    die "Доступ к серверу с ip=".($ENV{'SERVER_ADDR'} || '')." запрещен";
               }
               
               # тут была проверка SSL-сертификатов, но их уж нет
               $result = {
                    uid => $uid,
                    login => '',
                    application_id => ''
               };
          }
     };
     if ($@) { # паспорт лежит
          warn Dumper {authorize_user => $@};
          return {error => 'AuthTmpUnavail'};
     }

     return $result if exists $result->{error}; # выбрали ошибку

     return undef if !$valid; # не выбрали ошибку и авторизация не прошла
     return $result; # авторизация прошла успешно
}


=head2 authorize_user_by_persistent_token(%auth)

    авторизация пользователя по захардкоженному в API::Settings токену
    аргументы и результат такие же, как у authorize_user

=cut
sub authorize_user_by_persistent_token {
    my (%auth) = @_;

    if ($auth{token}) {
        return {
            error => 'UserInvalid',
            error_detail => iget('Неверное использование persistent_token'),
        };
    }

    my $result = check_token($auth{persistent_token}, $auth{remote_addr});

    return undef unless $result;

    return undef unless is_valid_login($auth{login}, $result->{login});

    return $result;
}

=head2 is_need_show_error($app_reg_time)

    Для приложений которые зарегистрированы раньше, чем мы ввели обязательную регистрацию
    будем иногда показывать сообщение об ошибке, так как совсем отключать их нельзя(приносят деньги),
    но дать стимул чтоб все же зарегистрировались хочется 

=cut

sub is_need_show_error($) {
    my $application_id = shift;
    if (-f $Settings::UNREGISTERED_APPS_STOP_FLAG) {
        my $app_group = hex( substr $application_id, -2, 2 ) % $APPS_GROUP_COUNT;
        my $now = now();
        if ( ($now->hour % $APPS_GROUP_COUNT == $app_group) and ($now->minute < $SHOW_ERROR_PERIOD) ) {
            return 1;
        }
    }
    return 0;
}


=head2 is_new_application($app_reg_time)

    Зарегистрировано ли приложение после даты, когда начали требовать регистрацию

=cut

sub is_new_application($){
    my $app_reg_time = shift;
    my $last_date_without_cert = datetime($API::Settings::NEED_CHECK_REGISTRATION_DATE);
    return 0 if !$app_reg_time
                || (datetime($app_reg_time) lt $last_date_without_cert);
    return 1;
}

=head2 is_valid_login($app_reg_time)
     
     Проверяет соответствие логина переданного в запросе, логину которому принадлежит токен

=cut

sub is_valid_login($$){
    my ($login, $real_login) = (shift, shift);
    $real_login =  normalize_login($real_login);
    $login = normalize_login($login);
    # если логин не указан ничего страшного просто берем его из OAuth
    return 1 if !defined $login;
    # если логин указан он должен быть валидным
    return 1 if lc($login) eq lc($real_login);
    return 0;
}


1;
